% Generated by roxygen2 (4.0.2): do not edit by hand
\name{nblast}
\alias{nblast}
\title{Calculate similarity score for neuron morphologies}
\usage{
nblast(query, target = getOption("nat.default.neuronlist"), smat = NULL,
  sd = 3, version = c(2, 1), normalised = FALSE, UseAlpha = FALSE,
  OmitFailures = NA, ...)
}
\arguments{
\item{query}{the query neuron.}

\item{target}{a \code{\link[nat]{neuronlist}} to compare neuron against.
Defaults to \code{options("nat.default.neuronlist")}. See
\code{\link{nat-package}}.}

\item{smat}{the scoring matrix to use (see details)}

\item{sd}{Standard deviation to use in distance dependence of nblast v1
algorithm. Ignored when \code{version=2}.}

\item{version}{the version of the algorithm to use (the default, 2, is the
latest).}

\item{normalised}{whether to divide scores by the self-match score of the
query}

\item{UseAlpha}{whether to consider local directions in the similarity
calculation (default: FALSE).}

\item{OmitFailures}{Whether to omit neurons for which \code{FUN} gives an
error. The default value (\code{NA}) will result in nblast stopping with an
error message the moment there is an eror. For other values, see details.}

\item{...}{Additional arguments passed to NeuriteBlast or the function used
to compute scores from distances/dot products. (expert use only).}
}
\value{
Named list of similarity scores.
}
\description{
Uses the NBLAST algorithm that compares the morphology of two neurons. For
more control over the parameters of the algorithm, see the arguments of
\code{\link{NeuriteBlast}}.
}
\details{
when \code{smat=NULL} options("nat.nblast.defaultsmat") will be
  checked and if NULL, then \code{smat.fcwb} or \code{smat_alpha.fcwb} will
  be used depending on the value of \code{UseAlpha}.

  When \code{OmitFailures} is not \code{NA}, individual nblast calls will be
  wrapped in \code{try} to ensure that failure for any single neuron does not
  abort the whole nblast call. When \code{OmitFailures=FALSE}, missing values
  will be left as \code{NA}. \code{OmitFailures=TRUE} is not (yet)
  implemented. If you want to drop scores for neurons that failed you will
  need to set \code{OmitFailures=FALSE} and then use \code{\link{na.omit}} or
  similar to post-process the scores.

  Note thatn when \code{OmitFailures=FALSE} error messages will not be
  printed because the call is wrapped as \code{try(expr, silent=TRUE)}.

  Internally, the \code{\link{plyr}} package is used to provide options for
  parallelising NBLASTs and displaying progress. To display a progress bar as
  the scores are computed, add \code{.progress="text"} to the arguments
  (non-text progress bars are available -- see
  \code{\link[plyr]{create_progress_bar}}). To parallelise, add
  \code{.parallel=TRUE} to the arguments. In order to make use of parallel
  calculation, you must register a parallel backend that will distribute the
  computations. There are several possible backends, the simplest of which is
  the multicore option made available by \code{doMC}, which spreads the load
  across cores of the same machine. Before using this, the backend must be
  registered using \code{registerDoMC} (see example below).
}
\examples{
# load sample Kenyon cell data from nat package
data(kcs20, package='nat')
# search one neuron against all neurons
scores=nblast(kcs20[['GadMARCM-F000142_seg002']], kcs20)
# scores from best to worst, top hit is of course same neuron
sort(scores, decreasing = TRUE)
hist(scores, breaks=25, col='grey')
abline(v=1500, col='red')

# plot query neuron
open3d()
# plot top 3 hits (including self match with thicker lines)
plot3d(kcs20[which(sort(scores, decreasing = TRUE)>1500)], lwd=c(3,1,1))
rest=names(which(scores<1500))
plot3d(rest, db=kcs20, col='grey', lwd=0.5)

# normalised scores (i.e. self match = 1) of all neurons vs each other
# note use of progress bar
scores.norm=nblast(kcs20, kcs20, normalised = TRUE, .progress="text")
hist(scores.norm, breaks=25, col='grey')
# produce a heatmap from normalised scores
jet.colors <- colorRampPalette( c("blue", "green", "yellow", "red") )
heatmap(scores.norm, labCol = with(kcs20,type), col=jet.colors(20), symm = TRUE)

\dontrun{
# Parallelise NBLASTing across 4 cores using doMC package
library(doMC)
registerDoMC(4)
scores.norm2=nblast(kcs20, kcs20, normalised=TRUE, .parallel=TRUE)
stopifnot(all.equal(scores.norm2, scores.norm))
}
}
\seealso{
\code{\link{nat-package}}
}

