\name{netposet}
\alias{netposet}
\alias{print.netposet}

\title{Partial order of treatments in network meta-analysis}

\description{
  Partial order of treatments in network meta-analysis. The set of
  treatments in a network is called a partially ordered set (in short,
  a \emph{poset}), if different outcomes provide different treatment
  ranking lists.
}

\usage{
netposet(..., outcomes, treatments, small.values,
         comb.fixed, comb.random)

\method{print}{netposet}(x,
      pooled=ifelse(x$comb.random, "random", "fixed"),
      ...)
}

\arguments{
  \item{\dots}{See details.}
  \item{outcomes}{A character vector with outcome names.}
  \item{treatments}{A character vector with treatment names.}
  \item{small.values}{See details.}
  \item{comb.fixed}{A logical indicating whether to show results for the
    fixed effects (common effects) model.}
  \item{comb.random}{A logical indicating whether to show results for
    the random effects model.}
  \item{x}{An object of class \code{netposet}.}
  \item{pooled}{A character string indicating whether Hasse diagram
    should be drawn for fixed effect (\code{"fixed"}) or random
    effects model (\code{"random"}). Can be abbreviated.}
}

\details{
  In network meta-analysis, frequently different outcomes are
  considered which may each provide a different ordering of
  treatments. The concept of a partially ordered set (in short, a
  \emph{poset}, Carlsen & Bruggemann, 2014) of treatments can be used
  to gain further insights in situations with apparently conflicting
  orderings. This implementation for rankings in network meta-analyis
  is described in Rücker & Schwarzer (2017).

  In function \code{netposet}, argument \code{\dots} can be any of the
  following:
  \itemize{
  \item arbitrary number of \code{netrank} objects providing P-scores;
  \item arbitrary number of \code{netmeta} objects;
  \item single ranking matrix with each column providing P-scores
    (Rücker & Schwarzer 2015) or SUCRA values (Salanti et al. 2011)
    for an outcome and rows corresponding to treatments.
  }
  Note, albeit in general a ranking matrix is not constrained to have
  values between 0 and 1, \code{netposet} stops with an error in this
  case as this function expects a matrix with P-scores or SUCRA
  values.

  Argument \code{outcomes} can be used to label outcomes. If argument
  \code{outcomes} is missing,
  \itemize{
  \item column names of the ranking matrix are used as outcome labels
    (if first argument is a ranking matrix and column names are
    available);
  \item capital letters 'A', 'B', \dots are used as outcome labels and
    a corresponding warning is printed.
  }
  Argument \code{treatments} can be used to provide treatment labels
  if the first argument is a ranking matrix. If argument
  \code{treatment} is missing,
  \itemize{
  \item row names of the ranking matrix are used as treatment labels
    (if available);
  \item letters 'a', 'b', \dots are used as treatment labels and a
    corresponding warning is printed.
  }
  If argument \code{\dots} consists of \code{netmeta} objects,
  \code{netrank} is called internally to calculate P-scores. In this
  case, argument \code{small.values} can be used to specify for each
  outcome whether small values are good or bad; see
  \code{\link{netrank}}. This argument is ignored for a ranking matrix
  and \code{netrank} objects.
  
  Arguments \code{comb.fixed} and \code{comb.random} can be used to
  define whether results should be printed and plotted for fixed
  effect and / or random effects model. If netmeta and netrank objects
  are provided in argument \code{\dots}, values for \code{comb.fixed}
  and \code{comb.random} within these objects are considered; if these
  values are not unique, argument \code{comb.fixed} and / or
  \code{comb.random} are set to \code{TRUE}.
  
  In function \code{print.netposet}, argument \code{\dots} is passed on
  to the printing function.
}

\value{
  An object of class \code{netposet} with corresponding \code{print},
  \code{plot}, and \code{hasse} functions. The object is a list
  containing the following components:
  \item{P.fixed}{Ranking matrix with rows corresponding to treatments
    and columns corresponding to outcomes (fixed effect model).}
  \item{M0.fixed}{Hasse matrix skipping unnecessary paths (fixed
    effect model).}
  \item{M.fixed}{"Full" Hasse matrix (fixed effect model).}
  \item{O.fixed}{Matrix with information about partial ordering (fixed
    effect model).}
  \item{P.random}{Ranking matrix with rows corresponding to treatments
    and columns corresponding to outcomes (random effects model).}
  \item{M0.random}{Hasse matrix skipping unnecessary paths (random
    effects model).}
  \item{M.random}{"Full" Hasse matrix (random effects model).}
  \item{O.random}{Matrix with information about partial ordering
    (random effects model).}
  \item{small.values, comb.fixed, comb.random}{As.defined above.}
  \item{call}{Function call.}
  \item{version}{Version of R package netmeta used to create object.}
}

\references{
  Carlsen L, Bruggemann R (2014):
  Partial order methodology: a valuable tool in chemometrics.
  \emph{Journal of Chemometrics},
  \bold{28}, 226--34

  Rücker G, Schwarzer G (2015):
  Ranking treatments in frequentist network meta-analysis works
  without resampling methods.
  \emph{BMC Medical Research Methodology},
  \bold{15}, 58

  Rücker G, Schwarzer G (2017):
  Resolve conflicting rankings of outcomes in network meta-analysis:
  Partial ordering of treatments.
  \emph{Research Synthesis Methods},
  \bold{8}, 526--36

  Salanti G, Ades AE, Ioannidis JP (2011):
  Graphical methods and numerical summaries for presenting results
  from multiple-treatment meta-analysis: an overview and tutorial.
  \emph{Journal of Clinical Epidemiology},
  \bold{64}: 163--71
}

\author{Gerta Rücker \email{ruecker@imbi.uni-freiburg.de}, Guido Schwarzer \email{sc@imbi.uni-freiburg.de}}

\seealso{\code{\link{netmeta}}, \code{\link{netrank}}, \code{\link{hasse}}, \code{\link{plot.netposet}}}

\examples{
# Use depression dataset
#
data(Linde2015)
#
# Define order of treatments
#
trts <- c("TCA", "SSRI", "SNRI", "NRI",
          "Low-dose SARI", "NaSSa", "rMAO-A", "Hypericum",
          "Placebo")
#
# Outcome labels
#
outcomes <- c("Early response", "Early remission")
#
# (1) Early response
#
p1 <- pairwise(treat = list(treatment1, treatment2, treatment3),
               event = list(resp1, resp2, resp3),
               n = list(n1, n2, n3),
               studlab = id, data = Linde2015, sm = "OR")
#
net1 <- netmeta(p1, comb.fixed = FALSE,
                seq = trts, ref = "Placebo")
#
# (2) Early remission
#
p2 <- pairwise(treat = list(treatment1, treatment2, treatment3),
               event = list(remi1, remi2, remi3),
               n = list(n1, n2, n3),
               studlab = id, data = Linde2015, sm = "OR")
#
net2 <- netmeta(p2, comb.fixed = FALSE,
                seq = trts, ref = "Placebo")
#
# Partial order of treatment rankings (two outcomes)
#
po <- netposet(netrank(net1, small.values = "bad"),
               netrank(net2, small.values = "bad"),
               outcomes = outcomes)
#
# Hasse diagram
#
hasse(po)


\dontrun{
#
# Outcome labels
#
outcomes <- c("Early response", "Early remission",
              "Lost to follow-up", "Lost to follow-up due to AEs",
              "Adverse events (AEs)")
#
# (3) Loss to follow-up
#
p3 <- pairwise(treat = list(treatment1, treatment2, treatment3),
               event = list(loss1, loss2, loss3),
               n = list(n1, n2, n3),
               studlab = id, data = Linde2015, sm = "OR")
#
net3 <- netmeta(p3, comb.fixed = FALSE,
                seq = trts, ref = "Placebo")
#
# (4) Loss to follow-up due to adverse events
#
p4 <- pairwise(treat = list(treatment1, treatment2, treatment3),
               event = list(loss.ae1, loss.ae2, loss.ae3),
               n = list(n1, n2, n3),
               studlab = id, data = subset(Linde2015, id != 55),
               sm = "OR")
#
net4 <- netmeta(p4, comb.fixed = FALSE,
                seq = trts, ref = "Placebo")
#
# (5) Adverse events
#
p5 <- pairwise(treat = list(treatment1, treatment2, treatment3),
               event = list(ae1, ae2, ae3),
               n = list(n1, n2, n3),
               studlab = id, data = Linde2015, sm = "OR")
#
net5 <- netmeta(p5, comb.fixed = FALSE,
                seq = trts, ref = "Placebo")
#
# Partial order of treatment rankings (all five outcomes)
#
po.ranks <- netposet(netrank(net1, small.values = "bad"),
                     netrank(net2, small.values = "bad"),
                     netrank(net3, small.values = "good"),
                     netrank(net4, small.values = "good"),
                     netrank(net5, small.values = "good"),
                     outcomes = outcomes)
#
# Same result
#
po.nets <- netposet(net1, net2, net3, net4, net5,
                    small.values = c("bad", "bad", "good", "good", "good"),
                    outcomes = outcomes)
#
all.equal(po.ranks, po.nets)
#
# Print matrix with P-scores (random effects model)
#
po.nets$P.random
#
# Hasse diagram for all outcomes (random effects model)
#
hasse(po.ranks)
#
# Hasse diagram for outcomes early response and early remission
#
po12 <- netposet(netrank(net1, small.values = "bad"),
                 netrank(net2, small.values = "bad"),
                 outcomes = outcomes[1:2])
hasse(po12)
#
# Scatter plot
#
oldpar <- par(pty = "s")
plot(po12)
par(oldpar)
}


# Example using ranking matrix with P-scores
#
# Ribassin-Majed L, Marguet S, Lee A.W., et al. (2017):
# What is the best treatment of locally advanced nasopharyngeal
# carcinoma? An individual patient data network meta-analysis.
# Journal of Clinical Oncology, 35, 498-505
#
outcomes <- c("OS", "PFS", "LC", "DC")
treatments <- c("RT", "IC-RT", "IC-CRT", "CRT",
                "CRT-AC", "RT-AC", "IC-RT-AC")
#
# P-scores (from Table 1)
#
pscore.os  <- c(15, 33, 63, 70, 96, 28, 45) / 100
pscore.pfs <- c( 4, 46, 79, 52, 94, 36, 39) / 100
pscore.lc  <- c( 9, 27, 47, 37, 82, 58, 90) / 100
pscore.dc  <- c(16, 76, 95, 48, 72, 32, 10) / 100
#
pscore.matrix <- data.frame(pscore.os, pscore.pfs, pscore.lc, pscore.dc)
rownames(pscore.matrix) <- treatments
colnames(pscore.matrix) <- outcomes
pscore.matrix
#
po <- netposet(pscore.matrix)
po12 <- netposet(pscore.matrix[, 1:2])
po
po12
#
hasse(po)
hasse(po12)
#
oldpar <- par(pty = "s")
plot(po12)
par(oldpar)
}

\keyword{Partial order of rankings (poset)}
