#' Open Parking and Camera Violations
#'
#' Downloads Open Parking and Camera Violations data from NYC Open Data.
#'
#' @param limit Number of rows to retrieve (default = 10,000).
#' @param filters Optional list of field-value pairs to filter results.
#' @return A tibble containing Open Parking and Camera Violations data.
#'
#' @details
#' The Open Parking and Camera Violations dataset was initially loaded with all violations contained in the mainframe database as of May 2016.
#'
#'
#' @source NYC Open Data: <https://data.cityofnewyork.us/City-Government/Open-Parking-and-Camera-Violations/nc67-uf89/about_data>
#'
#' @examples
#' # Quick example (fetch 10 rows)
#' small_sample <- nyc_open_parking_camera_violations(limit = 10)
#' head(small_sample)
#'
#' \donttest{
#' nyc_open_parking_camera_violations(limit = 5000)
#' nyc_open_parking_camera_violations(filters = list(state = "NJ"))
#' }
#' @export
nyc_open_parking_camera_violations <- function(limit = 10000, filters = list()) {
  endpoint <- "https://data.cityofnewyork.us/resource/nc67-uf89.json"

  query_list <- list(
    "$limit" = limit,
    "$order" = "issue_date DESC"
  )

  if (length(filters) > 0) {
    where_clauses <- paste0(names(filters), " = '", filters, "'")
    query_list[["$where"]] <- paste(where_clauses, collapse = " AND ")
  }

  resp <- httr::GET(endpoint, query = query_list)
  httr::stop_for_status(resp)
  data <- jsonlite::fromJSON(httr::content(resp, as = "text"), flatten = TRUE)
  tibble::as_tibble(data)
}
