% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optweight.R
\name{optweight}
\alias{optweight}
\alias{optweightMV}
\title{Estimate Stable Balancing Weights}
\usage{
optweight(
  formula,
  data = NULL,
  tols = 0,
  estimand = "ATE",
  targets = NULL,
  s.weights = NULL,
  b.weights = NULL,
  focal = NULL,
  norm = "l2",
  verbose = FALSE,
  ...
)

optweightMV(
  formula.list,
  data = NULL,
  tols.list = list(0),
  estimand = "ATE",
  targets = NULL,
  s.weights = NULL,
  b.weights = NULL,
  focal = NULL,
  norm = "l2",
  verbose = FALSE,
  ...
)
}
\arguments{
\item{formula}{A formula with a treatment variable on the left hand side and the covariates to be balanced on the right hand side, or a list thereof. See \code{\link[=glm]{glm()}} for more details. Interactions and functions of covariates are allowed.}

\item{data}{An optional data set in the form of a data frame that contains the variables in \code{formula}.}

\item{tols}{A vector of balance tolerance values for each covariate, or a list thereof. The resulting weighted balance statistics will be at least as small as these values. If only one value is supplied, it will be applied to all covariates. Can also be the output of a call to \code{\link[=process_tols]{process_tols()}}. See Details.}

\item{estimand}{The desired estimand, which determines the target population. For binary treatments, can be "ATE", "ATT", "ATC", or \code{NULL}. For multi-category treatments, can be "ATE", "ATT", or \code{NULL}. For continuous treatments, can be "ATE" or \code{NULL}. The default for both is "ATE". For \code{optweightMV()}, only "ATE" or \code{NULL} are supported. \code{estimand} is ignored when \code{targets} is non-\code{NULL}. If both \code{estimand} and \code{targets} are \code{NULL}, no targeting will take place. See Details.}

\item{targets}{A vector of target population mean values for each baseline covariate. The resulting weights will yield sample means within \code{tols}/2 units of the target values for each covariate. If \code{NULL} or all \code{NA}, \code{estimand} will be used to determine targets. Otherwise, \code{estimand} is ignored. If any target values are \code{NA}, the corresponding variable will not be targeted and its weighted mean will be wherever the weights yield the smallest variance. Can also be the output of a call to \code{\link[=process_targets]{process_targets()}}. See Details.}

\item{s.weights}{A vector of sampling weights or the name of a variable in \code{data} that contains sampling weights.}

\item{b.weights}{A vector of base weights or the name of a variable in \code{data} that contains base weights. If supplied, the desired norm of the distance between the estimated weights and the base weights is minimized.}

\item{focal}{When multi-category treatments are used and \code{estimand = "ATT"}, which group to consider the "treated" or focal group. This group will not be weighted, and the other groups will be weighted to be more like the focal group. If specified, \code{estimand} will automatically be set to \code{"ATT"}.}

\item{norm}{\code{character}; a string containing the name of the norm corresponding to the objective function to minimize. Allowable options include \code{"l1"} for the L1 norm, \code{"l2"} for the L2 norm (the default), \code{"linf"} for the L\eqn{\infty} norm, \code{"entropy"} for the negative entropy, and \code{"log"} for the sum of the logs. See \code{\link[=optweight.fit]{optweight.fit()}} for details.}

\item{verbose}{\code{logical}; whether information on the optimization problem solution should be printed. Default is \code{FALSE}.}

\item{...}{
  Arguments passed on to \code{\link[=optweight.fit]{optweight.fit}}, \code{\link[=optweightMV.fit]{optweightMV.fit}}
  \describe{
    \item{\code{std.binary,std.cont}}{\code{logical}; whether the tolerances are in standardized mean units (\code{TRUE}) or raw units (\code{FALSE}) for binary variables and continuous variables, respectively. The default is \code{FALSE} for \code{std.binary} because raw proportion differences make more sense than standardized mean difference for binary variables. These arguments are analogous to the \code{binary} and \code{continuous} arguments in \code{bal.tab()} in \pkg{cobalt}.}
    \item{\code{min.w}}{\code{numeric}; a single value less than 1 for the smallest allowable weight. Some analyses require nonzero weights for all units, so a small, nonzero minimum may be desirable. The default is \code{1e-8} (\eqn{10^{-8}}), which does not materially change the properties of the weights from a minimum of 0 but prevents warnings in some packages that use weights to estimate treatment effects. When \code{norm} is \code{"entropy"} or \code{"log"} and \code{min.w <= 0}, \code{min.w} will be set to the smallest nonzero value.}
    \item{\code{covs.list}}{a list containing one numeric matrix of covariates to be balanced for each treatment.}
    \item{\code{treat.list}}{a list containing one vector of treatment statuses for each treatment.}
    \item{\code{solver}}{string; the name of the optimization solver to use. Allowable options depend on \code{norm}. Default is to use whichever eligible solver is installed, if any, or the default solver for the corresponding \code{norm}. See Details for information.}
  }}

\item{formula.list}{A list of formulas, each with a treatment variable on the left hand side and the covariates to be balanced on the right hand side.}

\item{tols.list}{A list of vectors of balance tolerance values for each covariate for each treatment. The resulting weighted balance statistics will be at least as small as these values. If only one value is supplied, it will be applied to all covariates. See Details.}
}
\value{
For \code{optweight()}, an \code{optweight} object with the following elements:
\item{weights}{The estimated weights, one for each unit.}
\item{treat}{The values of the treatment variable.}
\item{covs}{The covariates used in the fitting. Only includes the raw covariates, which may have been altered in the fitting process.}
\item{s.weights}{The provided sampling weights.}
\item{b.weights}{The provided base weights.}
\item{estimand}{The estimand requested.}
\item{focal}{The focal variable if the ATT was requested with a multi-category treatment.}
\item{call}{The function call.}
\item{tols}{The tolerance values for each covariate.}
\item{duals}{A data.frame containing the dual variables for each covariate. See Details for interpretation of these values.}
\item{info}{Information about the performance of the optimization at termination.}

For \code{optweightMV()}, an \code{optweightMV} object with the following elements:
\item{weights}{The estimated weights, one for each unit.}
\item{treat.list}{A list of the values of the treatment variables.}
\item{covs.list}{A list of the covariates for each treatment used in the fitting. Only includes the raw covariates, which may have been altered in the fitting process.}
\item{s.weights}{The provided sampling weights.}
\item{b.weights}{The provided base weights.}
\item{call}{The function call.}
\item{tols}{A list of tolerance values for each covariate for each treatment.}
\item{duals}{A list of data.frames containing the dual variables for each covariate for each treatment. See Details for interpretation of these values.}
\item{info}{Information about the performance of the optimization at termination.}
}
\description{
Estimate stable balancing weights for treatments and covariates specified in \code{formula}. The degree of balance for each covariate is specified by \code{tols} and the target population can be specified with \code{targets} or \code{estimand}. See Zubizarreta (2015) and Wang & Zubizarreta (2019) for details of the properties of the weights and the methods used to fit them.
}
\details{
The optimization is performed by the lower-level function \code{\link[=optweight.fit]{optweight.fit()}} (for \code{optweight()}) or \code{\link[=optweightMV.fit]{optweightMV.fit()}} (for \code{optweightMV()}).

For binary and multi-category treatments, weights are estimated so that the weighted mean differences of the covariates are within the given tolerance thresholds (unless \code{std.binary} or \code{std.cont} are \code{TRUE}, in which case standardized mean differences are considered for binary and continuous variables, respectively). For a covariate \eqn{x} with specified tolerance \eqn{\delta}, the weighted means of each each group will be within \eqn{\delta} of each other. Additionally, when the ATE is specified as the estimand or a target population is specified, the weighted means of each group will each be within \eqn{\delta/2} of the target means; this ensures generalizability to the same population from which the original sample was drawn.

If standardized tolerance values are requested, the standardization factor corresponds to the estimand requested: when the ATE is requested or a target population specified, the standardization factor is the square root of the average variance for that covariate across treatment groups, and when the ATT or ATC are requested, the standardization factor is the standard deviation of the covariate in the focal group. The standardization factor is computed accounting for \code{s.weights}.

For continuous treatments, weights are estimated so that the weighted correlation between the treatment and each covariate is within the specified tolerance threshold. If the ATE is requested or a target population is specified, the means of the weighted covariates and treatment are restricted to be equal to those of the target population to ensure generalizability to the desired target population. The weighted correlation is computed as the weighted covariance divided by the product of the \emph{unweighted} standard deviations. The means used to center the variables in computing the covariance are those specified in the target population.
\subsection{Dual Variables}{

Two types of constraints may be associated with each covariate: target constraints and balance constraints. Target constraints require the mean of the covariate to be at (or near) a specific target value in each treatment group (or for the whole group when treatment is continuous). Balance constraints require the means of the covariate in pairs of treatments to be near each other. For binary and multi-category treatments, balance constraints are redundant if target constraints are provided for a variable. For continuous variables, balance constraints refer to the correlation between treatment and the covariate and are not redundant with target constraints. In the \code{duals} component of the output, each covariate has a dual variable for each nonredundant constraint placed on it.

The dual variable for each constraint is the instantaneous rate of change of the objective function at the optimum corresponding to a change in the constraint. Because this relationship is not linear, large changes in the constraint will not exactly map onto corresponding changes in the objective function at the optimum, but will be close for small changes in the constraint. For example, for a covariate with a balance constraint of .01 and a corresponding dual variable of 40, increasing (i.e., relaxing) the constraint to .025 will decrease the value of the objective function at the optimum by approximately \eqn{(.025 - .01) * 40 = .6}.

For factor variables, \code{optweight()} takes the sum of the absolute dual variables for the constraints for all levels and reports it as the the single dual variable for the variable itself. This summed dual variable works the same way as dual variables for continuous variables do.
}
}
\examples{
\dontshow{if (requireNamespace("cobalt", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library("cobalt")
data("lalonde", package = "cobalt")

# Balancing covariates between treatment groups (binary)
(ow1 <- optweight(treat ~ age + educ + married +
                    nodegree + re74, data = lalonde,
                  tols = c(.01, .02, .03, .04, .05),
                  estimand = "ATE"))
bal.tab(ow1)

# Exactly alancing covariates with respect to race (multi-category)
(ow2 <- optweight(race ~ age + educ + married +
                    nodegree + re74, data = lalonde,
                  tols = 0, estimand = "ATT",
                  focal = "black"))
bal.tab(ow2)

# Balancing covariates between treatment groups (binary)
# and requesting a specified target population
targets <- process_targets(~ age + educ + married +
                             nodegree + re74,
                           data = lalonde,
                           targets = c(26, 12, .4, .5,
                                       1000))

(ow3a <- optweight(treat ~ age + educ + married +
                     nodegree + re74, data = lalonde,
                   targets = targets,
                   estimand = NULL))

bal.tab(ow3a, disp.means = TRUE)

# Balancing covariates between treatment groups (binary)
# and not requesting a target population
(ow3b <- optweight(treat ~ age + educ + married +
                     nodegree + re74, data = lalonde,
                   targets = NULL,
                   estimand = NULL))

bal.tab(ow3b, disp.means = TRUE)

# Balancing two treatments
(ow4 <- optweightMV(list(treat ~ age + educ + race + re74,
                         re75 ~ age + educ + race + re74),
                    data = lalonde))

summary(ow4)

bal.tab(ow4)

# Using a different norm
(ow1b <- optweight(treat ~ age + educ + married +
                    nodegree + re74, data = lalonde,
                  tols = c(.01, .02, .03, .04, .05),
                  estimand = "ATE",
                  norm = "l1"))

summary(ow1b, weight.range = FALSE)
summary(ow1, weight.range = FALSE)

# Allowing for negative weights
ow5 <- optweight(treat ~ age + educ + married + race +
                   nodegree + re74 + re75,
                 data = lalonde,
                 estimand = "ATE",
                 min.w = -Inf)

summary(ow5)
\dontshow{\}) # examplesIf}
}
\references{
Chattopadhyay, A., Cohn, E. R., & Zubizarreta, J. R. (2024). One-Step Weighting to Generalize and Transport Treatment Effect Estimates to a Target Population. \emph{The American Statistician}, 78(3), 280–289. \doi{10.1080/00031305.2023.2267598}

Källberg, D., & Waernbaum, I. (2023). Large Sample Properties of Entropy Balancing Estimators of Average Causal Effects. \emph{Econometrics and Statistics}. \doi{10.1016/j.ecosta.2023.11.004}

Wang, Y., & Zubizarreta, J. R. (2020). Minimal dispersion approximately balancing weights: Asymptotic properties and practical considerations. \emph{Biometrika}, 107(1), 93–105. \doi{10.1093/biomet/asz050}

Zubizarreta, J. R. (2015). Stable Weights that Balance Covariates for Estimation With Incomplete Outcome Data. \emph{Journal of the American Statistical Association}, 110(511), 910–922. \doi{10.1080/01621459.2015.1023805}
}
\seealso{
\code{\link[=optweight.fit]{optweight.fit()}}, the lower-level function that performs the fitting. Links on that page can help with diagnosing and fixing more subtle issues with the optimization.

\CRANpkg{sbw}, which was the inspiration for this package and provides some additional functionality for binary treatments.

\CRANpkg{WeightIt}, which provides a simplified interface to \code{optweight()} and a more efficient implementation of entropy balancing.
}
