% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TaskState.R
\name{TaskState}
\alias{TaskState}
\title{TaskState}
\description{
This class holds the state of a task deployed to an asynchronous backend
(i.e., \code{\link{AsyncBackend}}). See the \strong{Details} section for more
information.
}
\details{
The task state is useful to check if an asynchronous backend is free to
execute other operations. A task can only be in one of the following three
states at a time:
\itemize{
\item \code{task_not_started}: When \code{TRUE}, it indicates whether the backend is free
to execute another operation.
\item \code{task_is_running}: When \code{TRUE}, it indicates that there is a task running
on the backend.
\item \code{task_is_completed}: When \code{TRUE}, it indicates that the task has been
completed, but the backend is still busy because the task output has not
been retrieved.
}

The task state is determined based on the state of the background
\code{\link[callr:r_session]{session}} (i.e., see the \code{get_state} method for
\code{\link[callr:r_session]{callr::r_session}}) and the state of the task execution inferred from
polling the process (i.e., see the \code{poll_process} method for
\code{\link[callr:r_session]{callr::r_session}}) as follows:\tabular{ccccc}{
   Session State \tab Execution State \tab Not Started \tab Is Running \tab Is Completed \cr
   \code{idle} \tab \code{timeout} \tab \code{TRUE} \tab \code{FALSE} \tab \code{FALSE} \cr
   \code{busy} \tab \code{timeout} \tab \code{FALSE} \tab \code{TRUE} \tab \code{FALSE} \cr
   \code{busy} \tab \code{ready} \tab \code{FALSE} \tab \code{FALSE} \tab \code{TRUE} \cr
}
}
\examples{
# Handy function to print the task states all at once.
check_state <- function(session) {
    # Create a task state object and determine the state.
    task_state <- TaskState$new(session)

    # Print the state.
    cat(
        "Task not started: ", task_state$task_not_started, "\n",
        "Task is running: ", task_state$task_is_running, "\n",
        "Task is completed: ", task_state$task_is_completed, "\n",
        sep = ""
    )
}

# Create a specification object.
specification <- Specification$new()

# Set the number of cores.
specification$set_cores(cores = 2)

# Set the cluster type.
specification$set_type(type = "psock")

# Create an asynchronous backend object.
backend <- AsyncBackend$new()

# Start the cluster on the backend.
backend$start(specification)

# Check that the task has not been started (i.e., the backend is free).
check_state(backend$cluster)

{
    # Run a task in parallel (i.e., approx. 0.25 seconds).
    backend$sapply(
        x = 1:10,
        fun = function(x) {
            # Sleep a bit.
            Sys.sleep(0.05)

            # Compute something.
            output <- x + 1

            # Return the result.
            return(output)
        }
    )

    # And immediately check the state to see that the task is running.
    check_state(backend$cluster)
}

# Sleep for a bit to wait for the task to complete.
Sys.sleep(1)

# Check that the task is completed (i.e., the output needs to be retrieved).
check_state(backend$cluster)

# Get the output.
output <- backend$get_output(wait = TRUE)

# Check that the task has not been started (i.e., the backend is free again).
check_state(backend$cluster)

# Stop the backend.
backend$stop()

}
\seealso{
\code{\link{AsyncBackend}} and \code{\link{ProgressTrackingContext}}.
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{task_not_started}}{A logical value indicating whether the task
has been started. It is used to determine if the backend is free to
execute another operation.}

\item{\code{task_is_running}}{A logical value indicating whether the task is
running.}

\item{\code{task_is_completed}}{A logical value indicating whether the task
has been completed and the output needs to be retrieved.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-TaskState-new}{\code{TaskState$new()}}
\item \href{#method-TaskState-clone}{\code{TaskState$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TaskState-new"></a>}}
\if{latex}{\out{\hypertarget{method-TaskState-new}{}}}
\subsection{Method \code{new()}}{
Create a new \code{\link{TaskState}} object and determine the state of
a task on a given session.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TaskState$new(session)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{session}}{A \code{\link[callr:r_session]{callr::r_session}} object.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An object of class \code{\link{TaskState}}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-TaskState-clone"></a>}}
\if{latex}{\out{\hypertarget{method-TaskState-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{TaskState$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
