% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cluster_analysis.R
\name{cluster_analysis}
\alias{cluster_analysis}
\title{Cluster Analysis}
\usage{
cluster_analysis(
  x,
  n = NULL,
  method = "kmeans",
  include_factors = FALSE,
  standardize = TRUE,
  verbose = TRUE,
  distance_method = "euclidean",
  hclust_method = "complete",
  kmeans_method = "Hartigan-Wong",
  dbscan_eps = 15,
  iterations = 100,
  ...
)
}
\arguments{
\item{x}{A data frame.}

\item{n}{Number of clusters used for supervised cluster methods. If \code{NULL},
the number of clusters to extract is determined by calling \code{\link[=n_clusters]{n_clusters()}}. Note
that this argument does not apply for unsupervised clustering methods like
\code{dbscan}, \code{hdbscan}, \code{mixture}, \code{pvclust}, or \code{pamk}.}

\item{method}{Method for computing the cluster analysis. Can be \code{"kmeans"}
(default; k-means using \code{kmeans()}), \code{"hkmeans"} (hierarchical k-means
using \code{factoextra::hkmeans()}), \code{pam} (K-Medoids using \code{cluster::pam()}),
\code{pamk} (K-Medoids that finds out the number of clusters), \code{"hclust"}
(hierarchical clustering using \code{hclust()} or \code{pvclust::pvclust()}),
\code{dbscan} (DBSCAN using \code{dbscan::dbscan()}), \code{hdbscan} (Hierarchical DBSCAN
using \code{dbscan::hdbscan()}), or \code{mixture} (Mixture modelling using
\code{mclust::Mclust()}, which requires the user to run \code{library(mclust)}
before).}

\item{include_factors}{Logical, if \code{TRUE}, factors are converted to numerical
values in order to be included in the data for determining the number of
clusters. By default, factors are removed, because most methods that
determine the number of clusters need numeric input only.}

\item{standardize}{Standardize the dataframe before clustering (default).}

\item{verbose}{Toggle warnings and messages.}

\item{distance_method}{Distance measure to be used for methods based on
distances (e.g., when \code{method = "hclust"} for hierarchical clustering. For
other methods, such as \code{"kmeans"}, this argument will be ignored). Must be
one of \code{"euclidean"}, \code{"maximum"}, \code{"manhattan"}, \code{"canberra"}, \code{"binary"}
or \code{"minkowski"}. See \code{\link[=dist]{dist()}} and \code{pvclust::pvclust()} for more
information.}

\item{hclust_method}{Agglomeration method to be used when \code{method = "hclust"}
or \code{method = "hkmeans"} (for hierarchical clustering). This should be one
of \code{"ward"}, \code{"ward.D2"}, \code{"single"}, \code{"complete"}, \code{"average"},
\code{"mcquitty"}, \code{"median"} or \code{"centroid"}. Default is \code{"complete"} (see
\code{\link[=hclust]{hclust()}}).}

\item{kmeans_method}{Algorithm used for calculating kmeans cluster. Only applies,
if \code{method = "kmeans"}. May be one of \code{"Hartigan-Wong"} (default),
\code{"Lloyd"} (used by SPSS), or \code{"MacQueen"}. See \code{\link[=kmeans]{kmeans()}} for details on
this argument.}

\item{dbscan_eps}{The 'eps' argument for DBSCAN method. See \code{\link[=n_clusters_dbscan]{n_clusters_dbscan()}}.}

\item{iterations}{The number of replications.}

\item{...}{Arguments passed to or from other methods.}
}
\value{
The group classification for each observation as vector. The
returned vector includes missing values, so it has the same length
as \code{nrow(x)}.
}
\description{
Compute hierarchical or kmeans cluster analysis and return the group
assignment for each observation as vector.
}
\details{
The \code{print()} and \code{plot()} methods show the (standardized) mean value for
each variable within each cluster. Thus, a higher absolute value indicates
that a certain variable characteristic is more pronounced within that
specific cluster (as compared to other cluster groups with lower absolute
mean values).
}
\note{
There is also a \href{https://easystats.github.io/see/articles/parameters.html}{\code{plot()}-method} implemented in the \href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\examples{
set.seed(33)
# K-Means ====================================================
rez <- cluster_analysis(iris[1:4], n = 3, method = "kmeans")
rez # Show results
predict(rez) # Get clusters
summary(rez) # Extract the centers values (can use 'plot()' on that)
cluster_discrimination(rez) # Perform LDA

# Hierarchical k-means (more robust k-means)
if (require("factoextra", quietly = TRUE)) {
  rez <- cluster_analysis(iris[1:4], n = 3, method = "hkmeans")
  rez # Show results
  predict(rez) # Get clusters
}

# Hierarchical Clustering (hclust) ===========================
rez <- cluster_analysis(iris[1:4], n = 3, method = "hclust")
rez # Show results
predict(rez) # Get clusters

# K-Medoids (pam) ============================================
if (require("cluster", quietly = TRUE)) {
  rez <- cluster_analysis(iris[1:4], n = 3, method = "pam")
  rez # Show results
  predict(rez) # Get clusters
}

# PAM with automated number of clusters
if (require("fpc", quietly = TRUE)) {
  rez <- cluster_analysis(iris[1:4], method = "pamk")
  rez # Show results
  predict(rez) # Get clusters
}

# DBSCAN ====================================================
if (require("dbscan", quietly = TRUE)) {
  # Note that you can assimilate more outliers (cluster 0) to neighbouring
  # clusters by setting borderPoints = TRUE.
  rez <- cluster_analysis(iris[1:4], method = "dbscan", dbscan_eps = 1.45)
  rez # Show results
  predict(rez) # Get clusters
}

# Mixture ====================================================
if (require("mclust", quietly = TRUE)) {
  library(mclust) # Needs the package to be loaded
  rez <- cluster_analysis(iris[1:4], method = "mixture")
  rez # Show results
  predict(rez) # Get clusters
}
}
\references{
\itemize{
\item Maechler M, Rousseeuw P, Struyf A, Hubert M, Hornik K (2014) cluster: Cluster
Analysis Basics and Extensions. R package.
}
}
\seealso{
\itemize{
\item \code{\link[=n_clusters]{n_clusters()}} to determine the number of clusters to extract,
\code{\link[=cluster_discrimination]{cluster_discrimination()}} to determine the accuracy of cluster group
classification via linear discriminant analysis (LDA) and
\code{\link[=check_clusterstructure]{check_clusterstructure()}} to check suitability of data
for clustering.
\item https://www.datanovia.com/en/lessons/
}
}
