% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/partR2.R
\name{partR2}
\alias{partR2}
\title{Partitioning R2 (R-square) in mixed models}
\usage{
partR2(
  mod,
  partvars = NULL,
  data = NULL,
  R2_type = "marginal",
  max_level = NULL,
  nboot = NULL,
  CI = 0.95,
  parallel = FALSE,
  expct = "meanobs",
  olre = TRUE,
  partbatch = NULL,
  allow_neg_r2 = FALSE
)
}
\arguments{
\item{mod}{Fitted lme4 model (a merMod object).}

\item{partvars}{Character vector specifying the predictors (fixed effects) for which to partition the R2.
Can be main effects like c("Var1", "Var2") and interactions ("Var1:Var2"). Predictors
specified in partvars have to be named precisely like the terms in the formula to
fit the model.}

\item{data}{The data.frame used to fit the lme4 model. If not provided,
partR2 will try to fetch it.}

\item{R2_type}{"marginal" or "conditional" R2. With "marginal", the variance explained
by fixed effects is calculated. With "conditional", the variance explained by
both fixed and random effects is calculated.}

\item{max_level}{Level up to which shared semi-partial R2s are calculated.
The number of sets for which to calculate R2 increases exponentially,
i.e. for 10 variables 2^10 - 1 R2s  can be calculated. If you are
only interested in the unique but not the shared effects, use max_level = 1.
If interested in unique effects and combinations of two terms,
use max_level = 2 etc.}

\item{nboot}{Number of parametric bootstrap iterations for confidence interval estimation
(defaults to NULL, i.e. no bootstrapping). Larger numbers of bootstraps give a better
asymptotic CI, but may be time-consuming. Bootstrapping can be switched on by setting
\code{nboot = 1000}.}

\item{CI}{Width of the required confidence interval between 0 and 1 (defaults to
0.95).}

\item{parallel}{If TRUE, computation uses \code{future} within \code{furrr::map} which allows
parallelisation. However, it is necessary to specify a plan before running
\code{partR2()}. To see which options you have, check \code{?future::plan} and have
a look at our vignette for details. When running RStudio,
usually \code{plan(multisession, workers = 4)} is a good choice,
when you want to use 4 cores. To detect how many cores you have, use
\code{parallel::detectCores()}. If no plan is specified, \code{partR2} will simply run
sequentially.}

\item{expct}{A string specifying the method for estimating the expectation in Poisson models
with log link and in Binomial models with logit link (in all other cases the argument is ignored).
The only valid terms are 'meanobs', 'latent', 'none' (and 'liability for binary and proportion data).
With the default 'meanobs', the expectation is estimated as the mean of the observations in the sample.
With 'latent', the expectation is estimated from estimates of the intercept and variances on the link scale.
While this is a preferred solution, it is susceptible to the distribution of fixed effect covariates and gives
appropriate results typically only when all covariances are centered to zero. With 'liability'
estimates follow formulae as presented in Nakagawa & Schielzeth (2010). With 'none', R2 is calculated
without distribution specific variance in the denominator.}

\item{olre}{Logical, defaults to TRUE. This argument allows the user to prevent the automatic fitting of
an observation level random effect (by setting it to FALSE) in Poisson and binomial models.
The OLRE is used to account for overdispersion.}

\item{partbatch}{List of character vectors with predictors that should be fitted and
removed together. For example, partbatch = list(batch1 = c("V1", "V2", "V3"),
batch2 = c("V4", "V5", "V6")) would calculate part R2 only for combinations of
predictors which contain the variables V1, V2, V3 together or/and V4,V5,V6 together.
This is useful when the number of potential subsets gets too large to
be computationally practical, for example when dummy coding is used.
See our vignette for details. This feature is still experimental and
should be used with caution.}

\item{allow_neg_r2}{Calculating part R2 involves fitting two models, one with
and one without the predictor of interest. In cases where the predictor
has little association with the response, the resulting part R2 value
can become negative. By default we set negative values to 0, but by
setting this parameter to TRUE, R2 values can become negative.}
}
\value{
Returns an object of class \code{partR2} that is a a list with the following elements:
\item{call}{model call}
\item{R2_type}{Marginal or conditional R2}
\item{R2}{R2 and confidence intervals for full model and semi-partial R2 for
predictors and their combinations}
\item{SC}{Structure coefficients and confidence intervals. SC are the
correlation between a predictor and the predicted response.}
\item{IR2}{Inclusive R2. This is SC^2 * R2_full.}
\item{BW}{Standardised model estimates (beta weights) for fixed effects. Beta weights for Gaussian models
are calculated as beta * sd(x)/sd(y), with beta being the estimated
slope of a fixed effect for predictor x and response y. Beta weight for Non-Gaussian
models are calculated as beta * sd(x). Beta weights for interactions or polynomial
terms are not informative at the moment and we recommend users to standardise
variables themselves before fitting the model and to look at the model estimates (Ests)
instead of beta weights (BW) in the partR2 output. See vignette for details. }
\item{Ests}{Model estimates and confidence intervals.}
\item{R2_boot}{Parametric bootstrap samples for R2 for full model and partitions}
\item{SC_boot}{Parametric bootstrap samples for structure coefficients}
\item{IR2_boot}{Parametric bootstrap samples for inclusive R2 values}
\item{BW_boot}{Parametric bootstrap samples for beta weights}
\item{Ests_boot}{Parametric bootstrap samples for model estimates}
\item{partvars}{Predictors to partition}
\item{CI}{Coverage of the confidence interval as specified by the \code{CI} argument.}
\item{boot_warnings}{Potential warnings from estimating partial R2s during
parametric bootstrapping}
\item{boot_message}{Potential messages from estimating partial R2s
during parametric bootstrapping. Common are for example singularity messages
in lme4.}
}
\description{
R2, semi-partial (part) R2 for predictors and their combinations as well as inclusive R2,
structure coefficients and beta weights for Gaussian, Poisson and binomial
mixed models.
}
\examples{

data(biomass)
library(lme4)

# scale data
biomass[] <- lapply(biomass, function(x) if (is.double(x)) scale(x) else x)

# Gaussian data
mod <- lmer(Biomass ~ Year + Temperature + Precipitation + SpeciesDiversity + (1 | Population),
  data = biomass)

# R2
(R2_1 <- partR2(mod))

# R2 with CI
(R2_2 <- partR2(mod, R2_type = "marginal", nboot = 15, CI = 0.95))

# Part (semi-partial) R2s with CIs
(R2_3 <- partR2(mod,
  partvars = c("SpeciesDiversity", "Temperature", "Precipitation"),
  R2_type = "marginal", nboot = 10, CI = 0.95
))
}
\references{
Nakagawa, S., & Schielzeth, H. (2013). \emph{A general and simple method for obtaining R2 from
generalized linear mixed-effects models}. Methods in Ecology and Evolution, 4(2), 133-142.

Nakagawa, S., Johnson, P. C., & Schielzeth, H. (2017). \emph{The coefficient of
determination R2 and intra-class correlation coefficient from generalized
linear mixed-effects models revisited and expanded}. Journal of the Royal Society Interface, 14(134), 20170213.
}
