all.tests.pass <- TRUE
all.tests.error.count <- 0

test <- function(expr, out="", val=eval.parent(parse(text=expr), 1), tol=1e-4) {
    # expr is a string representing an R expression, and
    # out is the output that is expected.  Prints and evaluates
    # expr.  If out is given and it matches the output of
    # evaluating expr, returns TRUE.  Otherwise, returns FALSE.
    
    cat(expr, "-> ")
    
    p <- function (v) {
        if (length(v) < 5) {
            cat(v)
        } else {
            cat(class(v), "(", length(val), ")")
        }
    }
    p(val)
        
    result <- all.equal(val, out, tolerance=tol)
    if (!isTRUE(result)) {
        if (!missing(out)) {
            cat(" (Expecting ")
            p(out)
            cat(")")
        }
        cat("\nERROR: ", result, "\n")
        all.tests.pass <<- FALSE
        all.tests.error.count <<- all.tests.error.count + 1
    } else {
        cat(" OK\n")
    }
    
    isTRUE(result)
}

assert <- function (expr, out) {
    # expr is astring representing an R expression,
    # and out is the output that is expected.  Prints
    # and evaluates expr.  If out matches the output of
    # evaluating expr, returns TRUE.  Otherwise, stops
    # the execution with an error message.
    if (!test(expr, out)) {
        stop("Expression ", deparse(substitute(expr)), 
             " does not evaluate to its expected value\n")
    }
}

build_par <- function (rho, eps_M, eps_R, R0=0, M0=0) {
    R <- R0
    M <- M0
    X <- numeric()
    for (i in 1:length(eps_M)) {
        M <- rho * M + eps_M[i]
        R <- R + eps_R[i]
        X[i] <- M + R
    }
    X
}

data.L <- structure(c(37.8517816659277, 37.3893346323175, 37.4385311252548, 
37.1138342718688, 37.2319058549183, 37.8616209645152, 37.7238707842909, 
37.900978158865, 37.6156384998289, 37.4188525280799, 37.7632279786407, 
37.9108174574525, 37.9403353532148, 38.314228699538, 37.8222637701654, 
37.5664420068916, 37.3401381393802, 37.0252805845818, 36.7202623283708, 
36.7104230297833, 37.2417451535057, 37.3893346323175, 37.9895318461521, 
37.7632279786407, 37.7435493814658, 37.8714602631026, 37.5861206040665, 
37.487727618192, 37.8025851729905, 37.5369241111293, 36.985923390232, 
37.4582097224297, 37.6845135899411, 38.1076034292015, 38.0879248320266, 
38.5405325670494, 38.511014671287, 38.6389255529239, 38.7798536105174, 
38.5728963231423, 38.6615923034459, 38.3068083822315, 38.2870981643863, 
37.6070956487254, 37.6563711933385, 37.7647773914873, 38.0899959859339, 
38.0111551145529, 38.7305780659043, 38.4546350160709, 38.9868108978925, 
38.9079700265115, 39.1050722049639, 39.1247824228092, 38.7699985015948, 
38.2378226197732, 38.6221718677554, 39.2824641655711, 39.1149273138865, 
39.0557966603508, 38.8981149175889, 39.2923192744937, 39.7850747206248, 
39.4795663440236, 39.1346375317318, 38.9966660068151, 38.4349247982256, 
37.8337631539457, 38.2279675108506, 38.8586944818984, 38.346228817922, 
38.6813025212912, 39.3415948191068, 39.0755068781961, 38.9769557889698, 
39.2627539477259, 39.0459415514282, 39.6569583046307, 40.0511626615356, 
40.4552221273631, 40.4158016916726, 40.5340629987441, 40.8888469199585, 
40.6720345236608, 40.5439181076667, 40.1792790775297, 40.1300035329166, 
40.3172506024464, 40.1694239686071, 40.40594658275, 40.0511626615356, 
39.5288418886367, 39.1346375317318, 38.5433309963745, 38.1688368573148, 
37.7647773914873, 38.3955043625351, 38.6320269766781, 38.6517371945233, 
38.7995638283626, 38.6517371945233, 39.0853619871187, 38.2690477622191, 
38.3874972265335, 37.8643454258119, 37.8051206936547, 38.0025364675119, 
39.0192277028765, 39.0488400689551, 39.3548345184338, 39.0093569141837, 
39.1574187445766, 38.7231040420907, 39.196901899348, 39.9372110513125, 
40.183980768634, 40.3419133877198, 40.3813965424912, 39.3252221523552, 
39.552250292291, 38.8119411403264, 38.8316827177122, 39.335092941048, 
39.621345813141, 40.3813965424912, 40.4801044294198, 40.8058404562842, 
40.0655313043197, 39.976694206084, 39.6805705452981, 39.4338008279766, 
39.8286323756911, 39.5719918696767, 40.3715257537984, 40.1642391912483, 
40.1938515573269, 40.4899752181127, 40.4603628520341, 40.0260481495483, 
39.9470818400054, 39.7792784322267, 39.7792784322267, 40.4603628520341, 
41.1611888492272, 39.0290984915694, 39.0784524350337, 38.9402613933336, 
38.9501321820265, 39.8286323756911, 39.8977278965411, 40.0556605156268, 
39.9372110513125, 39.9470818400054, 39.9865649947768, 39.9372110513125, 
39.9668234173911, 39.9174694739268, 39.9964357834697, 39.9569526286982, 
40.0852728817054, 39.9668234173911, 39.9174694739268, 39.2561266315052, 
39.7101829113767, 39.8977278965411, 39.7003121226839, 39.9108897674813, 
39.7922544746377, 40.1877054507832, 40.3261132924341, 40.464521134085, 
40.464521134085, 40.7116779941759, 40.7413368173868, 41.008266226285, 
40.9094034822486, 41.8387132761905, 42.204505429125, 41.8090544529796, 
41.5717838672923, 41.2455368119723, 40.9687211286705, 40.9588348542668, 
41.0379250494959, 40.7709956405977, 40.4941799572959, 40.8105407382123, 
40.790768189405, 41.0774701471105, 41.0576975983032, 40.8204270126159, 
41.4828073976596, 41.4828073976596, 41.6014426905032, 41.3246270072014, 
41.0774701471105, 41.1367877935323, 41.008266226285, 41.2158779887614, 
41.6212152393105, 42.204505429125, 42.6790466004996, 42.0265524898596, 
41.9672348434378, 41.334513281605, 41.5421250440814, 41.9178034714196, 
41.9079171970159, 41.7991681785759, 42.4318897404087, 41.6805328857323, 
41.6904191601359, 41.8485995505941, 40.7314505429832, 40.1580466275722, 
40.6622466221577, 40.5238387805068, 40.1481603531686, 39.4660074193177, 
39.3770309496849, 40.2569093716086, 40.3755446644523, 40.2667956460122, 
40.3755446644523, 40.8303132870195, 40.6227015245432, 40.2766819204159, 
40.4479793605475, 40.5667983941647, 40.4776841189518, 40.3192587407954, 
40.5469952218952, 40.6559126693777, 40.8737475643427, 40.8737475643427, 
40.6361094971081, 41.0915824593077, 41.0321729424991, 40.5866015664343, 
40.249947637852, 40.6262079109734, 41.2302046651945, 40.7648301168602, 
40.8341412198036, 40.4974872912213, 40.339061913065, 40.1311286042347, 
40.0915222596956, 39.7845730895177, 40.3489634991997, 41.121287217712, 
41.2995157681379, 41.646071282855, 41.6955792135288, 41.646071282855, 
41.6955792135288, 42.3391823122891, 42.2302648648066, 42.1213474173241, 
42.081741072785, 42.0520363143807, 42.3094775538848, 42.3985918290977, 
42.9629822387798, 42.9431790665102, 42.8144584467582, 43.042194927858, 
42.8936711358364, 43.2402266505534, 42.9332774803755, 42.9431790665102, 
43.6461916820791, 43.2798329950925, 43.3788488564402, 43.4877663039228, 
43.3095377534968, 43.4184552009793, 43.0818012723971, 43.2204234782839, 
43.6461916820791, 43.6461916820791, 43.477864717788, 43.4877663039228, 
42.9134743081059, 42.9827854110493, 43.2501282366882, 41.9827252114373, 
42.2599696232109, 42.6362298963323, 42.6857378270061, 42.7550489299496, 
42.428296587502, 42.6560330686018, 42.7581687685824, 42.85732964235, 
43.2143087879133, 43.8687705547794, 43.5911201082302, 43.6010361956069, 
44.2654140498497, 44.2257497003427, 43.7894418557653, 43.432462710202, 
43.6407005451139, 43.4820431470858, 43.6109522829837, 43.432462710202, 
43.6208683703604, 43.3531340111879, 43.7001970693745, 43.6010361956069, 
43.9183509916632, 43.5613718460999, 43.8588544674026, 43.432462710202, 
43.6010361956069, 43.5316235839696, 43.6407005451139, 43.6307844577372, 
43.6208683703604, 42.2623643997445, 43.0259031277549, 42.7581687685824, 
42.6391757200613, 42.817665292843, 42.6292596326846, 43.2738053121739, 
44.1365049139519, 44.6323092827898, 44.0670923023145, 44.1563370887054, 
44.29516231198, 44.3546588362405, 44.5529805837757, 45.0388688652369, 
45.237190612772, 45.2570227875256, 45.3958480108002, 45.0091206031066, 
45.0686171273672, 45.0785332147439, 45.6140019330889, 45.8916523796381, 
45.6536662825959, 46.0503097776662, 46.3180441368387, 46.2684636999549, 
46.0007293407824, 45.6734984573494, 45.5644214962051, 45.812323680624, 
45.7924915058705, 45.8649778315189, 45.4975813864516, 45.4975813864516, 
44.9315922683748, 44.3953920512495, 44.7925773972682, 45.4181443172478, 
45.0308886048795, 44.4648994868027, 44.1173623090363, 44.6734217934626, 
44.2663068137933, 44.554266189657, 44.8521551991711, 44.3358142493466, 
43.2733434487464, 43.2733434487464, 42.9158766373295, 43.5414435573091, 
43.6605991611147, 44.1967993782401, 44.0875734080849, 44.365603150298, 
44.1868697445896, 43.9485585369783, 44.554266189657, 44.6734217934626, 
44.9514515356758, 44.6833514271131, 45.1301849413842, 45.358566515345, 
45.4876517528011, 45.3387072480441, 45.6663851585096, 46.013922336276, 
46.182726108334, 46.4309669495957, 46.1032890391302, 45.8351889305675, 
45.7954703959657, 45.7259629604124, 45.7358925940628, 45.4280739508983, 
45.2294812778889, 46.2423039102368, 45.9642741680236, 45.7954703959657, 
45.5968777229563, 45.3287776143936, 45.7061036931114, 45.8550481978685, 
46.0933594054798, 46.8678708302164, 47.1955487406819, 46.6394892562556, 
46.0834297718293, 45.3486368816945, 45.4479332181992, 45.2394109115394, 
45.6068073566067, 45.6266666239077, 45.5472295547039, 44.7528588626664, 
44.755841922559, 44.348157070565, 44.2089476089085, 44.2785523397368, 
44.7160677906571, 44.5569712630497, 44.4376488673442, 45.2231879724058, 
45.4717762967923, 45.8496305498599, 45.7601387530808, 45.9689529455655, 
46.7346049846761, 47.03291097394, 47.1522333696456, 47.3312169632039, 
47.1721204355965, 46.8340403144308, 46.8937015122835, 46.8937015122835, 
47.0726851058419, 46.6848873197988, 46.4760731273141, 46.8937015122835, 
46.7843226495534, 45.9490658796146, 46.7942661825289, 46.5953955230197, 
45.9092917477127, 46.2274848029275, 47.8582242109035, 48.0570948704127, 
48.2857961288484, 47.7886194800752, 48.2758525958729, 48.4846667883576, 
48.792916310597, 48.643763315965, 48.922182239278, 48.1664737331428, 
48.3156267277748, 48.5343844532349, 48.4250055905049, 48.8625210414252, 
48.6537068489405, 48.0372078044618, 48.0670384033882, 48.0173207385109, 
47.848280677928, 48.0968690023146, 47.4704264248604, 48.2957396618238, 
48.126699601241, 47.5996923535414, 47.7786759470997, 47.7886194800752, 
47.7985630130507, 47.6692970843697, 47.4704264248604, 47.311329897253, 
47.8681677438789, 47.9676030736335, 47.7786759470997, 47.6422706997462, 
47.2937901408139, 47.1444413298429, 47.1444413298429, 46.8258305331048, 
46.557002673357, 46.3080879884054, 47.0847018054545, 47.3634862526004, 
47.2041808542313, 46.3578709253957, 46.407653862386, 46.3578709253957, 
46.5769158481532, 46.2383918766189, 47.1942242668333, 47.0647886306584, 
47.4132691895907, 47.5626180005617, 47.7916195107172, 47.8214892729114, 
47.4630521265811, 47.751793161125, 48.0305776082708), .Dim = c(502L, 
1L), .Dimnames = list(NULL, "L"), index = structure(c(15342, 
15343, 15344, 15345, 15348, 15349, 15350, 15351, 15352, 15356, 
15357, 15358, 15359, 15362, 15363, 15364, 15365, 15366, 15369, 
15370, 15371, 15372, 15373, 15376, 15377, 15378, 15379, 15380, 
15383, 15384, 15385, 15386, 15387, 15391, 15392, 15393, 15394, 
15397, 15398, 15399, 15400, 15401, 15404, 15405, 15406, 15407, 
15408, 15411, 15412, 15413, 15414, 15415, 15418, 15419, 15420, 
15421, 15422, 15425, 15426, 15427, 15428, 15429, 15432, 15433, 
15434, 15435, 15439, 15440, 15441, 15442, 15443, 15446, 15447, 
15448, 15449, 15450, 15453, 15454, 15455, 15456, 15457, 15460, 
15461, 15462, 15463, 15464, 15467, 15468, 15469, 15470, 15471, 
15474, 15475, 15476, 15477, 15478, 15481, 15482, 15483, 15484, 
15485, 15489, 15490, 15491, 15492, 15495, 15496, 15497, 15498, 
15499, 15502, 15503, 15504, 15505, 15506, 15509, 15510, 15511, 
15512, 15513, 15516, 15517, 15518, 15519, 15520, 15523, 15524, 
15526, 15527, 15530, 15531, 15532, 15533, 15534, 15537, 15538, 
15539, 15540, 15541, 15544, 15545, 15546, 15547, 15548, 15551, 
15552, 15553, 15554, 15555, 15558, 15559, 15560, 15561, 15562, 
15565, 15566, 15567, 15568, 15569, 15572, 15573, 15574, 15575, 
15576, 15579, 15580, 15581, 15582, 15583, 15587, 15588, 15589, 
15590, 15593, 15594, 15595, 15596, 15597, 15600, 15601, 15602, 
15603, 15604, 15607, 15608, 15609, 15610, 15611, 15614, 15615, 
15616, 15617, 15618, 15621, 15622, 15623, 15624, 15625, 15628, 
15629, 15630, 15631, 15632, 15635, 15636, 15637, 15638, 15639, 
15644, 15645, 15646, 15649, 15650, 15651, 15652, 15653, 15656, 
15657, 15658, 15659, 15660, 15663, 15664, 15665, 15667, 15670, 
15671, 15672, 15673, 15674, 15677, 15678, 15679, 15680, 15681, 
15684, 15685, 15686, 15687, 15688, 15691, 15692, 15693, 15694, 
15695, 15698, 15700, 15701, 15702, 15705, 15707, 15708, 15709, 
15712, 15713, 15714, 15715, 15716, 15719, 15720, 15721, 15722, 
15723, 15727, 15728, 15729, 15730, 15733, 15734, 15735, 15736, 
15737, 15740, 15741, 15742, 15743, 15744, 15747, 15748, 15749, 
15750, 15751, 15755, 15756, 15757, 15758, 15761, 15762, 15763, 
15764, 15765, 15768, 15769, 15770, 15771, 15772, 15775, 15776, 
15777, 15778, 15779, 15782, 15783, 15784, 15785, 15786, 15789, 
15790, 15791, 15792, 15796, 15797, 15798, 15799, 15800, 15803, 
15804, 15805, 15806, 15807, 15810, 15811, 15812, 15813, 15814, 
15817, 15818, 15819, 15820, 15821, 15824, 15825, 15826, 15827, 
15828, 15831, 15832, 15833, 15834, 15835, 15838, 15839, 15840, 
15841, 15842, 15845, 15846, 15847, 15848, 15849, 15853, 15854, 
15855, 15856, 15859, 15860, 15861, 15862, 15863, 15866, 15867, 
15868, 15869, 15870, 15873, 15874, 15875, 15876, 15877, 15880, 
15881, 15882, 15883, 15884, 15887, 15888, 15889, 15891, 15894, 
15895, 15896, 15897, 15898, 15901, 15902, 15903, 15904, 15905, 
15908, 15909, 15910, 15911, 15912, 15915, 15916, 15917, 15918, 
15919, 15922, 15923, 15924, 15925, 15926, 15929, 15930, 15931, 
15932, 15933, 15936, 15937, 15938, 15939, 15940, 15943, 15944, 
15945, 15946, 15947, 15951, 15952, 15953, 15954, 15957, 15958, 
15959, 15960, 15961, 15964, 15965, 15966, 15967, 15968, 15971, 
15972, 15973, 15974, 15975, 15978, 15979, 15980, 15981, 15982, 
15985, 15986, 15987, 15988, 15989, 15992, 15993, 15994, 15995, 
15996, 15999, 16000, 16001, 16002, 16003, 16006, 16007, 16008, 
16009, 16010, 16013, 16014, 16015, 16016, 16017, 16020, 16021, 
16022, 16023, 16024, 16027, 16028, 16029, 16030, 16031, 16034, 
16035, 16036, 16038, 16041, 16042, 16043, 16044, 16045, 16048, 
16049, 16050, 16051, 16052, 16055, 16056, 16057, 16058, 16059, 
16062, 16063, 16065, 16066, 16069, 16070), class = "Date"), class = "zoo")

data.IBM <- structure(c(176.668606104443, 175.947896814914, 175.113391321774, 
173.102991724665, 172.202105112753, 171.936580637663, 172.89436535138, 
171.215871348133, 169.897731989651, 170.694305414921, 171.708988230443, 
171.187422297231, 178.773835871227, 180.158356348482, 182.007544657143, 
181.817884317793, 181.106658045231, 180.613541162922, 182.548076624291, 
182.642906793966, 182.661872827901, 181.628223978444, 183.629140558585, 
182.85153316725, 183.354133066528, 183.687331127656, 183.858690130522, 
183.182774063662, 183.373172955735, 182.992375171589, 183.0209350054, 
183.753970739882, 184.134768524028, 184.106208690217, 184.563166031193, 
188.123625312962, 188.266424482017, 188.047465756133, 188.475863263298, 
187.28587018784, 188.047465756133, 189.266018665402, 191.027208417079, 
187.790427251834, 188.275944426621, 190.218013125768, 190.989128638665, 
191.350886533604, 193.997431133422, 194.892305926166, 196.110858835435, 
196.120378780038, 195.844300386532, 194.444868529794, 194.863746092355, 
195.625341660648, 195.615821716044, 197.795889030283, 197.234212298667, 
197.338931689307, 198.271886260466, 198.633644155405, 199.414279612905, 
199.442839446716, 196.158458558453, 195.606301771441, 195.101744707447, 
192.617039165891, 192.855037780982, 195.463502602386, 193.064476562263, 
192.988317005434, 197.491250802966, 190.522651353085, 189.932414787658, 
190.018094289091, 189.085139717932, 190.398892073238, 193.797512296745, 
195.711021162081, 196.881974348331, 197.13901285263, 198.014847756167, 
198.071967423789, 197.291331966289, 195.149344430465, 193.968871299611, 
192.611375995296, 192.372380343128, 191.770111299664, 192.315021386608, 
190.661171473605, 190.278778430136, 190.93840643012, 189.179398430163, 
187.257873386731, 189.055120691035, 188.156497038883, 187.487309212813, 
187.458629734552, 185.747420865029, 187.812343299761, 185.967296865024, 
184.409045212888, 180.75719164776, 180.240961039076, 180.8719095608, 
185.451066256341, 185.881258430243, 186.550446256314, 184.036211995506, 
185.986416517197, 184.600241734622, 186.512206951967, 190.336137386656, 
189.561791473631, 190.173620343182, 190.030222951881, 184.877476691137, 
185.173831299826, 184.370805908541, 183.500861734649, 184.504643473755, 
182.975071299879, 186.97107860413, 187.210074256298, 187.305672517165, 
186.693843647615, 182.984631125966, 181.321221386876, 178.061320691304, 
177.095778256545, 175.030855821812, 177.822325039136, 176.656026256555, 
175.566206082669, 179.963726082561, 186.741642778048, 183.978853038985, 
182.430161212936, 181.961729734687, 182.669156865104, 185.412826951994, 
187.745424517154, 188.022659473669, 187.353471647598, 186.588685560661, 
185.89081825633, 189.781667473626, 190.011103299707, 191.129602951854, 
191.081599736325, 190.495960506866, 191.331216457077, 191.062398450113, 
190.37115214649, 190.476759220654, 192.819316138489, 193.184140576512, 
192.492894272889, 190.716775298301, 189.372685263478, 187.884585582067, 
189.871918704984, 187.874984938961, 187.087732204279, 187.289345709502, 
185.647635738397, 187.068530918067, 186.770910981785, 187.250943137079, 
191.148804238066, 191.532829962301, 192.924923212654, 195.152272413218, 
195.632304568512, 198.118871132934, 198.550900072699, 198.877321938299, 
198.800516793452, 198.186075634676, 197.946059557029, 197.754046694911, 
197.091602320605, 196.793982384323, 195.853119359947, 197.68684219317, 
199.165341231475, 202.064735449451, 201.459894933781, 202.103138021875, 
201.987930304604, 202.179943166722, 201.440693647569, 199.683775959193, 
197.600436405217, 197.542832546582, 199.501363740181, 200.586236411146, 
202.573569534063, 192.617702633265, 187.174137992232, 185.638035095291, 
186.636501978302, 183.61229939995, 183.103465315339, 183.948321908656, 
185.551629307338, 186.761310338679, 189.276678832419, 185.705239597032, 
186.386885257549, 187.279745066396, 184.329091066277, 183.306969092379, 
182.863406726348, 182.487342981235, 181.590575589042, 178.8809880922, 
179.208838536658, 180.259888490949, 183.548035595657, 182.43912968058, 
183.49017963487, 186.575830876825, 185.987628608828, 184.396589687195, 
185.119789197028, 184.685869491128, 183.278041111986, 182.709124164251, 
182.593412242677, 181.908783373369, 182.921262687135, 185.090861216635, 
185.736919445419, 187.260459746134, 186.055127229745, 185.129431857159, 
184.907650674143, 186.70118545853, 188.697216105669, 188.109013837672, 
187.810091373607, 186.508332255908, 185.524780922534, 185.090861216635, 
185.823703386599, 183.046617268839, 184.70515481139, 189.333631674322, 
188.292224380163, 187.057963883381, 186.238337772236, 185.977985948697, 
185.447639641486, 185.987628608828, 187.501526249412, 185.736919445419, 
185.621207523845, 185.707991465025, 186.730113438923, 187.520811569674, 
189.073279850782, 197.404538204061, 197.115258400127, 197.645604707338, 
197.607034066814, 196.61384007331, 196.247418988327, 195.813499282428, 
197.848100570092, 196.507770811867, 195.543504798757, 194.652646006963, 
193.413190296641, 195.291740357597, 193.81988670159, 193.703687728747, 
193.752103967432, 193.326041067009, 194.613913016015, 193.97481866538, 
192.996810643954, 192.047852365739, 194.720428741121, 191.253826051314, 
192.832195432427, 195.921151460495, 194.468664299962, 196.482779829235, 
198.690560313246, 199.988115509989, 201.779516341314, 202.78657410595, 
203.716165888692, 203.425668456585, 203.880781100219, 205.342951508489, 
208.964486162086, 208.11236036124, 206.456524998232, 206.67923969618, 
208.247925829556, 205.536616463227, 205.362318003963, 204.06476280722, 
205.633448940596, 204.210011523273, 206.543674227864, 205.65281543607, 
207.570098487974, 205.923946372703, 204.616707928222, 202.776890858213, 
202.689741628581, 202.592909151212, 205.284852022068, 206.175710813862, 
204.684490662381, 202.63164214216, 205.284852022068, 203.028655299372, 
200.588476869676, 183.98170700091, 181.880442242005, 185.540709886549, 
185.637542363918, 187.806589856981, 188.155186775509, 192.841878680164, 
196.12449966297, 193.306674571535, 195.979250946917, 198.032099467137, 
196.356897608655, 197.179973666291, 199.261901550867, 197.724777224872, 
198.921399326754, 196.975672331823, 197.695591319948, 197.802606304669, 
199.135429296196, 202.783667411692, 201.966462073821, 202.987968746159, 
201.373015340367, 200.565538637471, 200.137478698586, 202.141577503365, 
202.27777839301, 203.67870182936, 202.375064742756, 203.279827795399, 
200.594724542395, 197.238345476139, 198.269580783452, 200.750382701989, 
199.45647425036, 198.444696212996, 195.740135690042, 198.240394878528, 
196.712999187508, 197.530204525378, 199.31054472574, 196.460054678167, 
191.9946112248, 190.155899214591, 188.288001299457, 189.688924735807, 
189.572181116111, 190.340743279109, 185.923943000616, 186.089329795185, 
186.303359764628, 188.005870885192, 189.640281560934, 189.688924735807, 
186.108787065134, 187.033007387727, 187.568082311333, 186.857891958183, 
188.735518508291, 188.589588983671, 189.270593431897, 192.617243863178, 
188.288001299457, 188.823076223063, 189.688924735807, 191.274692236676, 
191.86813897013, 191.9946112248, 190.88554683769, 190.690974138197, 
189.747296545655, 190.496401438704, 189.864040165351, 190.194813754489, 
185.80719938092, 184.36016373009, 183.744195851696, 183.636645904675, 
184.878358929374, 184.223281979336, 183.353105135256, 181.651860518739, 
181.211883462743, 180.126606724621, 180.449256565684, 180.742574603015, 
181.065224444078, 181.290101606031, 180.625247388082, 178.66979380588, 
178.102712267041, 178.572021126769, 178.210262214062, 179.862620491023, 
179.051107254409, 180.048388581333, 178.953334575299, 180.859901817947, 
182.443819219531, 186.452499063047, 186.48183086678, 187.889757445966, 
188.847929701246, 187.879980178055, 190.089642725945, 189.08258413111, 
185.787644845098, 186.736039832467, 185.738758505543, 185.249895109992, 
185.983190203318, 182.756691792684, 181.055447176167, 182.228719325489, 
180.840347282125, 179.764847811913, 179.999502241778, 177.956053248376, 
174.739332105652, 177.281421762516, 180.654579191816, 182.013619431447, 
182.805578132239, 180.547029244794, 182.570923702374, 170.935974888267, 
169.909361757611, 169.009853109797, 171.072856639021, 171.855038071902, 
173.839823457838, 172.910983006292, 173.399846401843, 178.063603195397, 
176.137481416927, 175.218418233291, 175.237972769113, 176.254808631859, 
173.888709797393, 176.139728143555, 176.935939872984, 176.926110098546, 
179.766914910951, 179.953680625262, 180.425509798257, 179.108320023647, 
180.071637918511, 181.329849046496, 182.096571452612, 182.037592805988, 
180.995636715625, 178.213810549844, 175.893983782621, 174.291730549326, 
175.923473105933, 176.621387090987, 174.458836714762, 173.082668293528, 
172.748455962656, 173.082668293528, 174.645602429072, 174.439177165887, 
174.104964835016, 172.217648143038, 170.418799420996, 169.858502278064, 
174.822538368945, 172.768115511531, 175.658069196123, 177.152194910606, 
176.955599421859, 179.127979572521, 180.101127241823, 182.194869196986, 
181.929465287177, 183.236825287349, 184.377079122086), .Dim = c(502L, 
1L), .Dimnames = list(NULL, "IBM"), index = structure(c(15342, 
15343, 15344, 15345, 15348, 15349, 15350, 15351, 15352, 15356, 
15357, 15358, 15359, 15362, 15363, 15364, 15365, 15366, 15369, 
15370, 15371, 15372, 15373, 15376, 15377, 15378, 15379, 15380, 
15383, 15384, 15385, 15386, 15387, 15391, 15392, 15393, 15394, 
15397, 15398, 15399, 15400, 15401, 15404, 15405, 15406, 15407, 
15408, 15411, 15412, 15413, 15414, 15415, 15418, 15419, 15420, 
15421, 15422, 15425, 15426, 15427, 15428, 15429, 15432, 15433, 
15434, 15435, 15439, 15440, 15441, 15442, 15443, 15446, 15447, 
15448, 15449, 15450, 15453, 15454, 15455, 15456, 15457, 15460, 
15461, 15462, 15463, 15464, 15467, 15468, 15469, 15470, 15471, 
15474, 15475, 15476, 15477, 15478, 15481, 15482, 15483, 15484, 
15485, 15489, 15490, 15491, 15492, 15495, 15496, 15497, 15498, 
15499, 15502, 15503, 15504, 15505, 15506, 15509, 15510, 15511, 
15512, 15513, 15516, 15517, 15518, 15519, 15520, 15523, 15524, 
15526, 15527, 15530, 15531, 15532, 15533, 15534, 15537, 15538, 
15539, 15540, 15541, 15544, 15545, 15546, 15547, 15548, 15551, 
15552, 15553, 15554, 15555, 15558, 15559, 15560, 15561, 15562, 
15565, 15566, 15567, 15568, 15569, 15572, 15573, 15574, 15575, 
15576, 15579, 15580, 15581, 15582, 15583, 15587, 15588, 15589, 
15590, 15593, 15594, 15595, 15596, 15597, 15600, 15601, 15602, 
15603, 15604, 15607, 15608, 15609, 15610, 15611, 15614, 15615, 
15616, 15617, 15618, 15621, 15622, 15623, 15624, 15625, 15628, 
15629, 15630, 15631, 15632, 15635, 15636, 15637, 15638, 15639, 
15644, 15645, 15646, 15649, 15650, 15651, 15652, 15653, 15656, 
15657, 15658, 15659, 15660, 15663, 15664, 15665, 15667, 15670, 
15671, 15672, 15673, 15674, 15677, 15678, 15679, 15680, 15681, 
15684, 15685, 15686, 15687, 15688, 15691, 15692, 15693, 15694, 
15695, 15698, 15700, 15701, 15702, 15705, 15707, 15708, 15709, 
15712, 15713, 15714, 15715, 15716, 15719, 15720, 15721, 15722, 
15723, 15727, 15728, 15729, 15730, 15733, 15734, 15735, 15736, 
15737, 15740, 15741, 15742, 15743, 15744, 15747, 15748, 15749, 
15750, 15751, 15755, 15756, 15757, 15758, 15761, 15762, 15763, 
15764, 15765, 15768, 15769, 15770, 15771, 15772, 15775, 15776, 
15777, 15778, 15779, 15782, 15783, 15784, 15785, 15786, 15789, 
15790, 15791, 15792, 15796, 15797, 15798, 15799, 15800, 15803, 
15804, 15805, 15806, 15807, 15810, 15811, 15812, 15813, 15814, 
15817, 15818, 15819, 15820, 15821, 15824, 15825, 15826, 15827, 
15828, 15831, 15832, 15833, 15834, 15835, 15838, 15839, 15840, 
15841, 15842, 15845, 15846, 15847, 15848, 15849, 15853, 15854, 
15855, 15856, 15859, 15860, 15861, 15862, 15863, 15866, 15867, 
15868, 15869, 15870, 15873, 15874, 15875, 15876, 15877, 15880, 
15881, 15882, 15883, 15884, 15887, 15888, 15889, 15891, 15894, 
15895, 15896, 15897, 15898, 15901, 15902, 15903, 15904, 15905, 
15908, 15909, 15910, 15911, 15912, 15915, 15916, 15917, 15918, 
15919, 15922, 15923, 15924, 15925, 15926, 15929, 15930, 15931, 
15932, 15933, 15936, 15937, 15938, 15939, 15940, 15943, 15944, 
15945, 15946, 15947, 15951, 15952, 15953, 15954, 15957, 15958, 
15959, 15960, 15961, 15964, 15965, 15966, 15967, 15968, 15971, 
15972, 15973, 15974, 15975, 15978, 15979, 15980, 15981, 15982, 
15985, 15986, 15987, 15988, 15989, 15992, 15993, 15994, 15995, 
15996, 15999, 16000, 16001, 16002, 16003, 16006, 16007, 16008, 
16009, 16010, 16013, 16014, 16015, 16016, 16017, 16020, 16021, 
16022, 16023, 16024, 16027, 16028, 16029, 16030, 16031, 16034, 
16035, 16036, 16038, 16041, 16042, 16043, 16044, 16045, 16048, 
16049, 16050, 16051, 16052, 16055, 16056, 16057, 16058, 16059, 
16062, 16063, 16065, 16066, 16069, 16070), class = "Date"), class = "zoo")

test_cfit <- function (fast_only=FALSE) {
    test("partialAR:::estimate_rho_par_c(numeric())", NA_real_)
    test("partialAR:::estimate_rho_par_c(rep(0,5))", NaN)    
    x1 <- build_par(0.95, rep(0,10), rep(0,10), M0=1)
    test("partialAR:::estimate_rho_par_c(x1)", 0.8497954230236)
    x1na <- x1
    x1na[1] <- NA
    test("partialAR:::estimate_rho_par_c(x1na)", NA_real_)
    
    test("partialAR:::estimate_par_c(numeric())", c(NA_real_, NA_real_, NA_real_))
    test("partialAR:::estimate_par_c(rep(0,5))", c(NaN, NaN, NaN))
    test("partialAR:::estimate_par_c(x1)", c(0.849795423024, 0, 0.00624752527433))
    test("partialAR:::estimate_par_c(x1na)", c(NA_real_, NA_real_, NA_real_))
  
    test("partialAR:::pvmr_par_c(0,0,0)", NA_real_)
    test("partialAR:::pvmr_par_c(-1,1,0)", 1)
    test("partialAR:::pvmr_par_c(1,-1,0)", NA_real_)
    test("partialAR:::pvmr_par_c(1,1,-1)", NA_real_)
    test("partialAR:::pvmr_par_c(0,0,1)", 0)
    test("partialAR:::pvmr_par_c(0,1,0)", 1)
    test("partialAR:::pvmr_par_c(0,1,1)", 2/3)
    test("partialAR:::pvmr_par_c(0.5,1,1)", 0.571428571429)
    test("partialAR:::pvmr_par_c(0.5,1,2)", 0.25)
    test("partialAR:::pvmr_par_c(0.5,0.5,1)", 0.25) 
        
    test("partialAR:::kalman_gain_par_mr(0,0,0)", NA_real_)
    test("partialAR:::kalman_gain_par_mr(0,1,0)", 1)
    test("partialAR:::kalman_gain_par_mr(0,0,1)", 0)
    test("partialAR:::kalman_gain_par_mr(0.5,1,1)", 1/3)
    
    test("partialAR:::loglik_par_c(numeric(),0,0,1,0,0)", NA_real_)
    test("partialAR:::loglik_par_c(0,0,0,1,0,0)", 0.918938533205)
    test("partialAR:::loglik_par_c(c(0,0,0),0,0,1,0,0)", 2.75681559961)
    test("partialAR:::loglik_par_c(1,0,0,1,0,0)", 1.4189385332)
    test("partialAR:::loglik_par_c(0,0,1,0,0,0)", 0.918938533205)
    test("partialAR:::loglik_par_c(c(0,0,0),0,1,0,0,0)", 2.75681559961)
    test("partialAR:::loglik_par_c(c(0,0,0),0.5,1,0,0,0)", 2.75681559961)
    test("partialAR:::loglik_par_c(c(0,1,2),0,0,1,0,1)", 4.25681559961)
    test("partialAR:::loglik_par_c(0.5,0.5,1,0,1,0)", 0.918938533205)
    test("partialAR:::loglik_par_c(data.L, 0.8720, 0.3385, 0.1927, 0, data.L[1])", 238.533361432)
    test("partialAR:::loglik_par_c(data.IBM, 0.9764, 2.0136, 0.4719, 0, data.IBM[1])", 1076.5235347)
    
    test("partialAR:::loglik_par_t_c(numeric(),0,0,1,0,0)", NA_real_)
    test("partialAR:::loglik_par_t_c(0,0,0,1,0,0)", 0.968619589055)
    test("partialAR:::loglik_par_t_c(c(0,0,0),0,0,1,0,0)", 2.90585876716)
    test("partialAR:::loglik_par_t_c(1,0,0,1,0,0)", 1.51558425944)
    test("partialAR:::loglik_par_t_c(0,0,1,0,0,0)", 0.968619589055)
    test("partialAR:::loglik_par_t_c(c(0,0,0),0,1,0,0,0)", 2.90585876716)
    test("partialAR:::loglik_par_t_c(c(0,0,0),0.5,1,0,0,0)", 2.90585876716)
    test("partialAR:::loglik_par_t_c(c(0,1,2),0,0,1,0,1)", 4.54675277831)
    test("partialAR:::loglik_par_t_c(0.5,0.5,1,0,1,0)", 0.968619589055)
    test("partialAR:::loglik_par_t_c(0,0,0,1,0,0,6)", 0.960418255752)  
    test("partialAR:::loglik_par_t_c(data.L, 0.8958, 0.2612, 0.1768, 0, data.L[1])", 229.807616531)
    test("partialAR:::loglik_par_t_c(data.IBM, 0.9829, 1.3072, 0.6901, 0, data.IBM[1])", 1020.88295106)
      
}


test_lr <- function (fast_only=FALSE) {
    test("partialAR:::loglik.par.kfas(numeric(),0,0,1,0,0)", NA_real_)
    test("partialAR:::loglik.par.kfas(0,0,0,1,0,0)", 0.918938533205)
    test("partialAR:::loglik.par.kfas(c(0,0,0),0,0,1,0,0)", 2.75681559961)
    test("partialAR:::loglik.par.kfas(1,0,0,1,0,0)", 1.4189385332)
    test("partialAR:::loglik.par.kfas(0,0,1,0,0,0)", 0.918938533205)
    test("partialAR:::loglik.par.kfas(c(0,0,0),0,1,0,0,0)", 2.75681559961)
    test("partialAR:::loglik.par.kfas(c(0,0,0),0.5,1,0,0,0)", 2.75681559961)
    test("partialAR:::loglik.par.kfas(c(0,1,2),0,0,1,0,1)", 4.25681559961)
    test("partialAR:::loglik.par.kfas(0.5,0.5,1,0,1,0)", 1.0439385332)  # Note difference
    test("partialAR:::loglik.par.kfas(data.L, 0.8720, 0.3385, 0.1927)", 238.53374143)
    test("partialAR:::loglik.par.kfas(data.IBM, 0.9764, 2.0136, 0.4719, 0, data.IBM[1])", 1077.02787353)

    test("partialAR:::loglik.par.ss(numeric(),0,0,1,0,0)", NA_real_)
    test("partialAR:::loglik.par.ss(0,0,0,1,0,0)", 0.918938533205)
    test("partialAR:::loglik.par.ss(c(0,0,0),0,0,1,0,0)", 2.75681559961)
    test("partialAR:::loglik.par.ss(1,0,0,1,0,0)", 1.4189385332)
    test("partialAR:::loglik.par.ss(0,0,1,0,0,0)", 0.918938533205)
    test("partialAR:::loglik.par.ss(c(0,0,0),0,1,0,0,0)", 2.75681559961)
    test("partialAR:::loglik.par.ss(c(0,0,0),0.5,1,0,0,0)", 2.75681559961)
    test("partialAR:::loglik.par.ss(c(0,1,2),0,0,1,0,1)", 4.25681559961)
    test("partialAR:::loglik.par.ss(0.5,0.5,1,0,1,0)", 0.918938533205)
    test("partialAR:::loglik.par.ss(data.L, 0.8720, 0.3385, 0.1927, 0, data.L[1])", 238.533361432)
    test("partialAR:::loglik.par.ss(data.IBM, 0.9764, 2.0136, 0.4719)", 1076.5235347)

    test("partialAR:::loglik.par.ss.t(numeric(),0,0,1,0,0)", NA_real_)
    test("partialAR:::loglik.par.ss.t(0,0,0,1,0,0)", 0.968619589055)
    test("partialAR:::loglik.par.ss.t(c(0,0,0),0,0,1,0,0)", 2.90585876716)
    test("partialAR:::loglik.par.ss.t(1,0,0,1,0,0)", 1.51558425944)
    test("partialAR:::loglik.par.ss.t(0,0,1,0,0,0)", 0.968619589055)
    test("partialAR:::loglik.par.ss.t(c(0,0,0),0,1,0,0,0)", 2.90585876716)
    test("partialAR:::loglik.par.ss.t(c(0,0,0),0.5,1,0,0,0)", 2.90585876716)
    test("partialAR:::loglik.par.ss.t(c(0,1,2),0,0,1,0,1)", 4.54675277831)
    test("partialAR:::loglik.par.ss.t(0.5,0.5,1,0,1,0)", 0.968619589055)
    test("partialAR:::loglik.par.ss.t(0,0,0,1,0,0,6)", 0.960418255752)    
    test("partialAR:::loglik.par.ss.t(data.L, 0.8958, 0.2612, 0.1768, 0, data.L[1])", 229.807616531)
    test("partialAR:::loglik.par.ss.t(data.IBM, 0.9829, 1.3072, 0.6901, 0, data.IBM[1])", 1020.88295106)

    test("partialAR:::loglik.par(data.L, 0.8720, 0.3385, 0.1927)", 238.533361432)
    test("partialAR:::loglik.par(data.L, 0.8720, 0.3385, 0.1927, calc_method=\"css\")", 238.533361432)
    test("partialAR:::loglik.par(data.L, 0.8720, 0.3385, 0.1927, calc_method=\"kfas\")", 238.53374143)
    test("partialAR:::loglik.par(data.L, 0.8720, 0.3385, 0.1927, calc_method=\"ss\")", 238.533361432)
    test("partialAR:::loglik.par(data.L, 0.8958, 0.2612, 0.1768, calc_method=\"sst\")", 229.807616531)
    test("partialAR:::loglik.par(data.L, 0.8958, 0.2612, 0.1768, calc_method=\"csst\")", 229.807616531)
}

test.likelihood_ratio.par <- function (fast_only=FALSE) {
    test("partialAR:::likelihood_ratio.par(data.L)", -4.44824727945)
    test("partialAR:::likelihood_ratio.par(data.L, robust=TRUE)", -2.64805301476)
    test("partialAR:::likelihood_ratio.par(data.L, null_model='rw')", -4.44824727945)
    test("partialAR:::likelihood_ratio.par(data.L, null_model='rw', robust=TRUE)", -2.64805301476)
    test("partialAR:::likelihood_ratio.par(data.L, null_model='ar1')", -4.44824693057)
    test("partialAR:::likelihood_ratio.par(data.L, null_model='ar1', robust=TRUE)", -2.6480522184)

    if (!fast_only) test("partialAR:::likelihood_ratio.par(data.L, opt_method='ss')", -4.44824727945)
    if (!fast_only)  test("partialAR:::likelihood_ratio.par(data.L, robust=TRUE, opt_method='ss')", -2.64805301476)
    if (!fast_only)  test("partialAR:::likelihood_ratio.par(data.L, null_model='rw', opt_method='ss')", -4.44824727945)
    if (!fast_only)  test("partialAR:::likelihood_ratio.par(data.L, null_model='rw', robust=TRUE, opt_method='ss')", -2.64805301476)
    if (!fast_only)  test("partialAR:::likelihood_ratio.par(data.L, null_model='ar1', opt_method='ss')", -4.44824693057)
    if (!fast_only)  test("partialAR:::likelihood_ratio.par(data.L, null_model='ar1', robust=TRUE, opt_method='ss')", -2.6480522184)
    
    test("partialAR:::likelihood_ratio.par(data.L, opt_method='css')", -4.44824727945)
    test("partialAR:::likelihood_ratio.par(data.L, robust=TRUE, opt_method='css')", -2.64805301476)
    test("partialAR:::likelihood_ratio.par(data.L, null_model='rw', opt_method='css')", -4.44824727945)
    test("partialAR:::likelihood_ratio.par(data.L, null_model='rw', robust=TRUE, opt_method='css')", -2.64805301476)
    test("partialAR:::likelihood_ratio.par(data.L, null_model='ar1', opt_method='css')", -4.44824693057)
    test("partialAR:::likelihood_ratio.par(data.L, null_model='ar1', robust=TRUE, opt_method='css')", -2.6480522184)

    if (!fast_only) test("partialAR:::likelihood_ratio.par(data.L, opt_method='kfas')", -4.59676088358)
    if (!fast_only) test("partialAR:::likelihood_ratio.par(data.L, null_model='rw', opt_method='kfas')", -4.59676088358)
    if (!fast_only) test("partialAR:::likelihood_ratio.par(data.L, null_model='ar1', opt_method='kfas')", -4.5967605347)

    SAMPLES <- partialAR:::sample.likelihood_ratio.par(nrep=10, use.multicore=FALSE)
    test("nrow(SAMPLES)", 10)
    test("sum(SAMPLES$seed)", 55)
    test("mean(SAMPLES$rw_lrt)", -4.43576369917)
    test("mean(SAMPLES$mr_lrt)", -3.8960913155)
    test("mean(SAMPLES$kpss_stat)", 3.7269871366)
}

test_lr2 <- function(fast_only=FALSE) {    
    test.likelihood_ratio.par(fast_only)

    test("partialAR:::par.rw.pvalue(-3.5,400) < 0.05", TRUE)
    test("partialAR:::par.rw.pvalue(-1,500) > 0.10", TRUE)
    test("partialAR:::par.mr.pvalue(-1,600) < 0.05", TRUE)
    test("partialAR:::par.mr.pvalue(-0.1, 700) > 0.05", TRUE)
    test("partialAR:::par.rw.pvalue(-3.5,400, robust=TRUE) < 0.05", TRUE)
    test("partialAR:::par.rw.pvalue(-1,500, robust=TRUE) > 0.10", TRUE)
    test("partialAR:::par.mr.pvalue(-1,600, robust=TRUE) < 0.05", TRUE)
    test("partialAR:::par.mr.pvalue(-0.1, 700, robust=TRUE) > 0.05", TRUE)

    test("partialAR:::par.mr.pvalue(-2,400,ar1test='kpss') < 0.05", TRUE)
    test("partialAR:::par.mr.pvalue(-0.5, 500,ar1test='kpss') > 0.05", TRUE)
    test("partialAR:::par.mr.pvalue(-2,600, robust=TRUE,ar1test='kpss') < 0.05", TRUE)
    test("partialAR:::par.mr.pvalue(-0.5, 700, robust=TRUE,ar1test='kpss') > 0.05", TRUE)

    test("partialAR:::par.joint.pvalue(-4,-0.5,500) < 0.05", TRUE)
    test("partialAR:::par.joint.pvalue(-1,-0.25,500) > 0.05", TRUE)
    test("partialAR:::par.joint.pvalue(-5,-0.8,500, robust=TRUE) < 0.05", TRUE)
    test("partialAR:::par.joint.pvalue(-3,-0.1,500, robust=TRUE) > 0.05", TRUE)
    test("partialAR:::par.joint.pvalue(-5,-2,500, ar1test='kpss') < 0.05", TRUE)
    test("partialAR:::par.joint.pvalue(-3,-1,500, ar1test='kpss') > 0.05", TRUE)
    test("partialAR:::par.joint.pvalue(-4,-0.5,50000)", 0.03)
    test("partialAR:::par.joint.pvalue(-4,-0.5,50)", 0.10)
    test("partialAR:::par.joint.pvalue(4,-0.5,50)", 1)
    test("partialAR:::par.joint.pvalue(-4,-0.5,49)", 1)

    test("partialAR:::test.par.nullrw(data.L)$p.value < 0.05", TRUE)
    test("partialAR:::test.par.nullrw(data.IBM)$p.value > 0.05", TRUE)
    test("partialAR:::test.par.nullrw(data.L, robust=TRUE)$p.value < 0.10", TRUE)
    test("partialAR:::test.par.nullrw(data.IBM, robust=TRUE)$p.value > 0.10", TRUE)

    test("partialAR:::test.par.nullmr(data.L)$p.value <= 0.01", TRUE)
    test("partialAR:::test.par.nullmr(data.L, robust=TRUE)$p.value <= 0.01", TRUE)
    test("partialAR:::test.par.nullmr(data.L, ar1test='kpss')$p.value <= 0.01", TRUE)
    test("partialAR:::test.par.nullmr(data.L, robust=TRUE, ar1test='kpss')$p.value <= 0.01", TRUE)
    
    test("partialAR:::test.par.nullmr(data.IBM)$p.value < 0.05", TRUE)
    test("partialAR:::test.par.nullmr(data.IBM, robust=TRUE)$p.value < 0.10", TRUE)
    test("partialAR:::test.par.nullmr(data.IBM, ar1test='kpss')$p.value > 0.10", TRUE)
    test("partialAR:::test.par.nullmr(data.IBM, ar1test='kpss', robust=TRUE)$p.value > 0.10", TRUE)

    test("partialAR:::test.par(data.L, null_hyp='rw')$p.value == partialAR:::test.par.nullrw(data.L)$p.value", TRUE)
    test("partialAR:::test.par(data.IBM, null_hyp='rw')$p.value == partialAR:::test.par.nullrw(data.IBM)$p.value", TRUE)
    test("partialAR:::test.par(data.L, null_hyp='mr')$p.value == partialAR:::test.par.nullmr(data.L)$p.value", TRUE)
    test("partialAR:::test.par(data.IBM, null_hyp='mr')$p.value == partialAR:::test.par.nullmr(data.IBM)$p.value", TRUE)
    
    test("partialAR:::test.par(data.L)$p.value['PAR'] <= 0.01", c(PAR=TRUE))
    test("partialAR:::test.par(data.L, robust=TRUE)$p.value['PAR'] <= 0.10", c(PAR=TRUE))
    test("partialAR:::test.par(data.IBM)$p.value['PAR'] > 0.10", c(PAR=TRUE))
    test("partialAR:::test.par(data.IBM, robust=TRUE)$p.value['PAR'] > 0.10", c(PAR=TRUE))
    test("partialAR:::test.par(data.L, ar1test='kpss')$p.value['PAR'] <= 0.01", c(PAR=TRUE))
    test("partialAR:::test.par(data.L, ar1test='kpss',robust=TRUE)$p.value['PAR'] <= 0.10", c(PAR=TRUE))
    test("partialAR:::test.par(data.IBM, ar1test='kpss')$p.value['PAR'] > 0.10", c(PAR=TRUE))
    
    print(partialAR:::test.par(data.L))
    print(partialAR:::test.par(data.L, robust=TRUE))
    
    test("partialAR:::which.hypothesis.partest(partialAR:::test.par(data.L))", "PAR")
    test("partialAR:::which.hypothesis.partest(partialAR:::test.par(data.L, robust=TRUE))", "RRW")
    test("partialAR:::which.hypothesis.partest(partialAR:::test.par(data.IBM))", "RW")
    
    partialAR:::print.par.lrt(); cat("\n\n")
    partialAR:::print.par.lrt(robust=TRUE); cat("\n\n")
    partialAR:::print.par.lrt(latex=TRUE); cat("\n\n")
    
#    partialAR:::print.par.lrt.mr(); cat("\n\n")
#    partialAR:::print.par.lrt.mr(robust=TRUE); cat("\n\n")
#    partialAR:::print.par.lrt.mr(latex=TRUE); cat("\n\n")

    partialAR:::print.par.lrt.rw(); cat("\n\n")
    partialAR:::print.par.lrt.rw(robust=TRUE); cat("\n\n")
    partialAR:::print.par.lrt.rw(latex=TRUE); cat("\n\n")
    
}

test_fit.par.both <- function (fast_only=FALSE) {
    test("partialAR:::fit.par.both(data.L)$par", 
        structure(c(0.871991364792238, 0.338198849510798, 0.192519577779812, 
        0, 37.8348806008997), .Names = c("rho", "sigma_M", "sigma_R", 
        "M0", "R0")) )
    test("partialAR:::fit.par.both(data.L)$stderr",
        structure(c(0.0493755130952366, 0.0306037545403534, 0.0507506043059735, 
        NA, 0.382843915239426), .Names = c("rho.se", "sigma_M.se", "sigma_R.se", 
        "M0.se", "R0.se")) )
    if (!fast_only) test("partialAR:::fit.par.both(data.L, opt_method='ss')$par", 
        structure(c(0.871991364792238, 0.338198849510798, 0.192519577779812, 
        0, 37.8348806008997), .Names = c("rho", "sigma_M", "sigma_R", 
        "M0", "R0")) )
    if (!fast_only) test("partialAR:::fit.par.both(data.L, opt_method='ss')$stderr",
        structure(c(0.0493755130952366, 0.0306037545403534, 0.0507506043059735, 
        NA, 0.382843915239426), .Names = c("rho.se", "sigma_M.se", "sigma_R.se", 
        "M0.se", "R0.se")) )
    if (!fast_only) test("partialAR:::fit.par.both(data.L, opt_method='kfas')$par", 
        structure(c(0.873239025413773, 0.334187559078876, 0.187013759524079, 
        0, 37.8228485852872), .Names = c("rho", "sigma_M", "sigma_R", 
        "M0", "R0")) )
    if (!fast_only) test("partialAR:::fit.par.both(data.L, opt_method='kfas')$stderr",
        structure(c(0.0480869790579741, 0.0299959210912542, 0.0482633848885082, 
        NA, 0.366440477748884), .Names = c("rho.se", "sigma_M.se", "sigma_R.se", 
        "M0.se", "R0.se")) )
    test("partialAR:::fit.par.both(data.IBM)$par", 
        structure(c(0.976388651908034, 2.01216604959705, 0.467711046901045, 
        0, 177.472892129038), .Names = c("rho", "sigma_M", "sigma_R", 
        "M0", "R0")) )
    test("partialAR:::fit.par.both(data.IBM)$stderr",
        structure(c(0.018222371388718, 0.153130468131214, 0.599803359236283, 
        NA, 2.12284254607983), .Names = c("rho.se", "sigma_M.se", "sigma_R.se", 
        "M0.se", "R0.se")) )
    test("partialAR:::fit.par.both(data.IBM, robust=TRUE)$par",
        structure(c(0.982921831279379, 1.30721045019958, 0.690103593777354, 
        0, 176.743925850553), .Names = c("rho", "sigma_M", "sigma_R", 
        "M0", "R0")) )
    if (!fast_only) test("partialAR:::fit.par.both(data.IBM, robust=TRUE, opt_method='ss')$par",
        structure(c(0.982921831279379, 1.30721045019958, 0.690103593777354, 
        0, 176.743925850553), .Names = c("rho", "sigma_M", "sigma_R", 
        "M0", "R0")) )
    test("partialAR:::fit.par.both(data.IBM, robust=TRUE, nu=3)$par",
        structure(c(0.985936838750558, 1.20382984003629, 0.587584874718192, 
        0, 176.716597228655), .Names = c("rho", "sigma_M", "sigma_R", 
        "M0", "R0")) )
    test("partialAR:::fit.par.both(data.IBM, rho.max=0.95)$par",
        structure(c(0.95, 1.8101310703133, 0.998701976498605, 0, 176.958377474755
        ), .Names = c("rho", "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par.both(data.IBM, lambda=2)$pvmr", c(pvmr=1))
    test("partialAR:::fit.par.both(data.IBM, lambda=-2)$pvmr", c(pvmr=0.0442039289027))
}

test_fit.par.mr <- function (fast_only=FALSE) {
    test("partialAR:::fit.par.mr(data.L)$par", 
        structure(c(1, 0.392621113046972, 0, 0, 37.8517816705337), .Names = c("rho", 
        "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par.mr(data.L)$stderr",
        structure(c(1.55086108092093e-05, 0.0123907243901383, NA, NA, 
        0.392621124942204), .Names = c("rho.se", "sigma_M.se", "sigma_R.se", 
        "M0.se", "R0.se")) )
    if (!fast_only) test("partialAR:::fit.par.mr(data.L, opt_method='ss')$par", 
        structure(c(1, 0.392621113046972, 0, 0, 37.8517816705337), .Names = c("rho", 
        "sigma_M", "sigma_R", "M0", "R0")) )
    if (!fast_only) test("partialAR:::fit.par.mr(data.L, opt_method='ss')$stderr",
        structure(c(1.55086108092093e-05, 0.0123907243901383, NA, NA, 
        0.392621124942204), .Names = c("rho.se", "sigma_M.se", "sigma_R.se", 
        "M0.se", "R0.se")) )
    if (!fast_only) test("partialAR:::fit.par.mr(data.L, opt_method='kfas')$par", 
        structure(c(1, 0.392621113047498, 0, 0, 37.8517816705312), .Names = c("rho", 
        "sigma_M", "sigma_R", "M0", "R0")) )
    if (!fast_only) test("partialAR:::fit.par.mr(data.L, opt_method='kfas')$stderr",
        structure(c(1.55086108092093e-05, 0.0123907243901654, NA, NA, 
        0.392621124727183), .Names = c("rho.se", "sigma_M.se", "sigma_R.se", 
        "M0.se", "R0.se")) )
    test("partialAR:::fit.par.mr(data.IBM)$par", 
        structure(c(0.989394562548544, 2.06766254187052, 0, 0, 177.378135957708
        ), .Names = c("rho", "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par.mr(data.IBM)$stderr",
        structure(c(0.00711953959492437, 0.0652545415824236, NA, NA, 
        2.18393834163026), .Names = c("rho.se", "sigma_M.se", "sigma_R.se", 
        "M0.se", "R0.se")) )
    test("partialAR:::fit.par.mr(data.IBM, robust=TRUE)$par",
        structure(c(0.996850903105148, 1.47881632988678, 0, 0, 176.742922370692
        ), .Names = c("rho", "sigma_M", "sigma_R", "M0", "R0")) )
    if (!fast_only) test("partialAR:::fit.par.mr(data.IBM, robust=TRUE, opt_method='ss')$par",
        structure(c(0.996850903105148, 1.47881632988678, 0, 0, 176.742922370692
        ), .Names = c("rho", "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par.mr(data.IBM, robust=TRUE, nu=3)$par",
        structure(c(0.996784426974733, 1.33994364448777, 0, 0, 176.717640850721
        ), .Names = c("rho", "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par.mr(data.IBM, rho.max=0.95)$par",
        structure(c(0.95, 2.10195614607977, 0, 0, 183.429724544732), .Names = c("rho", 
        "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par.mr(data.IBM)$pvmr", c(pvmr=1))

}

test_fit.par.rw <- function (fast_only=FALSE) {
    test("partialAR:::fit.par.rw(data.L)$par", 
        structure(c(0, 0, 0.392609091324016, 0, 37.8517816659277), .Names = c("rho", 
        "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par.rw(data.L)$stderr",
        structure(c(NA, NA, 0.0175230013091655, NA, 0), .Names = c("rho.se", 
        "sigma_M.se", "sigma_R.se", "M0.se", "R0.se")) )
    if (!fast_only) test("partialAR:::fit.par.rw(data.L, opt_method='ss')$par", 
        structure(c(0, 0, 0.392609091324016, 0, 37.8517816659277), .Names = c("rho", 
        "sigma_M", "sigma_R", "M0", "R0")) )
    if (!fast_only) test("partialAR:::fit.par.rw(data.L, opt_method='kfas')$par", 
        structure(c(0, 0, 0.392609091324016, 0, 37.8517816659277), .Names = c("rho", 
        "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par.rw(data.IBM)$par", 
        structure(c(0, 0, 2.07281796275108, 0, 176.668606104443), .Names = c("rho", 
        "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par.rw(data.IBM)$stderr",
        structure(c(NA, NA, 0.0925143932669985, NA, 0), .Names = c("rho.se", 
        "sigma_M.se", "sigma_R.se", "M0.se", "R0.se")) )
    test("partialAR:::fit.par.rw(data.IBM, robust=TRUE)$par",
        structure(c(0, 0, 1.47924935869178, 0, 176.668606104443), .Names = c("rho", 
        "sigma_M", "sigma_R", "M0", "R0")) )
    if (!fast_only) test("partialAR:::fit.par.rw(data.IBM, robust=TRUE, opt_method='ss')$par",
        structure(c(0, 0, 1.47924935869178, 0, 176.668606104443), .Names = c("rho", 
        "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par.rw(data.IBM, robust=TRUE, nu=3)$par",
        structure(c(0, 0, 1.34077692991459, 0, 176.668606104443), .Names = c("rho", 
        "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par.rw(data.IBM)$pvmr", c(pvmr=0))
}

test_fit.par <- function (fast_only=FALSE) {
    test("partialAR:::fit.par(data.L)$par", 
        structure(c(0.871991364792238, 0.338198849510798, 0.192519577779812, 
        0, 37.8348806008997), .Names = c("rho", "sigma_M", "sigma_R", 
        "M0", "R0")) )
    test("partialAR:::fit.par(data.L)$stderr",
        structure(c(0.0493755130952366, 0.0306037545403534, 0.0507506043059735, 
        NA, 0.382843915239426), .Names = c("rho.se", "sigma_M.se", "sigma_R.se", 
        "M0.se", "R0.se")) )
    if (!fast_only) test("partialAR:::fit.par(data.L, opt_method='kfas')$par", 
        structure(c(0.873239025413773, 0.334187559078876, 0.187013759524079, 
        0, 37.8228485852872), .Names = c("rho", "sigma_M", "sigma_R", 
        "M0", "R0")) )
    test("partialAR:::fit.par(data.IBM)$par", 
        structure(c(0.976388651908034, 2.01216604959705, 0.467711046901045, 
        0, 177.472892129038), .Names = c("rho", "sigma_M", "sigma_R", 
        "M0", "R0")) )
    test("partialAR:::fit.par(data.IBM)$stderr",
        structure(c(0.018222371388718, 0.153130468131214, 0.599803359236283, 
        NA, 2.12284254607983), .Names = c("rho.se", "sigma_M.se", "sigma_R.se", 
        "M0.se", "R0.se")) )
    test("partialAR:::fit.par(data.IBM, robust=TRUE)$par",
        structure(c(0.982921831279379, 1.30721045019958, 0.690103593777354, 
        0, 176.743925850553), .Names = c("rho", "sigma_M", "sigma_R", 
        "M0", "R0")) )
    test("partialAR:::fit.par(data.IBM, robust=TRUE, nu=3)$par",
        structure(c(0.985936838750558, 1.20382984003629, 0.587584874718192, 
        0, 176.716597228655), .Names = c("rho", "sigma_M", "sigma_R", 
        "M0", "R0")) )
    test("partialAR:::fit.par(data.IBM, rho.max=0.95)$par",
        structure(c(0.95, 1.8101310703133, 0.998701976498605, 0, 176.958377474755
        ), .Names = c("rho", "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par(data.IBM, lambda=2)$pvmr", c(pvmr=1))
    test("partialAR:::fit.par(data.IBM, lambda=-2)$pvmr", c(pvmr=0.0442039289027))
    test("partialAR:::fit.par(data.L, model='ar1')$par", 
        structure(c(1, 0.392621113046972, 0, 0, 37.8517816705337), .Names = c("rho", 
        "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par(data.L, model='ar1')$stderr",
        structure(c(1.55086108092093e-05, 0.0123907243901383, NA, NA, 
        0.392621124942204), .Names = c("rho.se", "sigma_M.se", "sigma_R.se", 
        "M0.se", "R0.se")) )
    test("partialAR:::fit.par(data.L, model='rw')$par", 
        structure(c(0, 0, 0.392609091324016, 0, 37.8517816659277), .Names = c("rho", 
        "sigma_M", "sigma_R", "M0", "R0")) )
    test("partialAR:::fit.par(data.L, model='rw')$stderr",
        structure(c(NA, NA, 0.0175230013091655, NA, 0), .Names = c("rho.se", 
        "sigma_M.se", "sigma_R.se", "M0.se", "R0.se")) )
}

test_fit <- function (fast_only=FALSE) {
    test("partialAR:::par.rho.cutoff(25)", NA_real_)
    test("partialAR:::par.rho.cutoff(50)", 0.724)
    test("partialAR:::par.rho.cutoff(50,0.01)", 0.594)
    test("partialAR:::par.rho.cutoff(50,.00001)", 0.438)
    
    test("partialAR:::estimate.rho.par(numeric())", NA_real_)
    test("partialAR:::estimate.rho.par(rep(0,5))", NaN)    
    x1 <- build_par(0.95, rep(0,10), rep(0,10), M0=1)
    test("partialAR:::estimate.rho.par(x1)", 0.8497954230236)
    x1na <- x1
    x1na[1] <- NA
    test("partialAR:::estimate.rho.par(x1na)", NA_real_)
    
    test("partialAR:::estimate.par(numeric())", c(rho=NA_real_, sigma_M=NA_real_, sigma_R=NA_real_))
    test("partialAR:::estimate.par(rep(0,5))", c(rho=NaN, sigma_M=NaN, sigma_R=NaN))
    test("partialAR:::estimate.par(x1)", c(rho=0.849795423024, sigma_M=0, sigma_R=0.00624752527433))
    test("partialAR:::estimate.par(x1na)", c(rho=NA_real_, sigma_M=NA_real_, sigma_R=NA_real_))
  
    test("partialAR:::pvmr.par(0,0,0)", c(pvmr=NA_real_))
    test("partialAR:::pvmr.par(-1,1,0)", c(pvmr=1))
    test("partialAR:::pvmr.par(1,-1,0)", c(pvmr=NA_real_))
    test("partialAR:::pvmr.par(1,1,-1)", c(pvmr=NA_real_))
    test("partialAR:::pvmr.par(0,0,1)", c(pvmr=0))
    test("partialAR:::pvmr.par(0,1,0)", c(pvmr=1))
    test("partialAR:::pvmr.par(0,1,1)", c(pvmr=2/3))
    test("partialAR:::pvmr.par(0.5,1,1)", c(pvmr=0.571428571429))
    test("partialAR:::pvmr.par(0.5,1,2)", c(pvmr=0.25))
    test("partialAR:::pvmr.par(0.5,0.5,1)", c(pvmr=0.25)) 
        
    test("partialAR:::kalman.gain.par(0,0,0)", c(NA_real_, NA_real_))
    test("partialAR:::kalman.gain.par(0,1,0)", c(1,0))
    test("partialAR:::kalman.gain.par(0,0,1)", c(0,1))
    test("partialAR:::kalman.gain.par(0.5,1,1)", c(1/3,2/3))

    test("partialAR:::kalman.gain.from.pvmr(0,0)", c(0,1))
    test("partialAR:::kalman.gain.from.pvmr(1,0)", c(0,1))
    test("partialAR:::kalman.gain.from.pvmr(0,1)", c(1,0))
    test("partialAR:::kalman.gain.from.pvmr(0,0)", c(0,1))
    test("partialAR:::kalman.gain.from.pvmr(0,0)", c(0,1))
    test("partialAR:::kalman.gain.from.pvmr(0.8,0.8)", c(0.545454545455, 0.454545454545))
    
    test_fit.par.both (fast_only)
    test_fit.par.mr(fast_only)
    test_fit.par.rw(fast_only)
    test_fit.par(fast_only)

    test("partialAR:::statehistory.par(partialAR:::fit.par(data.L))[1,]",
        structure(list(X = 37.8517816659277, M = 0.00867470536387833, 
            R = 37.8431069605638, eps_M = 0.00867470536387833, eps_R = 0.00822635966417289), 
            .Names = c("X", 
        "M", "R", "eps_M", "eps_R"), row.names = 1L, class = "data.frame") )
    test("partialAR:::statehistory.par(partialAR:::fit.par(data.L))[length(data.L),]",
        structure(list(X = 48.0305776082708, M = 0.379272544771068, R = 47.6513050634997, 
            eps_M = 0.159638785630931, eps_R = 0.151387973638877), .Names = c("X", 
        "M", "R", "eps_M", "eps_R"), row.names = 502L, class = "data.frame") )

    print(partialAR:::fit.par(data.L))
    print(partialAR:::fit.par(data.IBM))
    
    test("as.data.frame(partialAR:::fit.par(data.L))",
        structure(list(robust = FALSE, nu = 5, 
            opt_method = structure(1L, .Label = "css", class = "factor"), 
            n = 502L, rho = 0.871991364792238, sigma_M = 0.338198849510798, 
            sigma_R = 0.192519577779812, M0 = 0, R0 = 37.8348806008997, 
            rho.se = 0.0493755130952366, sigma_M.se = 0.0306037545403534, 
            sigma_R.se = 0.0507506043059735, M0.se = NA_real_, R0.se = 0.382843915239426, 
            lambda = 0, pvmr = 0.767280179062111, negloglik = 238.531977143138), .Names = c("robust", 
        "nu", "opt_method", "n", "rho", "sigma_M", "sigma_R", "M0", "R0", 
        "rho.se", "sigma_M.se", "sigma_R.se", "M0.se", "R0.se", "lambda", 
        "pvmr", "negloglik"), row.names = c(NA, -1L), class = "data.frame") )
}

test_par <- function (fast_only=FALSE) {
    # Comprehensive unit testing for PAR package

    options(warn=1)
    
    test_cfit(fast_only)
    test_lr(fast_only)
    test_fit(fast_only)
    test_lr2(fast_only)
    
    if (all.tests.pass) {
        cat("SUCCESS! All tests passed.\n")
    } else {
        stop("ERRORS! ", all.tests.error.count," tests failed\n")
    }
}

test_par(TRUE)
