\name{evaluate.pm}
\alias{evaluate.pm}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{
Evaluate the periodogram maximum of a time series
}
\description{
Calculate the Lomb-Scargle periodogram of a time series and estimate the statistical significance of the  periodogram maximum based on the null hypothesis of an Ornstein-Uhlenbeck state space (OUSS) process.
}
\usage{
evaluate.pm(times, signal, minPeakFreq = 0, minFitFreq = 0, 
            iterations = 100, accuracy = 0.005, startRadius = 5, 
            verbose = FALSE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{times}{
     Numerical vector. Time points of the time series.
}
  \item{signal}{
     Numerical vector of same size as \code{times}. Values of the time series.
}
  \item{minPeakFreq}{
    Non-negative number. Minimum considered frequency when determining periodogram peak. Use this if you want to ignore low-frequency components in the spectrum when searching for the periodogram maximum.
}
  \item{minFitFreq}{
    Non-negative number. Minimum considered frequency when fitting the OUSS model to the periodogram. Use this to ignore low-frequency components in the spectrum when estimating the OUSS parameters, e.g. if you suspect your time series to be trended. \code{minFitFreq} might differ from \code{minPeakFreq}.
}
  \item{iterations}{
    Number of iterations for the maximum-likelihood fitter. Increasing this will result in greater estimation accuracy but also reduced performance.
}
  \item{startRadius}{
    Single integer. The number of initial guesses for each OUSS parameter during optimization of the likelihood function, per direction (up and down). Increasing this will improve estimation accuracy. However, execution time scales with \code{startRadius^2}.
}
  \item{accuracy}{
    An upper bound for the standard deviation of the P-value estimator using repeated Bernoulli trials. The number of trials scales as \code{1/accuracy^2}. If you are using \code{0.05} as a nominal significance threshold, then \code{accuracy=0.005} is probably good enough.
}
  \item{verbose}{
    Single logical. If \code{TRUE}, the function prints out occasional progress reports.
}
}
\details{
The OUSS model describes the measurement of an Ornstein-Uhlenbeck (OU) stochastic process at discrete times with additional uncorrelated Gaussian measurement errors of fixed variance. The OU process itself is a continuous-time random walk with linear stabilizing forces, described by the stochastic differential equation
\deqn{
  dX = \lambda(\mu-X) dt + s dW,
}
where \eqn{W} is the standard Wiener process. The OUSS process is a parsimonious model for describing stochastically fluctuating variables subject to linear stabilizing forces and uncorrelated measurement errors.

Due to temporal correlations, the OUSS power spectrum increases gradually towards lower frequencies, as opposed to the white noise spectrum, which is flat. Using white noise as a null hypothesis for the evaluation of cyclicity in time series, particularly for systems with long correlation times, may result in increased false cycle detection rates because of the increased low-frequency power. The OUSS model is an attempt to account for these correlations.

The OUSS model parameters are estimated using maximum-likelihood fitting to the periodogram. The likelihood function, and therefore the OUSS parameter estimates, are only approximations that become exact for long regular time series. The statistical significance of the periodogram peak (power \eqn{S} at frequency \eqn{F}) under the null-hypothesis of an OUSS process is defined as the probability that the same OUSS process would generate a periodogram whose maximum (power \eqn{s} at frequency \eqn{f}) satisfies 
\deqn{
    s^2/e\ge S^2/E,
}
where \eqn{e} and \eqn{E} are the expected periodogram powers at the frequencies \eqn{f} and \eqn{F}, respectively.
The P-value is estimated via repeated Bernoulli trials in which random OUSS periodograms are emulated by exponentially distributed numbers.

If you want to evaluate secondary peaks (i.e. non-global periodogram maxima), you will need to either (a) adjust the parameters \code{minPeakFreq} and \code{minFitFreq} to omit low-frequency modes or (b) use \code{\link{significanceOfLocalPeak}} after using \code{evaluate.pm}.
}
\value{
A list with the following entries:
\item{error}{Will be \code{TRUE} if an error occured, \code{FALSE} otherwise.}
\item{errorMessage}{A short error message if \code{error==TRUE}.}
If \code{error==FALSE}, the returned list also includes:
\item{frequencies}{Available periodogram frequencies as a numerical vector.}
\item{periodogram}{Periodogram powers corresponding to the returned frequencies, as a numerical vector.}
\item{fittedPS}{Maximum-likelihood fitted OUSS power spectrum corresponding to \code{frequencies}, as a numerical vector.}
\item{power_o}{Estimated power spectrum at zero-frequency generated by the underlying OU process.}
\item{lambda}{Estimated resilience of the OU process (in inverse time units).}
\item{power_e}{Estimated power spectrum at large frequencies due to the random measurement errors.}
\item{sigma}{Estimated stationary standard deviation of the underlying OU process.}
\item{epsilon}{Estimated standard deviation of measurement errors.}
\item{time_step}{The average time step of the time series, as used to fit the OUSS power spectrum.}
\item{peakMode}{An integer indicating the position of the periodogram maximum in the vector \code{frequencies}.}
\item{minPeakMode}{The minimum periodogram mode considered for determining the periodogram maximum. This will be \code{1} if \code{minPeakFreq==0}.}
\item{minFitMode}{The minimum periodogram mode considered for estimating the white noise power. This will be \code{1} if \code{minFitFreq==0}.}
\item{MLL}{Log-likelihood value at calculated maximum.}
\item{P}{Statistical significance of the periodogram peak against the null hypothesis of the estimated OUSS process. This is the probability that the estimated OUSS process would generate a periodogram with global maximum at least as ``extreme'' as the observed peak (among the considered frequencies). See details above.}
\item{Plocal}{Statistical significance of the relative power of the periodogram peak. Mainly used for comparison to the statistical significance of secondary peaks. See \code{\link{significanceOfLocalPeak}}.}
}
\references{
Louca, S., Doebeli, M. - Detecting cyclicity in ecological time series (in review, as of June 2014)
}
\author{
Stilianos Louca
}

%% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{
\code{\link{evaluate.pm.wn}}, \code{\link{significanceOfLocalPeak}}, \code{\link{ps_ouss}}
}
\examples{
# In this example we generate a cyclic time series
# and analyse its periodogram using evaluate.pm

# Parameters
lambda         = 1;    # inverse correlation time of OU process
cyclePeriod    = 1;    # Cycle period
cycleAmplitude = 0.6;
times          = seq(0,20,0.25);

# generate cyclic time series by adding a periodic signal to an OUSS process
signal = cycleAmplitude * cos(2*pi*times/cyclePeriod) +
         generate_ouss(times, mu=0, sigma=1, lambda=lambda, epsilon=0.5);

# Find periodogram peak and estimate statistical significance
# Ignore frequencies lower than a pre-defined threshold
# to avoid masking by low-frequency maximum
report = evaluate.pm(times=times, signal=signal, 
                     minPeakFreq=lambda/3, 
                     minFitFreq=lambda/3,
                     startRadius=2);

# plot overview of periodogram peak analysis
plotReport(sprintf("Cyclic at frequency \%.2g",1/cyclePeriod), 
           times=times, signal=signal, report=report);     
      
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{ statistics }
