#' fit_phenology fits parameters to timeseries.
#' @title Fits the phenology parameters to timeseries.
#' @author Marc Girondot
#' @return Return a list of with data and result
#' @param data A dataset generated by add_format
#' @param parametersfixed Set of fixed parameters
#' @param parametersfit Set of parameters to be fitted
#' @param trace If 1, display the progression of fit; 0 is silent
#' @param method_incertitude 2 [default] is the correct one from a statistical point of view; 
#'                           0 is an aproximate method more rapid; 
#'                           1 is an alternative more rapid but biased.
#' @param zero_counts example c(TRUE, TRUE, FALSE) indicates whether the zeros have
#'                    been recorder for each of these timeseries. Defaut is TRUE for all.
#' @param hessian If FALSE does not estimate se of parameters
#' @param help If TRUE, an help is displayed
#' @description Function of the package phenology to fit parameters to timeseries.
#' To fit data, the syntaxe is :
#' Result<-fit_phenology(data=dataset, parametersfit=par, parametersfixed=pfixed, trace=1,
#' +      method_incertitude=2, zero_counts=TRUE, hessian=TRUE)
#' or if no parameter is fixed :
#' Result<-fit_phenology(data=dataset, parametersfit=par)
#' or
#' fit_phenology(help=TRUE) to have this help !
#' Add trace=1 [default] to have information on the fit progression.
#' or trace=0 to hide information on the fit progression.
#' method_incertitude=2 [default] is the correct one from a statistical point of view.
#' method_incertitude=0 is an aproximate method more rapid.
#' method_incertitude=1 is an alternative more rapid but biased.
#' zero_counts=c(TRUE, TRUE, FALSE) indicates whether the zeros have
#' been recorder for each of these timeseries. Defaut is TRUE for all.
#' hessian=FALSE does not estimate se of parameters.
#' @export


fit_phenology <-
function(data=NULL, parametersfit=NULL, parametersfixed=NA, trace=1, method_incertitude=2, zero_counts=TRUE, hessian=TRUE, help=FALSE) {
if ((help)||(is.null(data))||(is.null(parametersfit))) {
	cat("To fit data, the syntaxe is :\n")
	cat("Result<-fit_phenology(data=dataset, parametersfit=par, parametersfixed=pfixed, trace=1,\n")
	cat("+      method_incertitude=2, zero_counts=TRUE, hessian=TRUE)\n")
	cat("or if no parameter is fixed :\n")
	cat("Result<-fit_phenology(data=dataset, parametersfit=par)\n")
	cat("or\n")	
	cat("fit_phenology(help=TRUE) to have this help !\n")
	cat("Add trace=1 [default] to have information on the fit progression.\n")
	cat("or trace=0 to hide information on the fit progression.\n")
	cat("method_incertitude=2 [default] is the correct one from a statistical point of view.\n")
	cat("method_incertitude=0 is an aproximate method more rapid.\n")
	cat("method_incertitude=1 is an alternative more rapid but biased.\n")
	cat("zero_counts=c(TRUE, TRUE, FALSE) indicates whether the zeros have\n")
	cat("been recorder for each of these timeseries. Defaut is TRUE for all.\n")
	cat("hessian=FALSE does not estimate se of parameters.\n")
} else {

if (is.null(parametersfixed)) {parametersfixed<-NA}

	.phenology.env <<- new.env()
	.phenology.env$data<<-data
	.phenology.env$fixed<<-parametersfixed
	.phenology.env$incertitude<<-method_incertitude
	if (length(zero_counts)==1) {zero_counts<-rep(zero_counts, length(data))}
	if (length(zero_counts)!=length(data)) {
		print("zero_counts parameter must be TRUE (the zeros are used for all timeseries) or FALSE (the zeros are not used for all timeseries) or possess the same number of logical values than the number of series analyzed.")
		return()
	}
	.phenology.env$zerocounts<<-zero_counts
	
	
	repeat {
		resul<-optim(parametersfit, .Lnegbin, NULL, method="BFGS",control=list(trace=trace, REPORT=1, maxit=500),hessian=FALSE)
		if (resul$convergence==0) break
		parametersfit<-resul$par
		print("Convergence is not achieved. Optimization continues !")
	}
	
	resfit<-resul$par
	resfit[substr(names(resfit), 1, 4)=="Peak"]<-abs(resfit[substr(names(resfit), 1, 4)=="Peak"])
	resfit["Theta"]<-abs(resfit["Theta"])
	resfit["PMinE"]<-abs(resfit["PMinE"])
	resfit["PMinB"]<-abs(resfit["PMinB"])
	resfit["Flat"]<-abs(resfit["Flat"])
	resfit[substr(names(resfit), 1, 6)=="Length"]<-abs(resfit[substr(names(resfit), 1, 6)=="Length"])
	resfit[substr(names(resfit), 1, 3)=="Min"]<-abs(resfit[substr(names(resfit), 1, 3)=="Min"])
	resfit[substr(names(resfit), 1, 3)=="Max"]<-abs(resfit[substr(names(resfit), 1, 3)=="Max"])
	resfit<-resfit[!is.na(resfit)]
	cat("Fit done!\n")
	cat(paste("-Ln L=", format(resul$value, digits=max(3, trunc(log10(resul$value))+4)), "\n", sep=""))
	if (hessian) {
	cat("Estimation of the standard error of parameters. Be patient please.\n")
	
	resul<-optim(resfit, .Lnegbin, NULL, method="BFGS",control=list(trace=0, REPORT=1, maxit=10),hessian=TRUE)

	resfit<-resul$par

	mathessian<-resul$hessian
	inversemathessian=try(solve(mathessian), silent=TRUE)
	if (substr(inversemathessian[1], 1, 5)=="Error") {
		print("Error in the fit; probably one or more parameters are not estimable.")
		print("Standard errors cannot be estimated.")
		res_se<-rep(NA, length(resfit))
	
	} else {
		res_se_diag=diag(inversemathessian)
		
		res_se<-NULL
		res_se[res_se_diag>=0]<-sqrt(res_se_diag[res_se_diag>=0])

	}
	} else {
	
		print("Standard errors are not estimated.")
		res_se<-rep(NA, length(resfit))
	
	}
	names(res_se)<-names(resfit)
	
	resul$se<-res_se
	
	resul$parametersfixed<-parametersfixed
	
	resul$method_incertitude<-method_incertitude
	
	resul$zero_counts<-zero_counts
	
	resul$data<-data
	
	.phenology.env$result<<-resul
	


	
for(kl in 1:length(res_se)) {
if (is.na(res_se[kl])) {
	cat(paste(names(par[kl]), "=", format(resfit[kl], digits=max(3, trunc(log10(abs(resfit[kl])))+4)), "  SE= NaN\n", sep=""))
} else {
	cat(paste(names(par[kl]), "=", format(resfit[kl], digits=max(3, trunc(log10(abs(resfit[kl])))+4)), "  SE=", format(res_se[kl], , digits=max(3, trunc(log10(res_se[kl]))+4)), "\n", sep=""))
}
}
	cat(paste("-Ln L=", format(resul$value, digits=max(3, trunc(log10(resul$value))+4)), "\n", sep=""))
	cat(paste("Parameters=", format(length(resul$par), digits=max(3, trunc(log10(length(resul$par)))+4)), "\n", sep=""))
	cat(paste("AIC=", format(2*resul$value+2*length(resul$par), digits=max(3, trunc(log10(2*resul$value+2*length(resul$par)))+4)), "\n", sep=""))

	
	growlnotify('Fit is done!')

	return(resul)
}
	
}
