% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/solvers.R
\name{solvers}
\alias{solvers}
\title{Problem solvers}
\description{
Specify the software and configuration used to solve a conservation planning
\code{\link[=problem]{problem()}}. By default, the best available
software currently installed on the system will be used.
For information on the performance of different solvers,
please see Schuster \emph{et al.} (2020) for benchmarks comparing the
run time and solution quality of some of these solvers when applied to
different sized datasets.
}
\details{
The following solvers can be used to find solutions for a
conservation planning \code{\link[=problem]{problem()}}:

\describe{

\item{\code{\link[=add_default_solver]{add_default_solver()}}}{This solver uses the best software
currently installed on the system.}

\item{\code{\link[=add_gurobi_solver]{add_gurobi_solver()}}}{\href{https://www.gurobi.com/}{\emph{Gurobi}}
is a state-of-the-art commercial optimization software with an R package
interface. We recommend using this solver if at all possible.
It is by far the fastest of the solvers available for
generating prioritizations, however, it is not freely available. That
said, licenses are available to academics at no cost. The
\pkg{gurobi} package is distributed with the \emph{Gurobi} software
suite. This solver uses the \pkg{gurobi} package to solve problems.}

\item{\code{\link[=add_cplex_solver]{add_cplex_solver()}}}{\href{https://www.ibm.com/analytics/cplex-optimizer}{\emph{IBM CPLEX}}
is a commercial optimization software. It is faster than the open
source solvers available for generating prioritizations, however, it
is not freely available.
Similar to the \href{https://www.gurobi.com/}{\emph{Gurobi}}
software, licenses are available to academics at no cost.
This solver uses the \pkg{cplexAPI} package to solve problems using
\emph{IBM CPLEX}.}

\item{\code{\link[=add_cbc_solver]{add_cbc_solver()}}}{\href{https://github.com/coin-or/Cbc}{\emph{CBC}} is an
open-source mixed integer programming solver that is part of the
Computational Infrastructure for Operations Research (COIN-OR) project.
Preliminary benchmarks indicate that it is the fastest open source
solver currently supported.
We recommend using this solver if both \emph{Gurobi} and \emph{IBM CPLEX} are
unavailable.
It requires the \pkg{rcbc} package, which is currently only available on
\href{https://github.com/dirkschumacher/rcbc}{GitHub}.
}

\item{\code{\link[=add_lpsymphony_solver]{add_lpsymphony_solver()}}}{
\href{https://github.com/coin-or/SYMPHONY}{\emph{SYMPHONY}} is an
open-source mixed integer programming solver that is also part of the
COIN-OR project. Although both \emph{SYMPHONY} and \emph{CBC} are part of
the COIN-OR project, they are different software.
The \pkg{lpsymphony} package provides an interface to the \emph{SYMPHONY}
software, and is distributed through
\href{https://doi.org/doi:10.18129/B9.bioc.lpsymphony}{Bioconductor}.
We recommend using this solver if the CBC solver cannot be installed.
This solver can use parallel processing to solve problems, so it is
faster than \pkg{Rsymphony} package interface (see below).
}

\item{\code{\link[=add_rsymphony_solver]{add_rsymphony_solver()}}}{
This solver provides an alternative interface to the
\href{https://github.com/coin-or/SYMPHONY}{\emph{SYMPHONY}} solver using
the \pkg{Rsymphony} package.
Unlike other solvers, the \pkg{Rsymphony} package can be installed
directly from the Comprehensive R Archive Network (CRAN).
It is also the slowest of the available solvers.}

}
}
\examples{
\dontrun{
# load data
data(sim_pu_raster, sim_features)

# create basic problem
p <- problem(sim_pu_raster, sim_features) \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(0.1) \%>\%
  add_proportion_decisions()

# create vector to store plot titles
titles <- c()

# create empty stack to store solutions
s <- stack()

# if gurobi is installed: create problem with added gurobi solver
if (require("gurobi")) {
  titles <- c(titles, "gurobi")
  p3 <- p \%>\% add_gurobi_solver(verbose = FALSE)
  s <- addLayer(s, solve(p3))
}

# if cplexAPI is installed: create problem with added CPLEX solver
if (require("cplexAPI")) {
  titles <- c(titles, "CPLEX")
  p4 <- p \%>\% add_cplex_solver(verbose = FALSE)
  s <- addLayer(s, solve(p4))
}

# if rcbc is installed: create problem with added cbc solver
if (require("rcbc")) {
  titles <- c(titles, "CBC")
  p6 <- p \%>\% add_cbc_solver(verbose = FALSE)
  s <- addLayer(s, solve(p6))
}

# create problem with added rsymphony solver
if (require("Rsymphony")) {
  titles <- c(titles, "Rsymphony")
  p2 <- p \%>\% add_rsymphony_solver(verbose = FALSE)
  s <- addLayer(s, solve(p2))
}

# if lpsymphony is installed: create problem with added lpsymphony solver
if (require("lpsymphony")) {
  titles <- c(titles, "lpsymphony")
  p5 <- p \%>\% add_lpsymphony_solver(verbose = FALSE)
  s <- addLayer(s, solve(p5))
}

# plot solutions
plot(s, main = titles, axes = FALSE, box = FALSE)
}

}
\references{
Schuster R, Hanson JO, Strimas-Mackey M, and Bennett JR (2020). Exact
integer linear programming solvers outperform simulated annealing for
solving conservation planning problems. \emph{PeerJ}, 8: e9258.
}
\seealso{
Other overviews: 
\code{\link{constraints}},
\code{\link{decisions}},
\code{\link{importance}},
\code{\link{objectives}},
\code{\link{penalties}},
\code{\link{portfolios}},
\code{\link{summaries}},
\code{\link{targets}}
}
\concept{overviews}
