% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crosstabs.R
\name{crosstabs}
\alias{crosstabs}
\title{Display Cross Table of Two Variables}
\usage{
crosstabs(
  data_frame,
  rows,
  columns,
  statistics = c("sum"),
  formats = c(),
  by = c(),
  weight = NULL,
  titles = c(),
  footnotes = c(),
  style = excel_output_style(),
  output = "console",
  na.rm = FALSE,
  print = TRUE,
  monitor = FALSE
)
}
\arguments{
\item{data_frame}{A data frame in which are the variables to tabulate.}

\item{rows}{The variable that appears in the table rows.}

\item{columns}{The variable that appears in the table columns.}

\item{statistics}{The user requested statistics.Available functions:
\itemize{
\item "sum"        -> Weighted and unweighted sum
\item "freq"       -> Unweighted frequency
\item "pct_row"    -> Weighted and unweighted row percentages
\item "pct_column" -> Weighted and unweighted column percentages
\item "pct_total"  -> Weighted and unweighted percentages compared to the grand total
}}

\item{formats}{A list in which is specified which formats should be applied to which variables.}

\item{by}{Compute tables stratified by the expressions of the provided variables.}

\item{weight}{Put in a weight variable to compute weighted results.}

\item{titles}{Specify one or more table titles.}

\item{footnotes}{Specify one or more table footnotes.}

\item{style}{A list of options can be passed to control the appearance of excel outputs.
Styles can be created with \code{\link[=excel_output_style]{excel_output_style()}}.}

\item{output}{The following output formats are available: console (default), text,
excel and excel_nostyle.}

\item{na.rm}{FALSE by default. If TRUE removes all NA values from the variables.}

\item{print}{TRUE by default. If TRUE prints the output, if FALSE doesn't print anything. Can be used
if one only wants to catch the output data frame.}

\item{monitor}{FALSE by default. If TRUE outputs two charts to visualize the functions time consumption.}
}
\value{
Returns a data tables containing the results for the cross table.
}
\description{
\code{\link[=crosstabs]{crosstabs()}} produces a cross table of two variables. Statistics can be
weighted sums, unweighted frequencies or different percentages.
}
\details{
\code{\link[=crosstabs]{crosstabs()}} is based on the 'SAS' procedure Proc Freq, which provides
efficient and readable ways to perform cross tabulations.

To create a cross table you only need to provide a variable for the rows and columns.
Nothing special about this. The real power comes into play, when you output your
tables as a fully styled 'Excel' workbook. Setting up a custom, reusable style is
as easy as setting up options like: provide a color for the table header, set the
font size for the row header, should borders be drawn for the table cells yes/no,
and so on.

You can not only output sums and frequencies, but also different percentages, all
set up in separate, evenly designed tables. For just a quick overview, rather than
fully designed tables, you can also just output the tables in ASCII style format.
}
\examples{
# Example data frame
my_data <- dummy_data(1000)

# Define titles and footnotes. If you want to add hyperlinks you can do so by
# adding "link:" followed by the hyperlink to the main text.
titles <- c("This is title number 1 link: https://cran.r-project.org/",
            "This is title number 2",
            "This is title number 3")
footnotes <- c("This is footnote number 1",
               "This is footnote number 2",
               "This is footnote number 3 link: https://cran.r-project.org/")

# Output cross tables
my_data |> crosstabs(age, sex)
my_data |> crosstabs(age, sex,
                     weight = "weight")

# Also works with characters
my_data |> crosstabs("age", "sex")
my_data |> crosstabs("age", "sex",
                     weight = "weight")

# Applying formats and titles
age. <- discrete_format(
    "Total"          = 0:100,
    "under 18"       = 0:17,
    "18 to under 25" = 18:24,
    "25 to under 55" = 25:54,
    "55 to under 65" = 55:64,
    "65 and older"   = 65:100)

sex. <- discrete_format(
    "Total"  = 1:2,
    "Male"   = 1,
    "Female" = 2)

my_data |> crosstabs(age, sex,
                     formats   = list(age = age., sex = sex.),
                     titles    = titles,
                     footnotes = footnotes)

# Split cross table by expressions of another variable
my_data |> crosstabs(age, sex, by = education)

# Compute different stats
my_data |> crosstabs(age, sex,
                     statistics = c("sum", "freq", "pct_row", "pct_column", "pct_total"))

# Get a list with two data tables for further usage
result_list <- my_data |> crosstabs(age, sex,
                                    formats = list(age = age., sex = sex.))

# Output in text file
my_data |> crosstabs(age, sex, output = "text")

# Output to Excel
my_data |> crosstabs(age, sex, output = "excel")

# With individual styling
my_style <- excel_output_style(header_back_color = "0077B6",
                               font              = "Times New Roman")

my_data |> crosstabs(age, sex, output = "excel", style = my_style)

}
\seealso{
Creating a custom table style: \code{\link[=excel_output_style]{excel_output_style()}}, \code{\link[=modify_output_style]{modify_output_style()}},
\code{\link[=number_format_style]{number_format_style()}}, \code{\link[=modify_number_formats]{modify_number_formats()}}.

Creating formats: \code{\link[=discrete_format]{discrete_format()}} and \code{\link[=interval_format]{interval_format()}}.

Functions that can handle formats and styles: \code{\link[=frequencies]{frequencies()}}, \code{\link[=any_table]{any_table()}}.

Additional functions that can handle styles: \code{\link[=export_with_style]{export_with_style()}}

Additional functions that can handle formats: \code{\link[=summarise_plus]{summarise_plus()}}, \code{\link[=recode]{recode()}},
\code{\link[=recode_multi]{recode_multi()}}
}
