\name{qb.mcmc}
\alias{qb.mcmc}
\title{Bayesian Mutiple Interacting QTL mapping using MCMC}
\description{
  A computationally efficient MCMC algorithm using the Gibbs sampler or 
  Metropolis-Hastings algorithm is used to produce posterior samples for 
  QTL mapping.   
}
\usage{
qb.mcmc(cross, data, model, mydir = ".", n.iter = 3000, n.thin = 40,
  n.burnin = 0.01*n.iter*n.thin, algorithm = c("M-H","Gibbs"),
  genoupdate = TRUE, seed = 0, verbose = TRUE, \dots)   
}
\arguments{
  \item{cross}{ An object of class \code{cross}. See \code{\link[qtl]{read.cross}} for details. }
  \item{data}{ the list returned by calling the function \code{\link{qb.data}} .}
  \item{model}{the list returned by calling the function \code{\link{qb.model}}.}
  \item{mydir}{ a directory to save output from \code{qb.mcmc} in several \file{*.dat} files.
                A directory is created using the trait name and the system time and date.
                If no directory is specified, the default directory is the current working 
                directory.}
  \item{n.iter}{number of iterations to be saved in \code{mydir}, the default being
                3000. Note that, \code{n.iter} is not the total number of         iterations performed but the number iterations saved or considered as posterior
                samples for future analysis. The actual number of iterations would be \code{n.burnin + n.iter*n.thin}                                                
                }
  \item{n.thin}{the thinning number which must be a positive number (default=40)}
  \item{n.burnin}{the initial burn-in period, i.e number of iterations to discard
                  at the beginning of the MCMC run default being 0.01*n.iter*n.thin.}
  \item{algorithm}{specifies the sampling algorithm for MCMC : Gibbs sampler ("Gibbs")
                  or Metropolis-Hastings algorithm ("M-H")}
  \item{genoupdate}{=TRUE will update QTL genotypes and =FALSE will not do so and use
                      the expected value of the QTL genotypes.}
  \item{seed}{Specifies the seed for the random number generator. Using the same seed
              for two runs of the \code{qb.mcmc} function will generate the exact same
              output. The \code{seed} needs to be an integer. The default value for seed
              is the system time.}
  \item{verbose}{=TRUE will force periodic output of the number of MCMC iterations saved.
                 The location of the output directory where results are stored and
                  the time taken for the MCMC run will also be displayed to the user.}
  \item{\dots}{Paramters passed to \code{\link{qb.data}} or
    \code{\link{qb.model}} if \code{data} or \code{model}, respectively,
    is not provided.}
}

\details{
A composite model space approach to develop a Bayesian model selection framework for identifying
interacting QTL for complex traits in experimental crosses from two inbred lines. By placing a liberal
constraint on the upper bound of the number of detectable QTL we restrict attention to models of 
fixed dimension. Either Gibbs sampler or Metroplis-Hastings algorithm can be applied to sample
from the posterior distribution.            
   }
   

\value{
The \code{qb.mcmc} function returns a list of class \code{qb}, the components of which
contain input parameters from the \code{cross} object, \code{qb.data} and \code{qb.model}.
Since, the parameters have already been described in their respective man pages 
we only include the components which have been added on top of these. 
However, the \code{qbOject$subset} component is a way to manipulate the
Monte Carlo samples to make it ready for the high-end plotting routines and might not be
of much importance to the average user.

  \item{output.dir}{directory used for saving all outputs generated by \code{qb.mcmc}.}

  \item{subset}{
  \describe{  
            \item \strong{\code{iterdiag}}  is a vector of integers from 1 to \code{n.iter}.
            \item \strong{\code{mainloci}}  is a vector of length equaling the sum of the number of
                                   QTLs detected in each iteration. This vector is actually
                                   a permutation vector of the \code{mainloci.dat}
                                   file stored in \code{mydir} sorted with respect to the
                                   iteration number and ties are broken with the chromosome
                                   number and the locus of the putative QTL.
          \item \strong{\code{pairloci}}  is a list of 
          
                        \itemize{
                       \item \emph{\code{order}} is a vector of integers from 1 to the total number
                                        of epistatic effects in all iterations.
                          
                          
                        \item  \emph{\code{flip}}  is  a logical vector = TRUE if chromosome no. 1 for the paired
                                              loci is greater than chromosome no.2 or if they are the same
                                              on the same chromosome then =TRUE when position of the first is greater
                                              than the position of the second.  
                         \item  \emph{\code{left}} is a character vector containing "chrom1" and "locus1" if there is a single TRUE in \code{flip}
                          \item  \emph{\code{right}} is a character vector containing "chrom2" and "locus2" if there is a single TRUE in \code{flip}
                                 }
          \item \strong{\code{region}}   a data frame storing the first and last position of the 
                          marker map for each chromosome.
            
         }
 
 }
        
}

\section{Output Files}{
The following files are saved in \code{output.dir}:   
\enumerate{
 \item \bold{\file{iterdiag.dat}}
                        
                         The iteration file saved in \code{mydir} has \code{n.iter} rows and
                        4 major columns:\cr
                        \code{column no 1} :  iteration number.\cr
                        \code{column no 2} :  number of putative QTLs included.\cr
                        \code{column no 3} :  the overall mean.\cr
                        \code{column no 4} :  the residual variance.\cr

                        Depending on the type of cross, presence of covariates and epistatic
                        effects there would be more columns in the following order:
                        variance of additive effect, variance of dominant effect, variance of
                        additive-additive interaction, variance of additive-dominant interaction,
                        variance of dominant-additive interaction, variance of dominant-dominant
                        interaction, variance of environment-additive interaction, variance of
                        environmnet-dominant interaction, variance of environemnt effect, total
                        genetic variance.\cr\cr
                       

  \item \bold{\file{covariates.dat}}

                        The covariate file saved in \code{mydir} has \code{n.iter} rows and
                        L+M(\code{length(fixcov)+length(rancov)}) columns:\cr
                        \code{L columns} : Coeffecient of the fixed effect.\cr
                        \code{M columns} : Variance of the random effect. \cr\cr 
                        If an ordinal trait is analyzed, the cutoff points for the threshold
                        model are also included in additional columns. There would be C-3
                        bounded threshold values for an ordinal phenotype with C categories.\cr\cr
                             
  \item \bold{\file{mainloci.dat}}
                             
                        The mainloci file saved in \code{mydir} has the N rows (N=sum of number of QTLs
                        detected in \code{n.iter} iterations) and 4 major columns:\cr
                        \code{column no 1} : iteration number.\cr
                        \code{column no 2} : number of putative QTLs included in the model.\cr
                        \code{column no 3} : the chromosome number on which a putative QTL has been detected.\cr
                        \code{column no 4} : the QTL position indicator in the grid.\cr\cr
                        Depending on the type of cross there would be more columns in the 
                        following order: additive effect, dominant effect, variance of additive effect,
                        and variance of dominant effect.\cr\cr
                            
  \item \bold{\file{pairloci.dat}}
                        
                        The pairloci file saved in \code{mydir} has the N rows (N=sum of number
                         of pairs of QTLs with epistatic effect detected) and 6 major columns:\cr
                        \code{column no 1} : iteration number.\cr
                        \code{column no 2} : number of pairs of QTLs detected to have epistatic effect.\cr
                        \code{column no 3} : the chromosome number for the first one of each pair.\cr
                        \code{column no 4} : the QTL position for this one.\cr
                        \code{column no 5} : the chromosome number for the second one of each pair.\cr
                        \code{column no 6} : the QTL position for this one.\cr\cr

                        Depending on the type of cross there would be more columns in the following order:
                        additive-additive interaction effect, additive-dominant interaction effect,
                        dominant-additive interaction effect, dominant-dominant interaction effect,
                        variance of additive-additive interaction, variance of additive-dominant interaction,
                        variance of dominant-additive interaction, variance of dominant-dominant
                        interaction.\cr\cr
                        
  \item \bold{\file{gbye.dat}}
                        
                         The gbye (Gene by Environment) file saved in \code{mydir} has 5 major columns:\cr

                        \code{column no 1} :  iteration number.\cr
                        \code{column no 2} :  number of GxE interactions.\cr
                        \code{column no 3} :  fixed covariate number.\cr
                        \code{column no 4} :  chromosome number of the putative QTL in the GxE interaction.\cr
                        \code{column no 5} :  position of the corresponding QTL.\cr\cr

                        Depending on the type of cross there would be more columns in the following order:
                        additive effect, dominant effect, variance of additive effect,
                        and variance of dominant effect.
                        

                        
                        
                              
         }
}
   
   
\references{
  \url{http://www.qtlbim.org}}
\author{Nengjun Yi, nyi@ms.ssg.uab.edu}

\seealso{\code{\link{qb.sim.cross}},   \code{\link{qb.data}},   
  \code{\link{qb.model}},   \code{\link{qb.mcmc}}}


\examples{
\dontrun{
example(qb.sim.cross)

## Calculate grids and genotypic probabilites.
cross <- qb.genoprob(cross, step=2)

## Create MCMC samples
## First line as qb.data options; second line has qb.model options.
qbExample <- qb.mcmc(cross, pheno.col = 3, rancov = 2, fixcov = 1,
  chr.nqtl = rep(3,nchr(cross)), intcov = 1, interval = rep(10,3),
  n.iter = 1000, n.thin = 20)
}
}
\keyword{models}
\keyword{regression}
