% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/isomap.R
\name{step_isomap}
\alias{step_isomap}
\alias{tidy.step_isomap}
\title{Isomap Embedding}
\usage{
step_isomap(recipe, ..., role = "predictor", trained = FALSE,
  num_terms = 5, neighbors = 50, options = list(.mute = c("message",
  "output")), res = NULL, num = NULL, prefix = "Isomap",
  skip = FALSE, id = rand_id("isomap"))

\method{tidy}{step_isomap}(x, ...)
}
\arguments{
\item{recipe}{A recipe object. The step will be added to the
sequence of operations for this recipe.}

\item{...}{One or more selector functions to choose which
variables will be used to compute the dimensions. See
\code{\link[=selections]{selections()}} for more details. For the \code{tidy}
method, these are not currently used.}

\item{role}{For model terms created by this step, what analysis
role should they be assigned?. By default, the function assumes
that the new dimension columns created by the original variables
will be used as predictors in a model.}

\item{trained}{A logical to indicate if the quantities for
preprocessing have been estimated.}

\item{num_terms}{The number of isomap dimensions to retain as new
predictors. If \code{num_terms} is greater than the number of columns
or the number of possible dimensions, a smaller value will be
used.}

\item{neighbors}{The number of neighbors.}

\item{options}{A list of options to \code{\link[dimRed:Isomap]{dimRed::Isomap()}}.}

\item{res}{The \code{\link[dimRed:Isomap]{dimRed::Isomap()}} object is stored
here once this preprocessing step has be trained by
\code{\link[=prep.recipe]{prep.recipe()}}.}

\item{num}{The number of isomap dimensions (this will be deprecated
in factor of  \code{num_terms} in version 0.1.5). \code{num_terms} will
override this option.}

\item{prefix}{A character string that will be the prefix to the
resulting new variables. See notes below.}

\item{skip}{A logical. Should the step be skipped when the
recipe is baked by \code{\link[=bake.recipe]{bake.recipe()}}? While all operations are baked
when \code{\link[=prep.recipe]{prep.recipe()}} is run, some operations may not be able to be
conducted on new data (e.g. processing the outcome variable(s)).
Care should be taken when using \code{skip = TRUE} as it may affect
the computations for subsequent operations}

\item{id}{A character string that is unique to this step to identify it.}

\item{x}{A \code{step_isomap} object}
}
\value{
An updated version of \code{recipe} with the new step
added to the sequence of existing steps (if any). For the
\code{tidy} method, a tibble with columns \code{terms} (the
selectors or variables selected).
}
\description{
\code{step_isomap} creates a \emph{specification} of a recipe
step that will convert numeric data into one or more new
dimensions.
}
\details{
Isomap is a form of multidimensional scaling (MDS).
MDS methods try to find a reduced set of dimensions such that
the geometric distances between the original data points are
preserved. This version of MDS uses nearest neighbors in the
data as a method for increasing the fidelity of the new
dimensions to the original data values.

This step requires the \pkg{dimRed}, \pkg{RSpectra},
\pkg{igraph}, and \pkg{RANN} packages. If not installed, the
step will stop with a note about installing these packages.

It is advisable to center and scale the variables prior to
running Isomap (\code{step_center} and \code{step_scale} can be
used for this purpose).

The argument \code{num_terms} controls the number of components that
will be retained (the original variables that are used to derive
the components are removed from the data). The new components
will have names that begin with \code{prefix} and a sequence of
numbers. The variable names are padded with zeros. For example,
if \code{num_terms < 10}, their names will be \code{Isomap1} -
\code{Isomap9}. If \code{num_terms = 101}, the names would be
\code{Isomap001} - \code{Isomap101}.
}
\examples{
\donttest{
data(biomass)

biomass_tr <- biomass[biomass$dataset == "Training",]
biomass_te <- biomass[biomass$dataset == "Testing",]

rec <- recipe(HHV ~ carbon + hydrogen + oxygen + nitrogen + sulfur,
              data = biomass_tr)

im_trans <- rec \%>\%
  step_YeoJohnson(all_predictors()) \%>\%
  step_center(all_predictors()) \%>\%
  step_scale(all_predictors()) \%>\%
  step_isomap(all_predictors(),
              neighbors = 100,
              num_terms = 2)

if (require(dimRed) & require(RSpectra)) {
  im_estimates <- prep(im_trans, training = biomass_tr)

  im_te <- bake(im_estimates, biomass_te)

  rng <- extendrange(c(im_te$Isomap1, im_te$Isomap2))
  plot(im_te$Isomap1, im_te$Isomap2,
       xlim = rng, ylim = rng)

  tidy(im_trans, number = 4)
  tidy(im_estimates, number = 4)
}
}
}
\references{
De Silva, V., and Tenenbaum, J. B. (2003). Global
versus local methods in nonlinear dimensionality reduction.
\emph{Advances in Neural Information Processing Systems}.
721-728.

\pkg{dimRed}, a framework for dimensionality reduction,
https://github.com/gdkrmr
}
\seealso{
\code{\link[=step_pca]{step_pca()}} \code{\link[=step_kpca]{step_kpca()}}
\code{\link[=step_ica]{step_ica()}} \code{\link[=recipe]{recipe()}} \code{\link[=prep.recipe]{prep.recipe()}}
\code{\link[=bake.recipe]{bake.recipe()}}
}
\concept{preprocessing isomap projection_methods}
\keyword{datagen}
