% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/redist_ms_tidy.R, R/redist_ms.R
\name{redist_mergesplit}
\alias{redist_mergesplit}
\alias{redist.mergesplit}
\title{Merge-Split/Recombination MCMC Redistricting Sampler}
\usage{
redist_mergesplit(
  map,
  nsims,
  warmup = floor(nsims/2),
  init_plan = NULL,
  counties = NULL,
  compactness = 1,
  constraints = list(),
  constraint_fn = function(m) rep(0, ncol(m)),
  adapt_k_thresh = 0.975,
  k = NULL,
  init_name = NULL,
  verbose = TRUE,
  silent = FALSE
)

redist.mergesplit(
  adj,
  total_pop,
  nsims,
  ndists,
  pop_tol = 0.01,
  init_plan,
  counties,
  compactness = 1,
  constraints = list(),
  constraint_fn = function(m) rep(0, ncol(m)),
  adapt_k_thresh = 0.975,
  k = NULL,
  verbose = TRUE,
  silent = FALSE
)
}
\arguments{
\item{map}{A \code{\link{redist_map}} object.}

\item{nsims}{The number of samples to draw, including warmup.}

\item{warmup}{The number of warmup samples to discard.}

\item{init_plan}{The initial state of the map. If not provided, will default to
the reference map of the \code{map} object, or if none exists, will sample
a random initial state using \code{\link{redist_smc}}. You can also request
a random initial state by setting \code{init_plan="sample"}.}

\item{counties}{A vector containing county (or other administrative or
geographic unit) labels for each unit, which may be integers ranging from 1
to the number of counties, or a factor or character vector.  If provided, the
algorithm will generate maps tend to follow county lines.  You may combine this
with a Gibbs constraint on the number of county splits using the
\code{constraints} parameter; see below. If no county-split considerations
are desired, this parameter should be left blank.}

\item{compactness}{Controls the compactness of the generated districts, with
higher values preferring more compact districts. Must be nonnegative. See the
'Details' section for more information, and computational considerations.}

\item{constraints}{A list containing information on constraints to implement.
See the 'Details' section for more information.}

\item{constraint_fn}{A function which takes in a matrix where each column is
a redistricting plan and outputs a vector of log-weights, which will be
added the the final weights.}

\item{adapt_k_thresh}{The threshold value used in the heuristic to select a
value \code{k_i} for each splitting iteration. Set to 0.9999 or 1 if
the algorithm does not appear to be sampling from the target distribution.
Must be between 0 and 1.}

\item{k}{The number of edges to consider cutting after drawing a spanning
tree. Should be selected automatically in nearly all cases.}

\item{init_name}{a name for the initial plan, or \code{FALSE} to not include
the initial plan in the output.  Defaults to the column name of the
existing plan, or "\code{<init>}" if the initial plan is sampled.}

\item{verbose}{Whether to print out intermediate information while sampling.
Recommended.}

\item{silent}{Whether to suppress all diagnostic information.}

\item{adj}{adjacency matrix, list, or object of class
"SpatialPolygonsDataFrame."}

\item{total_pop}{A vector containing the populations of each geographic
unit}

\item{ndists}{The number of congressional districts.}

\item{pop_tol}{The desired population constraint.  All sampled districts
will have a deviation from the target district size no more than this value
in percentage terms, i.e., \code{pop_tol=0.01} will ensure districts have
populations within 1\% of the target population.}
}
\value{
\code{redist_mergesplit} returns an object of class
\code{\link{redist_plans}} containing the simulated plans.

\code{redist.mergesplit} returns an object of class list containing the
simulated plans.
}
\description{
\code{redist_mergesplit} uses a Markov Chain Monte Carlo algorithm to
generate congressional or legislative redistricting plans according to
contiguity, population, compactness, and administrative boundary constraints.
The MCMC proposal is the same as is used in the SMC sampler; it is similar
but not identical to those used in the references.  1-level hierarchical
Merge-split is supported through the \code{counties} parameter; unlike in
the SMC algorithm, this does not guarantee a maximum number of county splits.
}
\details{
This function draws samples from a specific target measure, controlled by the
\code{compactness}, \code{constraints}, and \code{constraint_fn} parameters.

Higher values of \code{compactness} sample more compact districts;
setting this parameter to 1 is computationally efficient and generates nicely
compact districts.

The \code{constraints} parameter allows the user to apply several common
redistricting constraints without implementing them by hand. This parameter
is a list, which may contain any of the following named entries:
\itemize{
\item \code{status_quo}: a list with two entries:
\itemize{
\item \code{strength}, a number controlling the tendency of the generated districts
to respect the status quo, with higher values preferring more similar
districts.
\item \code{current}, a vector containing district assignments for
the current map.
}
\item \code{vra}: a list with three entries:
\itemize{
\item \code{strength}, a number controlling the strength of the Voting Rights Act
(VRA) constraint, with higher values prioritizing majority-minority districts
over other considerations.
\item \code{tgts_min}, the target percentage(s) of minority voters in minority
opportunity districts. Defaults to \code{c(0.55)}.
\item \code{min_pop}, A vector containing the minority population of each
geographic unit.
}
\item \code{incumbency}: a list with two entries:
\itemize{
\item \code{strength}, a number controlling the tendency of the generated districts
to avoid pairing up incumbents.
\item \code{incumbents}, a vector of precinct indices, one for each incumbent's
home address.
}
\item \code{splits}: a list with one entry:
\itemize{
\item \code{strength}, a number controlling the tendency of the generated districts
to avoid splitting counties.
}
\item \code{vra_old}: a list with five entries, which may be set up using
\code{\link{redist.constraint.helper}}:
\itemize{
\item \code{strength}, a number controlling the strength of the Voting Rights Act
(VRA) constraint, with higher values prioritizing majority-minority districts
over other considerations.
\item \code{tgt_vra_min}, the target percentage of minority voters in minority
opportunity districts. Defaults to 0.55.
\item \code{tgt_vra_other} The target percentage of minority voters in other
districts. Defaults to 0.25, but should be set to reflect the total minority
population in the state.
\item \code{pow_vra}, which controls the allowed deviation from the target
minority percentage; higher values are more tolerant. Defaults to 1.5
\item \code{min_pop}, A vector containing the minority population of each
geographic unit.
}
}

All constraints are fed into a Gibbs measure, with coefficients on each
constraint set by the corresponding \code{strength} parameters.
The \code{status_quo} constraint adds a term measuring the variation of
information distance between the plan and the reference, rescaled to [0, 1].
The \code{vra} constraint takes a list of target minority percentages. It
matches each district to its nearest target percentage, and then applies a
penalty of the form \eqn{\sqrt{max(0, tgt - minpct)}}, summing across
districts. This penalizes districts which are below their target population.
The \code{incumbency} constraint adds a term counting the number of districts
containing paired-up incumbents. The \code{splits} constraint adds a term
counting the number of counties which contain precincts belonging to more
than one district.
The \code{vra_old} constraint adds a term of the form
\eqn{(|tgtvramin-minpct||tgtvraother-minpct|)^{powvra})}, which
encourages districts to have minority percentages near either \code{tgt_vra_min}
or \code{tgt_vra_other}. This can be visualized with
\code{\link{redist.plot.penalty}}.
}
\examples{
\donttest{
data(fl25)

fl_map = redist_map(fl25, ndists=3, pop_tol=0.1)

sampled_basic = redist_mergesplit(fl_map, 10000)

sampled_constr = redist_mergesplit(fl_map, 10000, constraints=list(
                     incumbency = list(strength=1000, incumbents=c(3, 6, 25))
                 ))
}

data(fl25)
adj <- redist.adjacency(fl25)
out <- redist.mergesplit(adj = adj, total_pop = fl25$pop,
                         nsims = 5, ndists = 3, pop_tol = 0.1)

}
\references{
Carter, D., Herschlag, G., Hunter, Z., and Mattingly, J. (2019). A
merge-split proposal for reversible Monte Carlo Markov chain sampling of
redistricting plans. arXiv preprint arXiv:1911.01503.

DeFord, D., Duchin, M., and Solomon, J. (2019). Recombination: A family of
Markov chains for redistricting. arXiv preprint arXiv:1911.05725.
}
\concept{simulate}
