% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/integrate.R
\name{integrate_gk}
\alias{integrate_gk}
\title{Adaptive Gauss-Kronrod Quadrature for multiple limits}
\usage{
integrate_gk(
  fun,
  lower,
  upper,
  params = list(),
  .tolerance = .Machine$double.eps^0.25,
  .max_iter = 100L
)
}
\arguments{
\item{fun}{A function to integrate.
Must be vectorized and take one or two arguments, the first being points to
evaluate at and the second (optionally) being parameters to apply.
It must return a numeric vector the same length as its first input.

Currently, infinite bounds are not supported.}

\item{lower, upper}{Integration bounds. Must have the same length.}

\item{params}{Parameters to pass as a second argument to \code{fun}.
The actual parameters must have the same length as the number of integrals to
compute. Can be a possibly nested list structures containing numeric vectors.
Alternatively, can be a matrix with the same number of rows as the number of integrals to compute.}

\item{.tolerance}{Absolute element-wise tolerance.}

\item{.max_iter}{Maximum number of iterations. The number of
integration intervals will be at most \code{length(lower) * .max_iter}. Therefor the maximum
number of function evaluations per integration interval will be
\code{15 * .max_iter}.}
}
\value{
A vector of integrals with the i-th entry containing an approximation
of

int_{lower[i]}^{upper[i]} fun(t, pick_params_at(params, i)) dt
}
\description{
Integrates fun over the bounds [ lower, upper ] vectorized over \code{lower} and
\code{upper}. Vectorized list structures of parameters can also be passed.
}
\details{
The integration error is estimated by the Gauss-Kronrod quadrature as the
absolute difference between the 7-point quadrature and the 15-point
quadrature. Integrals that did not converge will be bisected at the midpoint.
The \code{params} object will be recursively subsetted on all numeric vectors with
the same length as the number of observations.
}
\examples{
# Argument recycling and parallel integration of two intervals
integrate_gk(sin, 0, c(pi, 2 * pi))

dist <- dist_exponential()
integrate_gk(
  function(x, p) dist$density(x, with_params = p),
  lower = 0, upper = 1:10,
  params = list(rate = 1 / 1:10)
)
dist$probability(1:10, with_params = list(rate = 1 / 1:10))

}
