#' Create a New Order For A Currency
#'
#' @name add_order
#'
#' @aliases place_order post_order
#'
#' @description This is an auth based function. User must have valid api keys generated by GADX which must be passed as mandatory arguments. The users can place different types of orders like \code{"limit"}, \code{"stop"} or \code{"market"}. Orders will be placed succesfully only if there is sufficient funds. Each order will result in a hold and the details of the hold can be tracked using \code{\link{holds}}. Margin Orders and Stop Orders are currently not supported.
#'
#' @param api.key Mandatory character value. This is the API key as generated by GDAX. Typically a 32 character value.
#' @param secret Mandatory character value. This is the API secret as generated by GDAX. Typically a 88 character value.
#' @param passphrase Mandatory character value. This is the passphrase as generated by GDAX. Typically a 11 character value.
#' @param product_id Optional character value for the currency pair. The default is \code{"LTC-USD"}. This param is case insensitive and must be one of the valid currency-pair. The list of valid currency-pairs can be fetched using \code{\link{public_info}}.
#' @param type Optional character value for the order type. The default is \code{"limit"}. This param is case insensitive and must be one of the valid order types. The current valid order types is only \code{"limit"}.
#' @param side Optional character value for the order side The default is \code{"b"} which stands for \code{buy}. This param is case insensitive and must be one of either \code{"b"} (buy) or \code{"s"} (sell).
#' @param price Mandatory numeric value. It can either be an integer or float. Float values of greater than 2 decimals will be rounded to 2 decimals using the generic \code{round} function from R.
#' @param size Mandatory numeric value. It can either be an integer or float. Float values will \bold{NOT} be rounded. The user must ensure that the fractional unit of a currency pair is valid for acceptance by GDAX. This information can also be determined by \code{\link{public_info}} which provides the minimun and maximum order sizes for each currency pairs.
#'
#' @return  Dataframe with status of the order, posted details and created time stamp etc.
#'
#' @examples
#' \dontrun{
#' add_order("BTC-USD", api.key = your_key, secret = your_api_secret, passphrase = your_api_pass,
#'      type="limit", side = "sell", price = 1000.25, size = 1)
#' }
#'
#' @importFrom jsonlite toJSON
#' @export


add_order <- function(api.key,
                      secret,
                      passphrase,
                      product_id = "LTC-USD",
                      type = "limit",
                      side = "b",
                      price,
                      size) {
  #get url extension----
  req.url <- "/orders"

  #define method----
  method <-  "POST"

  #transform params----
  product_id <- toupper(product_id)
  price <- as.character(price)
  size <- as.character(size)

  if (side == "b") {
    side <- "buy"
  } else if (side == "s") {
    side <- "sell"
  } else {
    stop("Unrecognized sell or buy side. Please select either 'b' or 's'.")
  }

  #generate order----
  order_attribs <- list(
    price = price,
    size = size,
    side = side,
    product_id = product_id
  )

  order <- toJSON(order_attribs, auto_unbox = TRUE)

  #fetch response----
  response <-
    auth(
      method = method,
      req.url = req.url,
      api.key = api.key,
      secret = secret,
      passphrase = passphrase,
      order = order
    )

  #transform----
  response <- as.data.frame(response)

  #return----
  return(response)
}
