#' Create Signature for GDAX API
#'
#'
#' @name auth
#'
#' @description This is an internal function that will be used for all private connections to the user account. This function determines the kind of GDAX call (GET / POST / DELETE).
#'
#' @param method Mandatory character value. Value must be upper case.
#' @param req.url THE URL component for the API. Default to "/accounts".
#' @param api.key Mandatory character value. This is the API key as generated by GDAX. Typically a 32 character value.
#' @param secret Mandatory character value. This is the API secret as generated by GDAX. Typically a 88 character value.
#' @param passphrase Mandatory character value. This is the passphrase as generated by GDAX. Typically a 11 character value.
#' @param order Optional named list. If method is \code{POST} then the field is required for post to work, else the api end point will return an error.
#'
#' @return  A named list of server response.
#'
#' @import digest
#'


# function definition ----
auth <- function(method,
                 req.url,
                 api.key,
                 secret,
                 passphrase,
                 order = NULL) {
  #define api base url----
  api.url <- "https://api.gdax.com"

  #generate nonce and key encodings----
  url <- paste0(api.url, req.url)
  timestamp <-
    format(as.numeric(Sys.time()), digits = 13) # create nonce
  key <- base64Decode(secret, mode = "raw") # encode api secret

  #create final end point----
  if (method == "GET") {
    what <- paste0(timestamp, method, req.url) # get method
  } else if (method == "POST") {
    what <- paste0(timestamp, method, req.url, order)
  } else if (method == "DELETE") {
    what <- paste0(timestamp, method, req.url)
  }

  #create encoded signature----
  sign <-
    base64Encode(hmac(key, what, algo = "sha256", raw = TRUE)) # hash

  #define headers----
  httpheader <- list(
    'CB-ACCESS-KEY' = api.key,
    'CB-ACCESS-SIGN' = sign,
    'CB-ACCESS-TIMESTAMP' = timestamp,
    'CB-ACCESS-PASSPHRASE' = passphrase,
    'Content-Type' = 'application/json'
  )

  #generating GET results----
  if (method == "GET") {
    #Get test macOS----
    if (Sys.info()["sysname"] == "Darwin") {
      response <- fromJSON(rawToChar(
        getURLContent(
          url = url,
          curl = getCurlHandle(useragent = "R"),
          httpheader = httpheader
        )
      ))
    }
    #Get test windows----
    else {
      response <- fromJSON(getURLContent(
        url = url,
        curl = getCurlHandle(useragent = "R"),
        httpheader = httpheader
      ))
    }
  }
  #generating POST results----
  else if (method == "POST") {
    #Post test macOS----
    if (Sys.info()["sysname"] == "Darwin") {
      response <- fromJSON(rawToChar(
        getURLContent(
          url = url,
          curl = getCurlHandle(useragent = "R"),
          httpheader = httpheader,
          postfields = order
        )
      ))
    }
    #Post test windows----
    else{
      response <- fromJSON(
        getURLContent(
          url = url,
          curl = getCurlHandle(useragent = "R"),
          httpheader = httpheader,
          postfields = order
        )
      )
    }
  }
  #Generating DELETE results
  else if (method == "DELETE") {
    #Post test macOS----
    if (Sys.info()["sysname"] == "Darwin") {
      response <- fromJSON(rawToChar(httpDELETE(
        url = url,
        curl = getCurlHandle(useragent = "R"),
        httpheader = httpheader
      )))
    } else {
      response <- fromJSON(httpDELETE(
        url = url,
        curl = getCurlHandle(useragent = "R"),
        httpheader = httpheader
      ))
    }
  }

  #return----
  return(response)
}
