% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ring_buffer_bytes_typed.R
\name{ring_buffer_bytes_typed}
\alias{ring_buffer_bytes_typed}
\title{Typed bytes ring buffer}
\usage{
ring_buffer_bytes_typed(size, what, len = NULL, on_overflow = "overwrite")
}
\arguments{
\item{size}{The maximum number of elements the buffer can hold.
Each element will be multiple bytes long.}

\item{what}{Either a vector on the style of \code{\link{vapply}} (e.g.,
\code{integer(4)} to indicate that each element of the buffer is a
4-element integer, or the \code{name} of a storage mode if \code{len} is
also provided.}

\item{len}{If given, then the length of the storage.  If it is
given, then if \code{length(what)} is zero, the storage mode of
\code{what} is used as the type.  Otherwise \code{what} is interpreted as
the \emph{name} of the storage mode (one of "logical", "integer",
"double" or "complex".}

\item{on_overflow}{Behaviour on buffer overflow.  The default is
to overwrite the oldest elements in the buffer
(\code{"overwrite"}).  Alternative actions are \code{"error"}
which will throw an error if a function tries to add more
elements than there are space for, or \code{"grow"} which will
grow the buffer to accept the new elements (this uses an
approximately golden ratio approach; see details below).}
}
\description{
Create a ring buffer, backed by a \code{\link{ring_buffer_bytes}},
where each element corresponds to a fixed-size vector of one of
R's atomic numeric types (logical, integer, double, and complex).
}
\details{
Note that a logical ring buffer and an integer ring buffer take
the same number of bytes because a logical vector is stored as an
integer (4 bytes per element) to deal with missing values; see
"writing R extensions".

Note that it is not possible to store character vectors in a ring
buffer of this type because each element of a character vector can
be any number of bytes.
}
\section{Methods}{


Note that this methods reference section is repeated verbatim between
the three main ring buffer classes; \code{ring_buffer_env}
("env"), \code{ring_buffer_bytes} ("bytes") and
\code{ring_buffer_bytes_typed} ("typed").  Almost all methods have
the same arguments and behaviour, but hopefully by listing everything together,
the differences between implementations will be a bit more apparent.

\describe{
\item{\code{reset}}{
Reset the state of the buffer.  This "zeros" the head and tail pointer (and may or may not actually reset the data) so that the buffer can be used as if fresh.

\emph{Usage:}
\code{reset(clear = FALSE)}

\emph{Arguments:}
\itemize{
\item{\code{clear}:   Logical, indicating if the memory should also be cleared. Generally this is not necessary, but with environment buffers this can let the garbage collector clean up large elements.  For the bytes buffer this zeros the memory.
}
}

\emph{Return value}:
Nothing; called for the side effect only.
}
\item{\code{duplicate}}{
Clone the ring buffer, creating a copy.  Copies both the underlying data and the position of the head and tail.

\emph{Usage:}
\code{duplicate()}

\emph{Return value}:
A new ring buffer object
}
\item{\code{grow}}{
Increase the size of the buffer by \code{n} elements.

\emph{Usage:}
\itemize{
\item{bytes, typed: \code{grow(n)}}
\item{env: \code{grow(n, exact = FALSE)}}
}

\emph{Arguments:}
\itemize{
\item{\code{n}:   The number of additional elements that space should be reserved for (scalar non-negative integer).
}

\if{html}{\out{<div class="sourceCode">}}\preformatted{\\item\{\code{exact}:   (For bytes buffer only) Logical scalar indicating if growth should increase the size by \\emph\{exactly\} \code{n} elements (if \code{TRUE}) or so that \\emph\{at least\} \code{n} additional elements will fit (growing the buffer geometrically if needed).
\}
}\if{html}{\out{</div>}}

}

\emph{Return value}:
Nothing; called for the side effect only.
}
\item{\code{size}}{
Return the capacity (maximum size) of the ring buffer

\emph{Usage:}
\itemize{
\item{env: \code{size()}}
\item{bytes, typed: \code{size(bytes = FALSE)}}
}

\emph{Arguments:}
\itemize{
\item{\code{bytes}:   (for \code{ring_buffer_bytes} only) Logical, indicating if the size should be returned in bytes (rather than logical entries, which is the default).
}
}

\emph{Return value}:
A scalar integer
}
\item{\code{bytes_data}}{
Return the total size of the data storage used in this object.

\emph{Usage:}
\itemize{
\item{env: \emph{(not supported)}}
\item{bytes, typed: \code{bytes_data()}}
}

\emph{Return value}:
A scalar integer
}
\item{\code{stride}}{
Length of each element in the ring buffer, in bytes.  Only implemented (and meaningful) for the bytes buffer; the environment buffer does not support this function as it makes no sense there.

\emph{Usage:}
\itemize{
\item{env: \emph{(not supported)}}
\item{bytes, typed: \code{stride()}}
}

\emph{Return value}:
A scalar integer
}
\item{\code{used}}{
Return the amount of space used in the ring buffer.

\emph{Usage:}
\itemize{
\item{env: \code{used()}}
\item{bytes, typed: \code{used(bytes = FALSE)}}
}

\emph{Arguments:}
\itemize{
\item{\code{bytes}:   (for \code{ring_buffer_bytes} only) Logical, indicating if the size should be returned in bytes (rather than logical entries, which is the default).
}
}

\emph{Return value}:
A scalar integer
}
\item{\code{free}}{
Return the amount of space free in the ring buffer.

\emph{Usage:}
\itemize{
\item{env: \code{free()}}
\item{bytes, typed: \code{free(bytes = FALSE)}}
}

\emph{Arguments:}
\itemize{
\item{\code{bytes}:   (for \code{ring_buffer_bytes} only) Logical, indicating if the size should be returned in bytes (rather than logical entries, which is the default).
}
}

\emph{Return value}:
A scalar integer
}
\item{\code{is_empty}}{
Test if the ring buffer is empty

\emph{Usage:}
\code{is_empty()}

\emph{Return value}:
A scalar logical
}
\item{\code{is_full}}{
Test if the ring buffer is full

\emph{Usage:}
\code{is_full()}

\emph{Return value}:
A scalar logical
}
\item{\code{head_pos}}{
Return the number of entries from the "start" of the ring buffer the head is.  This is mostly useful for debugging.

\emph{Usage:}
\itemize{
\item{env: \code{head_pos()}}
\item{bytes, typed: \code{head_pos(bytes = FALSE)}}
}

\emph{Arguments:}
\itemize{
\item{\code{bytes}:   (for \code{ring_buffer_bytes} only) Logical, indicating if the position should be returned in bytes (rather than logical entries, which is the default).
}
}

\emph{Return value}:
A scalar integer
}
\item{\code{tail_pos}}{
Return the number of entries from the "start" of the ring buffer the tail is.  This is mostly useful for debugging.

\emph{Usage:}
\itemize{
\item{env: \code{tail_pos()}}
\item{bytes, typed: \code{tail_pos(bytes = FALSE)}}
}

\emph{Arguments:}
\itemize{
\item{\code{bytes}:   (for \code{ring_buffer_bytes} only) Logical, indicating if the position should be returned in bytes (rather than logical entries, which is the default).
}
}

\emph{Return value}:
A scalar integer
}
\item{\code{head}}{
Return the contents of the head (the most recently written element in the ring buffer).

\emph{Usage:}
\code{head()}

\emph{Return value}:
It depends a little here.  For \code{ring_buffer_env} this is a single R object.  For \code{ring_buffer_bytes} it is a raw vector, the same length as the stride of the ring buffer.  For \code{ring_buffer_bytes_typed}, a single R object that has been translated from raw.
}
\item{\code{tail}}{
Return the contents of the tail (the least recently written element in the ring buffer).

\emph{Usage:}
\code{tail()}

\emph{Return value}:
As for \code{head}
}
\item{\code{set}}{
Set a number of ring entries to the same value.  The exact behaviour here varies depending on the type of ring buffer.  This function may overflow the ring buffer; in this case the tail will be moved.

\emph{Usage:}
\code{set(data, n)}

\emph{Arguments:}
\itemize{
\item{\code{data}:   The data to set each ring element to.  For an environment buffer, this may be any R object.  For a bytes buffer it may be either a single byte (in which case each ring element will be set to that byte, repeated \code{stride} times), or a raw vector of length \code{stride}.
}

\if{html}{\out{<div class="sourceCode">}}\preformatted{\\item\{\code{n}:   The number of entries to set to \code{data}
\}
}\if{html}{\out{</div>}}

}

\emph{Return value}:
Invisibly returns the number of elements actually written (which may be less than \code{n} if the buffer overflows).  Primarily called for its side effect.
}
\item{\code{push}}{
Push elements onto the ring buffer head.  This may overflow the ring buffer, destroying the oldest elements in the buffer (and moving the position of the tail).

\emph{Usage:}
\itemize{
\item{env: \code{push(data, iterate = TRUE)}}
\item{bytes, typed: \code{push(data)}}
}

\emph{Arguments:}
\itemize{
\item{\code{data}:   Data to push onto the ring buffer.  For \code{ring_buffer_bytes}, this must be a raw vector with a length that is a multiple of the buffer stride.  For \code{ring_buffer_bytes_typed} it must be a vector of the appropriate type.  For \code{ring_buffer_env} it may be an arbitrary R object (but see \code{iterate} .
}

\if{html}{\out{<div class="sourceCode">}}\preformatted{\\item\{\code{iterate}:   For \code{ring_buffer_env} only, changes the behaviour with vectors and lists.  Because each element of a \code{ring_buffer_env} can b an arbitrary R object, for a list \code{x} it is ambiguous if \code{push(x)} should push one object onto the buffer, or \code{length(x)} objects (i.e. equivalent to \code{push(x[[1]])}, \code{push(x[[2]])}, etc.  The \code{iterate} argument switches between interpretations; if \code{TRUE} (the default) the push will iterate over the object using \code{for (el in x)} (with appropriate S3 dispatch).  If \code{iterate = FALSE}, then the entire object is pushed at once, so always updating only by a single element.
\}
}\if{html}{\out{</div>}}

}

\emph{Return value}:
For \code{ring_buffer_bytes}, the data invisibly.  For \code{ring_buffer_bytes} and \code{ring_buffer_bytes_typed}, the position of the head pointer (relative to the beginning of the storage region).
}
\item{\code{take}}{
Destructively take elements from the ring buffer.  This consumes from the tail (the least recently added elements).  It is not possibly to underflow the buffer; if more elements are requested than can be supplied then an error will be thrown and the state of the buffer unmodified.

\emph{Usage:}
\code{take(n)}

\emph{Arguments:}
\itemize{
\item{\code{n}:   The number of elements to take.
}
}

\emph{Return value}:
For \code{ring_buffer_env} a \code{list} of \code{n} elements. For \code{ring_buffer_bytes}, a raw vector of \code{n * stride} bytes.  For \code{ring_buffer_bytes_typed}, an vector of \code{n} elements of the storage mode of the ring.
}
\item{\code{read}}{
Nondestructively read elements from the ring buffer.  This is identical to \code{take} except that the state of the buffer is not modified.

\emph{Usage:}
\code{read(n)}

\emph{Arguments:}
\itemize{
\item{\code{n}:   The number of elements to read.
}
}

\emph{Return value}:
For \code{ring_buffer_env} a \code{list} of \code{n} elements. For \code{ring_buffer_bytes}, a raw vector of \code{n * stride} bytes.  For \code{ring_buffer_bytes_typed}, an vector of \code{n} elements of the storage mode of the ring.
}
\item{\code{copy}}{
Copy from \emph{this} ring buffer into a different ring buffer. This is destructive with respect to both ring buffers; the tail pointer will be moved in this ring buffer as data are taken, and if the destination ring buffer overflows, the tail pointer will be moved too.

\emph{Usage:}
\code{copy(dest, n)}

\emph{Arguments:}
\itemize{
\item{\code{dest}:   The destination ring buffer - will be modified by this call.
}

\if{html}{\out{<div class="sourceCode">}}\preformatted{\\item\{\code{n}:   The number of elements to copy
\}
}\if{html}{\out{</div>}}

}
}
\item{\code{mirror}}{
Mirror the contents of \emph{this} ring buffer into a different ring buffer.  This differs from \code{copy} in that \emph{this} ring buffer is unaffected and in that \emph{all} of this ring buffer is copied over (including head/tail positions).  This provides an alternative way of duplicating state to \code{duplicate} if you already have an appropriately sized ring buffer handy.  No allocations will be done.

\emph{Usage:}
\code{mirror(dest)}

\emph{Arguments:}
\itemize{
\item{\code{dest}:   The destination ring buffer - will be modified by this call.
}
}

\emph{Return value}:
Nothing; called for the side effect only.
}
\item{\code{head_offset}}{
Nondestructively read the contents of the \code{head} of the buffer, offset by \code{n} entries.

\emph{Usage:}
\code{head_offset(n)}

\emph{Arguments:}
\itemize{
\item{\code{n}:   Head offset.  This moves away from the most recently added item. An offset of 0 reads the most recently added element, 1 reads the element added before that.
}
}

\emph{Return value}:
As for \code{head}
}
\item{\code{tail_offset}}{
Nondestructively read the contents of the \code{tail} of the buffer, offset by \code{n} entries.

\emph{Usage:}
\code{tail_offset(n)}

\emph{Arguments:}
\itemize{
\item{\code{n}:   Tail offset.  This moves away from the oldest item.  An offset of 0 reads the oldest element, 1 reads the element added after that.
}
}

\emph{Return value}:
As for \code{tail} (see \code{head})
}
\item{\code{take_head}}{
As for \code{take}, but operating on the head rather than the tail.  This is destructive with respect to the head.

\emph{Usage:}
\code{take_head(n)}

\emph{Arguments:}
\itemize{
\item{\code{n}:   Number of elements to take.
}
}

\emph{Return value}:
As for \code{take}
}
\item{\code{read_head}}{
As for \code{read}, but operating on the head rather than the tail.  This is not destructive with respect to the tail.

\emph{Usage:}
\code{read_head(n)}

\emph{Arguments:}
\itemize{
\item{\code{n}:   Number of elements to read.
}
}

\emph{Return value}:
As for \code{read}
}
\item{\code{head_set}}{
Set data to the head \emph{without advancing}.  This is useful in cases where the head data will be set and advanced separately (with \code{head_advance}).  This is unlikely to be useful for all users.  It is used extensively in dde (but called from C).

\emph{Usage:}
\code{head_set(data)}

\emph{Arguments:}
\itemize{
\item{\code{data}:   Data to set into the head.  For the bytes buffer this must be exactly \code{stride} bytes long, and for the environment buffer it corresponds to a single "element".
}
}

\emph{Return value}:
Nothing; called for the side effect only.
}
\item{\code{head_data}}{
Retrieve the current data stored in the head but not advanced. For many cases this may be junk - if the byte buffer has looped then it will be the bytes that will be overwritten on the next write.  However, when using \code{head_set} it will be the data that have been set into the buffer but not yet committed with \code{head_advance}.

\emph{Usage:}
\code{head_data()}

\emph{Return value}:
As for \code{head}
}
\item{\code{head_advance}}{
Shift the head around one position.  This commits any data written by \code{head_set}.

\emph{Usage:}
\code{head_advance()}

\emph{Return value}:
Nothing; called for the side effect only.
}
}
}

\examples{
# Create a ring buffer of 30 integers:
b <- ring_buffer_bytes_typed(30, integer(1))

# Alternatively you can create the same buffer this way:
b <- ring_buffer_bytes_typed(30, "integer", 1)

# The buffer is empty to start with
b$is_empty()

# Note that the buffer has a stride of 4 (see ?ring_buffer_bytes)
b$stride()

# Push some numbers into the buffer:
b$push(as.integer(1:10))

# Report the number of elements used:
b$used()

# Get the first added element:
b$tail()

# The buffer behaves basically the same way now as
# "ring_buffer_env" but will typecheck all inputs:
\dontrun{
  b$push(pi) # error because not an integer
  b$push(1)  # error because not an integer (you must convert to int)
}

# Recycling: the typed buffer operates by converting the input
# vector to a set of bytes and then pushing them onto the buffer;
# this works so long as the vector of bytes has the correct
# length.
b <- ring_buffer_bytes_typed(30, integer(3))

# These both fail because 2 and 4 do not end up as multiples of 3:
\dontrun{
  b$push(c(1L, 2L))
  b$push(c(1L, 2L, 3L, 4L))
}

# But this is fine:
b$push(seq_len(6))
b$tail()
b$tail_offset(1)
}
\author{
Rich FitzJohn
}
