% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ipcw.R
\name{ipcw}
\alias{ipcw}
\alias{ipcw.none}
\alias{ipcw.marginal}
\alias{ipcw.nonpar}
\alias{ipcw.cox}
\alias{ipcw.aalen}
\title{Estimation of censoring probabilities}
\usage{
ipcw(formula, data, method, args, times, subjectTimes, subjectTimesLag = 1,
  what, keep = NULL)
}
\arguments{
\item{formula}{A survival formula like, \code{Surv(time,status)~1}, where
as usual status=0 means censored. The status variable is internally
reversed for estimation of censoring rather than survival
probabilities. Some of the available models (see argument
\code{model}) will use predictors on the right hand side of the
formula.}

\item{data}{The data used for fitting the censoring model}

\item{method}{Censoring model used for estimation of the
(conditional) censoring distribution.}

\item{args}{A list of arguments which is passed to method}

\item{times}{For \code{what="IPCW.times"} a vector of times at
which to compute the probabilities of not being censored.}

\item{subjectTimes}{For \code{what="IPCW.subjectTimes"} a vector of
individual times at which the probabilities of not being censored
are computed.}

\item{subjectTimesLag}{If equal to \code{1} then obtain
\code{G(T_i-|X_i)}, if equal to \code{0} estimate the conditional
censoring distribution at the subjectTimes,
i.e. (\code{G(T_i|X_i)}).}

\item{what}{Decide about what to do: If equal to
\code{"IPCW.times"} then weights are estimated at given
\code{times}.  If equal to \code{"IPCW.subjectTimes"} then weights
are estimated at individual \code{subjectTimes}.  If missing then
produce both.}

\item{keep}{Which elements to add to the output. Any subset of the vector \code{c("times","fit","call")}.}
}
\value{
A list with elements depending on argument \code{keep}. \item{times}{The times at which weights are estimated}
\item{IPCW.times}{Estimated weights at \code{times}}
\item{IPCW.subjectTimes}{Estimated weights at individual time values
\code{subjectTimes}} \item{fit}{The fitted censoring model}
\item{method}{The method for modelling the censoring distribution}
\item{call}{The call}
}
\description{
This function is used internally to obtain
inverse of the probability of censoring weights.
}
\details{
Inverse of the probability of censoring weights (IPCW) usually refer to the
probabilities of not being censored at certain time points. These
probabilities are also the values of the conditional survival function of
the censoring time given covariates. The function ipcw estimates the
conditional survival function of the censoring times and derives the
weights.

IMPORTANT: the data set should be ordered, \code{order(time,-status)} in
order to get the values \code{IPCW.subjectTimes} in the right order for some
choices of \code{method}.
}
\examples{

library(prodlim)
library(rms)
dat=SimSurv(30)

dat <- dat[order(dat$time),]

# using the marginal Kaplan-Meier for the censoring times

WKM=ipcw(Hist(time,status)~X2,
  data=dat,
  method="marginal",
  times=sort(unique(dat$time)),
  subjectTimes=dat$time,keep=c("fit"))
plot(WKM$fit)
WKM$fit

# using the Cox model for the censoring times given X2
library(survival)
WCox=ipcw(Hist(time=time,event=status)~X2,
  data=dat,
  method="cox",
  times=sort(unique(dat$time)),
  subjectTimes=dat$time,keep=c("fit"))
WCox$fit

plot(WKM$fit)
lines(sort(unique(dat$time)),
      1-WCox$IPCW.times[1,],
      type="l",
      col=2,
      lty=3,
      lwd=3)
lines(sort(unique(dat$time)),
      1-WCox$IPCW.times[5,],
      type="l",
      col=3,
      lty=3,
      lwd=3)

# using the stratified Kaplan-Meier
# for the censoring times given X2

WKM2=ipcw(Hist(time,status)~X2,
  data=dat,
  method="nonpar",
  times=sort(unique(dat$time)),
  subjectTimes=dat$time,keep=c("fit"))
plot(WKM2$fit,add=FALSE)


}
\author{
Thomas A. Gerds \email{tag@biostat.ku.dk}
}
\keyword{survival}
