#! /usr/bin/env Rscript

library(testthat)
library(rly)

context("Lineno and position tracking with error tokens")

Lexer <- R6Class("Lexer",
  public = list(
    tokens = c('NAME','NUMBER', 'PLUS','MINUS','TIMES','DIVIDE','EQUALS', 'LPAREN','RPAREN'),
    t_PLUS = '\\+',
    t_MINUS = '-',
    t_TIMES = '\\*',
    t_DIVIDE = '/',
    t_EQUALS = '=',
    t_LPAREN = '\\(',
    t_RPAREN = '\\)',
    t_NAME = '[a-zA-Z_][a-zA-Z0-9_]*',
    t_NUMBER = function(re='\\d+', t) {
      t$value <- strtoi(t$value)
      return(t)
    },
    t_ignore = " \t",
    t_newline = function(re='\\n+', t) {
      t$lexer$lineno <- t$lexer$lineno + nchar(t$value)
      return(NULL)
    },
    t_error = function(t) {
      cat(sprintf("Illegal character '%s'\n", t$value[1]))
      t$lexer$skip(1)
      return(t)
    }
  )
)

Parser <- R6Class("Parser",
  public = list(
    tokens = Lexer$public_fields$tokens,
    # Parsing rules
    precedence = list(c('left','PLUS','MINUS'),
                      c('left','TIMES','DIVIDE','MINUS'),
                      c('right','UMINUS')),
    # dictionary of names
    names = new.env(hash=TRUE),
    p_statement_assign = function(doc='statement : NAME EQUALS expression', t) {
      names[t[1]] <- t[3]
    },
    p_statement_assign_error = function(doc='statement : NAME EQUALS error', t) {
      line_start <- t$linespan[3][1]
      line_end   <- t$linespan[3][2]
      pos_start  <- t$lexspan[3][1]
      pos_end    <- t$lexspan[3][2]
      
      cat(sprintf("Assignment Error at %d:%d to %d:%d\n", line_start, pos_start, line_end, pos_end))
    },
    p_statement_expr = function(doc='statement : expression', t) {
      cat(t[1])
    },
    p_expression_binop = function(doc='expression : expression PLUS expression
                                     | expression MINUS expression
                                     | expression TIMES expression
                                     | expression DIVIDE expression', t) {
      if(t[2] == '+')      t[0] <- t[1] + t[3]
      else if(t[2] == '-') t[0] <- t[1] - t[3]
      else if(t[2] == '*') t[0] <- t[1] * t[3]
      else if(t[2] == '/') t[0] <- t[1] / t[3]
    },
    p_expression_uminus = function(doc='expression : MINUS expression %prec UMINUS', t) {
      t[0] <- -t[2]
    },
    p_expression_group = function(doc='expression : LPAREN expression RPAREN', t) {
      t[0] <- t[2]
    },
    p_expression_group_error = function(doc='expression : LPAREN error RPAREN', t) {
      line_start <- t$linespan[3][1]
      line_end   <- t$linespan[3][2]
      pos_start  <- t$lexspan[3][1]
      pos_end    <- t$lexspan[3][2]

      cat(sprintf("Syntax error at %d:%d to %d:%d", line_start, pos_start, line_end, pos_end))
      t[0] <- 0
    },
    p_expression_number = function(doc='expression : NUMBER', t) {
      t[0] <- t[1]
    },
    p_expression_name = function(doc='expression : NAME', t) {
      t[0] <- names[t[1]]
    },
    p_error = function(t) {
      cat(sprintf("Syntax error at '%s'", t$value))
    }
  )
)

test_that("error", {
  lexer <- rly::lex(Lexer)
  parser <- rly::yacc(Parser)
  data <- "a = 3 +
(4*5) +
(a b c) +
+ 6 + 7"
#  parser$parse(data, lexer, debug=TRUE)
})
