% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fData.R
\name{cov_fun}
\alias{cov_fun}
\alias{cov_fun.fData}
\alias{cov_fun.mfData}
\title{Covariance function for functional data}
\usage{
cov_fun(X, Y = NULL)

\method{cov_fun}{fData}(X, Y = NULL)

\method{cov_fun}{mfData}(X, Y = NULL)
}
\arguments{
\item{X}{is the (eventually first) functional dataset, i.e. either an object
of class \code{fData} or an object of class \code{mfData};}

\item{Y}{is the (optional) second functional dataset to be used to compute the
cross-covariance function,  either \code{NULL} or an \code{fData} or
\code{mfData} object (see the Value section for details).}
}
\value{
The following cases are given:

\itemize{
\item{if \code{X} is of class \code{fData} and \code{Y} is \code{NULL}, then
the covariance function of \code{X} is returned;}
\item{if \code{X} is of class \code{fData} and \code{Y} is of
class \code{fData},
the cross-covariance function of the two datasets is returned;}
\item{if \code{X} is of class \code{mfData} and \code{Y} is \code{NULL},
the upper-triangular blocks of the covariance function of \code{X}
are returned (in form of list and by row, i.e. in the squence 1_1, 1_2, ...,
1_L, 2_2, ... - have a look at the labels of the list with \code{str});}
\item{if \code{X} is of class \code{mfData} and \code{Y} is of
class \code{fData},
the cross-covariances of \code{X}'s components and \code{Y} are
returned (in form of list);}
\item{if \code{X} is of class \code{mfData} and \code{Y} is of
class \code{mfData},
the upper-triangular blocks of the cross-covariance of \code{X}'s and
\code{Y}'s components are returned (in form of list and by row, i.e. in the
squence 1_1, 1_2, ..., 1_L, 2_2, ... - have a look at the labels
of the list with \code{str}));}}

In any case, the return type is either an instance of the \code{S3} class \code{Cov}
or a list of instances of such class (for the case of multivariate
functional data).
}
\description{
\code{S3} method to compute the sample covariance and cross-covariance
functions for a set of functional data.
}
\details{
Given a univariate random function X, defined
over the grid \eqn{I = [a,b]}, the covariance
function is defined as:

\deqn{C(s,t) = Cov( X(s), X(t) ), \qquad s,t \in I.}

Given another random function, Y, defined over the same grid as X, the cross-
covariance function of X and Y is:

\deqn{C^{X,Y}( s,t ) =  Cov( X(s), Y(t) ), \qquad s, t \in I.}

For a generic L-dimensional random function X, i.e. an L-dimensional
multivariate functional datum, the covariance function is defined as the set
of blocks:

\deqn{C_{i,j}(s,t) = Cov( X_i(s), X_j(t)), i,j = 1, ...,L, s,t \in I,}

while the cross-covariance function is defined by the blocks:

\deqn{C^{X,Y}_{i,j}(s,t) = Cov( X_i(s), Y_j(t))}



The method \code{cov_fun} provides the sample estimator of the covariance or
cross-covariance functions for univariate or multivariate functional datasets.

The class of \code{X} (\code{fData} or \code{mfData}) is used to dispatch the
correct implementation of the method.
}
\examples{

# Generating a univariate functional dataset
N = 1e2

P = 1e2
t0 = 0
t1 = 1

time_grid = seq( t0, t1, length.out = P )

Cov = exp_cov_function( time_grid, alpha = 0.3, beta = 0.4 )

D1 = generate_gauss_fdata( N, centerline = sin( 2 * pi * time_grid ), Cov = Cov )
D2 = generate_gauss_fdata( N, centerline = sin( 2 * pi * time_grid ), Cov = Cov )

fD1 = fData( time_grid, D1 )
fD2 = fData( time_grid, D2 )

# Computing the covariance function of fD1

C = cov_fun( fD1 )
str( C )

# Computing the cross-covariance function of fD1 and fD2
CC = cov_fun( fD1, fD2 )
str( CC )

# Generating a multivariate functional dataset
L = 3

C1 = exp_cov_function( time_grid, alpha = 0.1, beta = 0.2 )
C2 = exp_cov_function( time_grid, alpha = 0.2, beta = 0.5 )
C3 = exp_cov_function( time_grid, alpha = 0.3, beta = 1 )

centerline = matrix( c( sin( 2 * pi * time_grid ),
                        sqrt( time_grid ),
                        10 * ( time_grid - 0.5 ) * time_grid ),
                     nrow = 3, byrow = TRUE )

D3 = generate_gauss_mfdata( N, L, centerline,
                       correlations = c( 0.5, 0.5, 0.5 ),
                       listCov = list( C1, C2, C3 ) )

# adding names for better readability of BC3's labels
names( D3 ) = c( 'comp1', 'comp2', 'comp3' )
mfD3 = mfData( time_grid, D3 )

D1 = generate_gauss_fdata( N, centerline = sin( 2 * pi * time_grid ),
                              Cov = Cov )
fD1 = fData( time_grid, D1 )

# Computing the block covariance function of mfD3
BC3 = cov_fun( mfD3 )
str( BC3 )

# computing cross-covariance between mfData and fData objects
CC = cov_fun( mfD3, fD1 )
str( CC )

}
\seealso{
\code{\link{fData}}, \code{\link{mfData}}, \code{\link{plot.Cov}}
}
