% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{rrum_helper}
\alias{rrum_helper}
\title{Gibbs sampler to estimate the rRUM}
\usage{
rrum_helper(Y, Q, delta0, chain_length = 10000L, as = 1, bs = 1,
  ag = 1, bg = 1)
}
\arguments{
\item{Y}{A \code{matrix} with \eqn{N} rows and \eqn{J} columns
indicating the indviduals' responses to each of the
items.}

\item{Q}{A \code{matrix} with \eqn{J} rows and \eqn{K} columns
indicating which attributes are required to answer each
of the items.An entry of 1 indicates attribute \eqn{k}
is required to answer item \eqn{j}. An entry of one
indicates attribute \eqn{k} is not required.}

\item{chain_length}{A \code{numeric} indicating the number of iterations of
Gibbs sampler to be run.  Default is set to 10000.}

\item{as}{A \code{numeric}, parameter for the prior distribution of
\code{pistar}.  High values as encourage higher values of
\code{pistar} and lower values of \code{rstar}.}

\item{bs}{A \code{numeric}, parameter for the prior distribution of
\code{pistar}.  High values as encourage lower values of
\code{pistar} and higher values of \code{rstar}.}

\item{ag}{A \code{numeric}, parameter for the prior distribution of
\code{rstar}.  High values as encourage higher values of
\code{rstar}.}

\item{bg}{A \code{numeric}, parameter for the prior distribution of
\code{pistar}.  High values as encourage lower values of
\code{rstar}.}

\item{deltas}{A \code{vector}, parameters for the Dirichlet prior on \code{pi}.}
}
\value{
A \code{List}
\itemize{
\item \code{PISTAR} A \code{matrix} where each column represents one draw from the
posterior distribution of pistar.
\item \code{RSTAR} A \eqn{J x K x chain_length} \code{array} where \code{J} reperesents the
number of items, and \code{K} represents the number of attributes.
Each slice represents one draw from the posterior distribution
of \code{rstar}.
\item \code{PI} \code{matrix} where each column reperesents one draw from the posterior
distribution of \code{pi}.
\item \code{ALPHA} An \eqn{N x K x chain_length} \code{array} where \code{N} reperesents the
number of individuals, and \code{K} represents the number of
attributes. Each slice represents one draw from the posterior
distribution of \code{alpha}.
}
}
\description{
Obtains samples from posterior distributon for the reduced Reparametrized
Unified Model (rRUM).
}
\examples{
# Set seed for reproducibility
set.seed(217)

## Define Simulation Parameters

N = 1000 # Number of Individuals
J = 6    # Number of Items
K = 2    # Number of Attributes

# Matrix where rows represent attribute classes
As = attribute_classes(K) 

# Latent Class probabilities
pis = c(.1, .2, .3, .4) 

# Q Matrix
Q = rbind(c(1, 0),
          c(0, 1),
          c(1, 0),
          c(0, 1),
          c(1, 1),
          c(1, 1)
    )
    
# The probabiliies of answering each item correctly for individuals 
# who do not lack any required attribute
pistar = rep(.9, J)

# Penalties for failing to have each of the required attributes
rstar  = .5 * Q

# Randomized alpha profiles
alpha  = As[sample(1:(K ^ 2), N, replace = TRUE, pis),]

# Simulate data
rrum_items = simcdm::sim_rrum_items(Q, rstar, pistar, alpha)

\dontrun{
# Note: This portion of the code is computationally intensive.

# Recover simulation parameters with Gibbs Sampler
Gibbs.out = rrum(rrum_items, Q)

# Iterations to be discarded from chain as burnin
burnin = 1:5000 

# Calculate summarizes of posterior distributions
rstar.mean  = with(Gibbs.out, apply(RSTAR[,,-burnin], c(1, 2), mean))
pistar.mean = with(Gibbs.out, apply(PISTAR[,-burnin], 1, mean))
pis.mean    = with(Gibbs.out, apply(PI[,-burnin], 1 ,mean))
}
}
\references{
Culpepper, S. A. & Hudson, A. (In Press). An improved strategy for Bayesian
estimation of the reduced reparameterized unified model. Applied
Psychological Measurement.

Hudson, A., Culpepper, S. A., & Douglas, J. (2016, July). Bayesian estimation
of the generalized NIDA model with Gibbs sampling. Paper presented at the
annual International Meeting of the Psychometric Society, Asheville, North
Carolina.
}
\author{
Steven Andrew Culpepper, Aaron Hudson, and James Joseph Balamuta
}
\keyword{internal}
