% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rasterize.R
\name{rasterize}
\alias{rasterize}
\alias{rasterize,matrix-method}
\alias{rasterize,RasterLayer-method}
\alias{rasterize,SpatRaster-method}
\title{Rasterize input}
\usage{
rasterize(x)

\S4method{rasterize}{matrix}(x)

\S4method{rasterize}{RasterLayer}(x)

\S4method{rasterize}{SpatRaster}(x)
}
\arguments{
\item{x}{A matrix, RasterLayer, or SpatRaster}
}
\value{
A SpatRaster object
}
\description{
Convert input to a SpatRaster object
}
\details{
This function is primarily used to convert an input matrix or raster to a SpatRaster
object. The main thing it is useful for is setting a standard extent and CRS for
converting matrices. It is used internally by the package to ensure consistent
results for the different data types for maps.

When converting matrices, the extents are set to match the number of rows and
columns of the matrix. Pixels in the result are centered on whole number coordinates
with (1,1) corresponding to the bottom left pixel. The CRS is set to "local", which
treats it as Euclidean (Cartesian) plane with the units in meters.

The main benefit will be for users that want an easy way to plot matrix data.
If the input type to the \code{\link{samc}} function is matrices, then the output
of \code{\link{map}} will also be matrices. Plotting these matrices can require
more work than simply using SpatRaster objects for \code{\link{samc}} and getting
SpatRaster results back from \code{\link{map}}.

The raster and terra packages both also have a rasterize function that serves
a different purpose. If either of these packages are used directly, then the order of
package loading becomes very important because it will determine which version
of rasterize is used by default.
}
\examples{
# "Load" the data. In this case we are using data built into the package.
# In practice, users will likely load raster data using the raster() function
# from the raster package.
res_data <- samc::example_split_corridor$res
abs_data <- samc::example_split_corridor$abs
init_data <- samc::example_split_corridor$init


# Make sure our data meets the basic input requirements of the package using
# the check() function.
check(res_data, abs_data)
check(res_data, init_data)

# Setup the details for a random-walk model
rw_model <- list(fun = function(x) 1/mean(x), # Function for calculating transition probabilities
                 dir = 8, # Directions of the transitions. Either 4 or 8.
                 sym = TRUE) # Is the function symmetric?


# Create a `samc-class` object with the resistance and absorption data using
# the samc() function. We use the recipricol of the arithmetic mean for
# calculating the transition matrix. Note, the input data here are matrices,
# not RasterLayers.
samc_obj <- samc(res_data, abs_data, model = rw_model)


# Convert the initial state data to probabilities
init_prob_data <- init_data / sum(init_data, na.rm = TRUE)


# Calculate short- and long-term metrics using the analytical functions
short_mort <- mortality(samc_obj, init_prob_data, time = 50)
short_dist <- distribution(samc_obj, origin = 3, time = 50)
long_disp <- dispersal(samc_obj, init_prob_data)
visit <- visitation(samc_obj, dest = 4)
surv <- survival(samc_obj)


# Use the map() function to turn vector results into RasterLayer objects.
short_mort_map <- map(samc_obj, short_mort)
short_dist_map <- map(samc_obj, short_dist)
long_disp_map <- map(samc_obj, long_disp)
visit_map <- map(samc_obj, visit)
surv_map <- map(samc_obj, surv)
}
