% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scf_MIcombine.R
\name{scf_MIcombine}
\alias{scf_MIcombine}
\title{Combine Estimates Across SCF Implicates Using Rubin's Rules}
\usage{
scf_MIcombine(results, variances, call = sys.call(), df.complete = Inf)
}
\arguments{
\item{results}{A list of implicate-level model outputs. Each element must be a named numeric vector
or an object with methods for \code{coef()} and \code{vcov()}. Typically generated internally by modeling functions.}

\item{variances}{Optional list of variance-covariance matrices. If omitted, extracted using \code{vcov()}.}

\item{call}{Optional. The originating function call. Defaults to \code{sys.call()}.}

\item{df.complete}{Optional degrees of freedom for the complete-data model. Used for small-sample
corrections. Defaults to \code{Inf}, assuming large-sample asymptotics.}
}
\value{
An object of class \code{"scf_MIresult"} with components:
\describe{
\item{coefficients}{Pooled point estimates across implicates.}
\item{variance}{Pooled variance-covariance matrix.}
\item{df}{Degrees of freedom for each parameter, adjusted using Barnard-Rubin formula.}
\item{missinfo}{Estimated fraction of missing information for each parameter.}
\item{nimp}{Number of implicates used in pooling.}
\item{call}{Function call recorded for reproducibility.}
}

Supports \code{coef()}, \code{\link[=SE]{SE()}}, \code{confint()}, and \code{summary()} methods.
}
\description{
This function is the \strong{canonical implementation} of Rubin’s Rules in the
\code{scf} package. It defines how point estimates, standard errors, and degrees
of freedom are pooled across the SCF’s multiply-imputed replicate-weighted
implicates.
}
\details{
Most \code{scf} functions that compute descriptive statistics or model estimates
internally call this function (or apply its logic). As such, this
documentation serves as the authoritative explanation of the pooling
procedure and its assumptions for all SCF workflows in the package.
}
\section{Implementation}{

\code{scf_MIcombine()} pools a set of implicate-level estimates and their
associated variance-covariance matrices using Rubin’s Rules.

This includes:
\itemize{
\item Calculation of pooled point estimates
\item Total variance from within- and between-imputation components
\item Degrees of freedom via Barnard-Rubin method
\item Fraction of missing information
}

Inputs are typically produced by functions like \code{scf_mean()}, \code{scf_ols()},
or \code{scf_percentile()}.

This function is primarily used internally, but may be called directly by
advanced users constructing custom estimation routines from implicate-level
results.
}

\section{Details}{

The SCF provides five implicates per survey wave, each a plausible version
of the population under a specific missing-data model. Analysts conduct the
same statistical procedure on each implicate, producing a set of five
estimates \eqn{ Q_1, Q_2, ..., Q_5 }. These are then combined using Rubin’s
Rules, a procedure to combine results across these implicates with an
attempt to account for:
\itemize{
\item \strong{Within-imputation variance}: Uncertainty from complex sample design
\item \strong{Between-imputation variance}: Uncertainty due to missing data
}

For a scalar quantity \eqn{ Q }, the pooled estimate and
total variance are calculated as:

\deqn{ \bar{Q} = \frac{1}{M} \sum Q_m }
\deqn{ \bar{U} = \frac{1}{M} \sum U_m }
\deqn{ B = \frac{1}{M - 1} \sum (Q_m - \bar{Q})^2 }
\deqn{ T = \bar{U} + \left(1 + \frac{1}{M} \right) B }

Where:
\itemize{
\item \eqn{ M } is the number of implicates (typically 5 for SCF)
\item \eqn{ Q_m } is the estimate from implicate \eqn{ m }
\item \eqn{ U_m } is the sampling variance of \eqn{ Q_m }, accounting for replicate weights and design
}

The total variance \eqn{ T } reflects both within-imputation uncertainty (sampling error)
and between-imputation uncertainty (missing-data imputation).

The standard error of the pooled estimate is \eqn{ \sqrt{T} }. Degrees of freedom are
adjusted using the Barnard-Rubin method:

\deqn{ \nu = (M - 1) \left(1 + \frac{\bar{U}}{(1 + \frac{1}{M}) B} \right)^2 }

The fraction of missing information (FMI) is also reported:
it reflects the proportion of total variance attributable to imputation uncertainty.

See \code{\link[=scf_MIcombine]{scf_MIcombine()}} for full implementation details.
}

\examples{
# Do not implement these lines in real analysis:
# Use functions `scf_download()` and `scf_load()`
td  <- tempdir()
src <- system.file("extdata", "scf2022_mock_raw.rds", package = "scf")
file.copy(src, file.path(td, "scf2022.rds"), overwrite = TRUE)
scf2022 <- scf_load(2022, data_directory = td)

# Example for real analysis: Pool simple survey mean for mock data
outlist <- lapply(scf2022$mi_design, function(d) survey::svymean(~I(age >= 65), d))
pooled  <- scf_MIcombine(outlist)     # vcov/coef extracted automatically
SE(pooled); coef(pooled)

unlink("scf2022.rds", force = TRUE)

}
\references{
Barnard J, Rubin DB. Small-sample degrees of freedom with multiple imputation.
\doi{10.1093/biomet/86.4.948}.

Little RJA, Rubin DB. Statistical analysis with missing data.
ISBN: 9780470526798.

U.S. Federal Reserve. Codebook for 2022 Survey of Consumer Finances.
https://www.federalreserve.gov/econres/scfindex.htm
}
\seealso{
\code{\link[=scf_mean]{scf_mean()}}, \code{\link[=scf_ols]{scf_ols()}}, \code{\link[=scf_glm]{scf_glm()}}, \code{\link[=scf_logit]{scf_logit()}}
}
