% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/customlikelihood.R
\name{customll}
\alias{customll}
\alias{customll_test}
\title{Compile a custom log-likelihood function.}
\usage{
customll(
  code,
  rebuild = FALSE,
  includes = character(),
  cacheDir = getOption("rcpp.cache.dir", tempdir()),
  showOutput = verbose,
  verbose = getOption("verbose")
)

customll_test()
}
\arguments{
\item{code}{\verb{C++} code for a log-likelihood function (with normalising constant omitted if desired). See details for more.}

\item{rebuild}{passed to \code{\link[Rcpp:cppFunction]{Rcpp::cppFunction()}}.}

\item{includes}{passed to \code{\link[Rcpp:cppFunction]{Rcpp::cppFunction()}}. For internal use.}

\item{cacheDir}{passed to \code{\link[Rcpp:cppFunction]{Rcpp::cppFunction()}}.}

\item{showOutput}{passed to \code{\link[Rcpp:cppFunction]{Rcpp::cppFunction()}}.}

\item{verbose}{passed to \code{\link[Rcpp:cppFunction]{Rcpp::cppFunction()}}.}
}
\value{
\code{customll()} returns an \code{adloglikelood} object (which is just an \code{externalptr} with attributes) for the compiled log-likelihood function. The returned object has an attribute \code{fname}.

\code{customll_test()} returns \code{TRUE} if taping works in your \code{R} session. Otherwise it will return \code{FALSE} and generate warnings.
}
\description{
Supply \verb{C++} code to specify a custom log-likelihood, much like \code{TMB::compile()} is passed \verb{C++} code that formulate models.
For score matching the normalising constant of the log-likelihood can be omitted.
Taping of the function currently only works with the \code{gcc} compiler, typically on Linux; taping works if \code{customll_test()} returns \code{TRUE}.
Packages \code{RcppEigen} and \code{RcppXPtrUtils} must be installed.
}
\details{
The function uses \code{\link[RcppXPtrUtils:cppXPtr]{RcppXPtrUtils::cppXPtr()}} and \code{\link[Rcpp:cppFunction]{Rcpp::cppFunction()}}.
It is good practice to check the returned object using \code{\link[=evalll]{evalll()}}.
The first compilation in a session can be very slow.
}
\section{Code Argument}{
\code{code} must be \verb{C++} that uses only \code{CppAD} and \code{Eigen}, which makes it very similar to the requirements of the input to \code{TMB::compile()} (which also uses \code{CppAD} and \code{Eigen}).

The start of \code{code} should always be "\verb{a1type fname(const veca1 &x, const veca1 &theta)\{}" where \code{fname} is your chosen name of the log-likelihood function, \code{x} represents a point in the data space and \code{theta} is a vector of parameters for the log-likelihood. This specifies that the function will have two vector arguments (of type \code{veca1}) and will return a single numeric value (\code{a1type}).

The type \code{a1type} is a double with special ability for being taped by \code{CppAD}. The \code{veca1} type is a vector of \code{a1type} elements, with the vector wrapping supplied by the \code{Eigen} C++ package (that is an \code{Eigen} matrix with 1 column and dynamic number of rows).

The body of the function must use operations from Eigen and/or CppAD, prefixed by \verb{Eigen::} and \verb{CppAD::} respectively.
There are no easy instructions for writing these as it is genuine \verb{C++} code, which can be very opaque to those unfamiliar with \verb{C++}.
See the \href{https://eigen.tuxfamily.org/dox/group__QuickRefPage.html}{Eigen documentation} for quick reference to available operations from Eigen. Limited operations are available directly from \code{CppAD} without \code{Eigen}: \href{https://cppad.readthedocs.io/latest/unary_standard_math.html}{unary operations} and \href{https://cppad.readthedocs.io/latest/binary_math.html}{binary operations}.
For the purposes of score matching the operations should all be smooth to create a smooth log-likelihood and the normalising constant may be omitted.
}

\examples{
\dontrun{customll_test()

myll <- customll("a1type dirichlet(const veca1 &u, const veca1 &beta) {
  size_t d  = u.size();
  a1type y(0.);  // initialize summation at 0
  for(size_t i = 0; i < d; i++)
  {   y   += beta[i] * log(u[i]);
  }
  return y;
}")
evalll(myll, rep(1/3, 3), rep(-0.5, 3))

tapes <- buildsmdtape("sim", "identity", "sim", 
 myll, rep(1/3, 3), rep(NA, 3), 
 bdryw="minsq", acut = 0.01)
evaltape(tapes$lltape, rep(1/3, 3), rep(-0.5, 3))}

}
