% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/score.R
\name{score}
\alias{score}
\title{Evaluate forecasts}
\usage{
score(data, metrics = NULL, ...)
}
\arguments{
\item{data}{A data.frame or data.table with the predictions and observations.
For scoring using \code{\link[=score]{score()}}, the following columns need to be present:
\itemize{
\item \code{true_value} - the true observed values
\item \code{prediction} - predictions or predictive samples for one
true value. (You only don't need to provide a prediction column if
you want to score quantile forecasts in a wide range format.)}
For scoring integer and continuous forecasts a \code{sample} column is needed:
\itemize{
\item \code{sample} - an index to identify the predictive samples in the
prediction column generated by one model for one true value. Only
necessary for continuous and integer forecasts, not for
binary predictions.}
For scoring predictions in a quantile-format forecast you should provide
a column called \code{quantile}:
\itemize{
\item \code{quantile}: quantile to which the prediction corresponds
}

In addition a \code{model} column is suggested and if not present this will be
flagged and added to the input data with all forecasts assigned as an
"unspecified model").

You can check the format of your data using \code{\link[=check_forecasts]{check_forecasts()}} and there
are examples for each format (\link{example_quantile}, \link{example_continuous},
\link{example_integer}, and \link{example_binary}).}

\item{metrics}{the metrics you want to have in the output. If \code{NULL} (the
default), all available metrics will be computed. For a list of available
metrics see \code{\link[=available_metrics]{available_metrics()}}, or  check the \link{metrics} data set.}

\item{...}{additional parameters passed down to \code{\link[=score_quantile]{score_quantile()}} (internal
function used for scoring forecasts in a quantile-based format).}
}
\value{
A data.table with unsummarised scores. There will be one score per
quantile or sample, which is usually not desired, so you should almost
always run \code{\link[=summarise_scores]{summarise_scores()}} on the unsummarised scores.
}
\description{
This function allows automatic scoring of forecasts using a
range of metrics. For most users it will be the workhorse for
scoring forecasts as it wraps the lower level functions package functions.
However, these functions are also available if you wish to make use of them
independently.

A range of forecasts formats are supported, including quantile-based,
sample-based, binary forecasts. Prior to scoring, users may wish to make use
of \code{\link[=check_forecasts]{check_forecasts()}} to ensure that the input data is in a supported
format though this will also be run internally by \code{\link[=score]{score()}}. Examples for
each format are also provided (see the documentation for \code{data} below or in
\code{\link[=check_forecasts]{check_forecasts()}}).

Each format has a set of required columns (see below). Additional columns may
be present to indicate a grouping of forecasts. For example, we could have
forecasts made by different models in various locations at different time
points, each for several weeks into the future. It is important, that there
are only columns present which are relevant in order to group forecasts.
A combination of different columns should uniquely define the
\emph{unit of a single forecast}, meaning that a single forecast is defined by the
values in the other columns. Adding additional unrelated columns may alter
results.

To obtain a quick overview of the currently supported evaluation metrics,
have a look at the \link{metrics} data included in the package. The column
\code{metrics$Name} gives an overview of all available metric names that can be
computed. If interested in an unsupported metric please open a \href{https://github.com/epiforecasts/scoringutils/issues}{feature request} or consider
contributing a pull request.

For additional help and examples, check out the \href{https://epiforecasts.io/scoringutils/articles/getting-started.html}{Getting Started Vignette}.
}
\examples{
library(magrittr) # pipe operator
data.table::setDTthreads(1) # only needed to avoid issues on CRAN

check_forecasts(example_quantile)
score(example_quantile) \%>\%
  add_coverage(by = c("model", "target_type")) \%>\%
  summarise_scores(by = c("model", "target_type"))

# forecast formats with different metrics
\dontrun{
score(example_binary)
score(example_quantile)
score(example_integer)
score(example_continuous)
}

# score point forecasts (marked by 'NA' in the quantile column)
score(example_point) \%>\%
  summarise_scores(by = "model", na.rm = TRUE)

}
\references{
Funk S, Camacho A, Kucharski AJ, Lowe R, Eggo RM, Edmunds WJ
(2019) Assessing the performance of real-time epidemic forecasts: A
case study of Ebola in the Western Area region of Sierra Leone, 2014-15.
PLoS Comput Biol 15(2): e1006785. \doi{10.1371/journal.pcbi.1006785}
}
\author{
Nikos Bosse \email{nikosbosse@gmail.com}
}
