% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convenienceFunctions.R
\name{semPower.powerPath}
\alias{semPower.powerPath}
\title{semPower.powerPath}
\usage{
semPower.powerPath(
  type,
  comparison = "restricted",
  Beta,
  Psi = NULL,
  nullEffect = "beta = 0",
  nullWhich = NULL,
  nullWhichGroups = NULL,
  standardized = TRUE,
  ...
)
}
\arguments{
\item{type}{type of power analysis, one of \code{'a-priori'}, \code{'post-hoc'}, \code{'compromise'}.}

\item{comparison}{comparison model, one of \code{'saturated'} or \code{'restricted'} (the default). This determines the df for power analyses. \code{'saturated'} provides power to reject the model when compared to the saturated model, so the df equal the one of the hypothesized model. \code{'restricted'} provides power to reject the hypothesized model when compared to an otherwise identical model that just omits the restrictions defined in \code{nullEffect}, so the df equal the number of restrictions.}

\item{Beta}{matrix of regression slopes between latent variables (all-Y notation). A list for multiple group models. Exogenous variables must occupy the first rows in \code{Beta} when \code{standardized = TRUE}. See details.}

\item{Psi}{variance-covariance matrix of latent (residual) factors. If \code{standardized = TRUE}, the diagonal is ignored and all off-diagonal elements are treated as correlations. If \code{NULL}, an identity matrix is assumed. A list for multiple group models. See details.}

\item{nullEffect}{defines the hypothesis of interest, must be one of \code{'beta = 0'} (the default) to test whether a regression slope is zero, \code{'betaX = betaZ'} to test for the equality of slopes, and \code{'betaX = betaZ'} to test for the equality of a slope across groups. Define the slopes to be set to equality in \code{nullWhich} and the groups in \code{nullWhichGroups}.}

\item{nullWhich}{vector of size 2 indicating which slope in \code{Beta} is hypothesized to equal zero when \code{nullEffect = 'beta = 0'}, or to restrict to equality across groups when \code{nullEffect = 'betaA = betaB'}, or list of vectors defining which correlations to restrict to equality when \code{nullEffect = 'betaX = betaZ'}. Can also contain more than two slopes, e.g., \code{list(c(2, 1), c(3, 1), c(3, 2))} to set \code{Beta[2, 1] = Beta[3, 1] = Beta[3, 2]}.}

\item{nullWhichGroups}{for \code{nullEffect = 'betaA = betaB'}, vector indicating the groups for which equality constrains should be applied, e.g. \code{c(1, 3)} to constrain the relevant parameters of the first and the third group. If \code{NULL}, all groups are constrained to equality.}

\item{standardized}{whether all parameters should be standardized (\code{TRUE}, the default). If \code{FALSE}, all regression relations are unstandardized.}

\item{...}{mandatory further parameters related to the specific type of power analysis requested, see \code{\link[=semPower.aPriori]{semPower.aPriori()}}, \code{\link[=semPower.postHoc]{semPower.postHoc()}}, and \code{\link[=semPower.compromise]{semPower.compromise()}}, and parameters specifying the factor model. See details.}
}
\value{
a list. Use the \code{summary} method to obtain formatted results. Beyond the results of the power analysis and a number of effect size measures, the list contains the following components:
\item{\code{Sigma}}{the population covariance matrix. A list for multiple group models.}
\item{\code{mu}}{the population mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{SigmaHat}}{the H0 model implied covariance matrix. A list for multiple group models.}
\item{\code{muHat}}{the H0 model implied mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{modelH0}}{\code{lavaan} H0 model string.}
\item{\code{modelH1}}{\code{lavaan} H1 model string or \code{NULL} when the comparison refers to the saturated model.}
\item{\code{simRes}}{detailed simulation results when a simulated power analysis (\code{simulatedPower = TRUE}) was performed.}
}
\description{
Convenience function for performing power analyses for hypothesis arising
in a generic path model.
This requires the lavaan package.
}
\details{
This function performs a power analysis to reject a hypothesis arising
in a generic structural equation model specifying regression relations between the factors via the Beta matrix:
\itemize{
\item \code{nullEffect = 'beta = 0'}: Tests the hypothesis that a slope is zero.
\item \code{nullEffect = 'betaX = betaZ'}: Tests the hypothesis that two or more slopes are equal to each other.
\item \code{nullEffect = 'betaA = betaB'}: Tests the hypothesis that a slope is equal in two or more groups (always assuming metric invariance).
}

This function provides a generic way to perform power analyses (as compared to other functions covering special cases in a more accessible manner).

A specific hypothesis is defined by setting \code{nullEffect} to define the hypothesis type,
\code{nullWhich} to define the slope(s) that are targeted, and by providing
the \code{Beta} (and optionally the \code{Psi}) matrix to define the population structure.

To understand the structure of \code{Beta} and \code{Psi}, consider the general structural equation model,
\deqn{\Sigma = \Lambda (I - B)^{-1} \Psi [(I - B)^{-1}]'  \Lambda' + \Theta }
where \eqn{B} is the \eqn{m \cdot m} matrix containing the regression slopes and \eqn{\Psi} is the (residual) variance-covariance matrix of the \eqn{m} factors.

As an example, suppose there are four factors (X1, X2, X3, X4), and Beta is defined as follows:

\eqn{
\begin{array}{lrrrr} 
    & X_1 & X_2 & X_3 & X_4\\ 
X_1 & 0.0 & 0.0 & 0.0 & 0.0 \\ 
X_2 & 0.0 & 0.0 & 0.0 & 0.0  \\ 
X_3 & 0.2 & 0.3 & 0.0 & 0.0  \\ 
X_4 & 0.3 & 0.5 & 0.0 & 0.0  \\ 
\end{array}
}

Each row specifies how a particular factor is predicted by the available factors,
so the above implies the following regression relations:

\eqn{
X_1 = 0.0 \cdot X_1 +  0.0 \cdot X_2 + 0.0 \cdot X_3 + 0.0 \cdot X_4 \\
X_2 = 0.0 \cdot X_1 +  0.0 \cdot X_2 + 0.0 \cdot X_3 + 0.0 \cdot X_4 \\
X_3 = 0.2 \cdot X_1 +  0.3 \cdot X_2 + 0.0 \cdot X_3 + 0.0 \cdot X_4 \\
X_4 = 0.3 \cdot X_1 +  0.5 \cdot X_2 + 0.0 \cdot X_3 + 0.0 \cdot X_4 
}

which simplifies to

\eqn{
X_3 = 0.2 \cdot X_1 + 0.3 \cdot X_2 \\
X_4 = 0.3 \cdot X_1 + 0.5 \cdot X_2 
}

Further suppose that Psi is

\eqn{
\begin{array}{lrrrr} 
    & X_1 & X_2 & X_3 & X_4\\ 
X_1 & 1.0 & 0.3 & 0.0 & 0.0 \\ 
X_2 & 0.3 & 1.0 & 0.0 & 0.0 \\ 
X_3 & 0.0 & 0.0 & 1.0 & 0.2 \\ 
X_4 & 0.0 & 0.0 & 0.2 & 1.0 \\ 
\end{array}
}

which implies a correlation between X1 and X2 of .3 and a residual correlation
between X3 and X4 of .2.

Beyond the arguments explicitly contained in the function call, additional arguments
are required specifying the factor model and the requested type of power analysis.

Additional arguments related to the \strong{definition of the factor model}:
\itemize{
\item \code{Lambda}: The factor loading matrix (with the number of columns equaling the number of factors).
\item \code{loadings}: Can be used instead of \code{Lambda}: Defines the primary loadings for each factor in a list structure, e. g. \code{loadings = list(c(.5, .4, .6), c(.8, .6, .6, .4))} defines a two factor model with three indicators loading on the first factor by .5, , 4., and .6, and four indicators loading on the second factor by .8, .6, .6, and .4.
\item \code{nIndicator}: Can be used instead of \code{Lambda}: Used in conjunction with \code{loadM}. Defines the number of indicators by factor, e. g., \code{nIndicator = c(3, 4)} defines a two factor model with three and four indicators for the first and second factor, respectively. \code{nIndicator} can also be a single number to define the same number of indicators for each factor.
\item \code{loadM}: Can be used instead of \code{Lambda}: Used in conjunction with \code{nIndicator}. Defines the loading either for all indicators (if a single number is provided) or separately for each factor (if a vector is provided), e. g. \code{loadM = c(.5, .6)} defines the loadings of the first factor to equal .5 and those of the second factor do equal .6.
}

So either \code{Lambda}, or \code{loadings}, or \code{nIndicator} and \code{loadM} always need to be defined.

Additional arguments related to the requested type of \strong{power analysis}:
\itemize{
\item \code{alpha}: The alpha error probability. Required for \code{type = 'a-priori'} and \code{type = 'post-hoc'}.
\item Either \code{beta} or \code{power}: The beta error probability and the statistical power (1 - beta), respectively. Only for \code{type = 'a-priori'}.
\item \code{N}: The sample size. Always required for \code{type = 'post-hoc'} and \code{type = 'compromise'}. For \code{type = 'a-priori'} and multiple group analysis, \code{N} is a list of group weights.
\item \code{abratio}: The ratio of alpha to beta. Only for \code{type = 'compromise'}.
}

If a \strong{simulated power analysis} (\code{simulatedPower = TRUE}) is requested, optional arguments can be provided as a list to \code{simOptions}:
\itemize{
\item \code{nReplications}: The targeted number of simulation runs. Defaults to 250, but larger numbers greatly improve accuracy at the expense of increased computation time.
\item \code{minConvergenceRate}:  The minimum convergence rate required, defaults to .5. The maximum actual simulation runs are increased by a factor of 1/minConvergenceRate.
\item \code{type}: specifies whether the data should be generated from a population assuming multivariate normality (\code{'normal'}; the default), or based on an approach generating non-normal data (\code{'IG'}, \code{'mnonr'}, \code{'RC'}, or \code{'VM'}).
The approaches generating non-normal data require additional arguments detailed below.
\item \code{missingVars}: vector specifying the variables containing missing data (defaults to NULL).
\item \code{missingVarProp}: can be used instead of \code{missingVars}: The proportion of variables containing missing data (defaults to zero).
\item \code{missingProp}: The proportion of missingness for variables containing missing data (defaults to zero), either a single value or a vector giving the probabilities for each variable.
\item \code{missingMechanism}: The missing data mechanism, one of \code{MCAR} (the default), \code{MAR}, or \code{NMAR}.
\item \code{nCores}: The number of cores to use for parallel processing. Defaults to 1 (= no parallel processing). This requires the \code{doSNOW} package.
}

\code{type = 'IG'} implements the independent generator approach (IG, Foldnes & Olsson, 2016) approach
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors. This requires the \code{covsim} package.

\code{type = 'mnonr'} implements the approach suggested by Qu, Liu, & Zhang (2020) and requires provision of  Mardia's multivariate skewness (\code{skewness})  and kurtosis  (\code{kurtosis}), where
skewness must be non-negative and kurtosis must be at least 1.641 skewness + p (p + 0.774), where p is the number of variables. This requires the \code{mnonr} package.

\code{type = 'RK'} implements the approach suggested by Ruscio & Kaczetow (2008) and requires provision of the population distributions
of each variable (\code{distributions}). \code{distributions} must be a list (if all variables shall be based on the same population distribution) or a list of lists.
Each component must specify the population distribution (e.g. \code{rchisq}) and additional arguments (\code{list(df = 2)}).

\code{type = 'VM'} implements the third-order polynomial method (Vale & Maurelli, 1983)
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors.  This requires the \code{semTools} package.
}
\examples{
\dontrun{
# set up pathmodel in the form of
# f2 = .2*f1
# f3 = .3*f2
# f4 = .1*f1 + .4*f3
# obtain the required N to detect that the 
# slope f1 -> f4 is >= .10 
# with a power of 95\% on alpha = 5\%
# where f1 is measured by 3, f2 by 4, f3 by 5, and f4 by 6 indicators, 
# and all loadings are .5
Beta <- matrix(c(
  c(.00, .00, .00, .00),       # f1
  c(.20, .00, .00, .00),       # f2
  c(.00, .30, .00, .00),       # f3
  c(.10, .00, .40, .00)        # f4
), byrow = TRUE, ncol = 4)
powerPath <- semPower.powerPath(type = 'a-priori',
                                Beta = Beta,
                                nullWhich = c(4, 1),
                                nIndicator = c(3, 4, 5, 6), 
                                loadM = .5,
                                alpha = .05, beta = .05)
# show summary
summary(powerPath)
# optionally use lavaan to verify the model was set-up as intended
lavaan::sem(powerPath$modelH1, sample.cov = powerPath$Sigma,
sample.nobs = powerPath$requiredN, sample.cov.rescale = FALSE)
lavaan::sem(powerPath$modelH0, sample.cov = powerPath$Sigma,
sample.nobs = powerPath$requiredN, sample.cov.rescale = FALSE)

# same as above, but detect that the slope f3 -> f4 is >= .30 
powerPath <- semPower.powerPath(type = 'a-priori',
                                Beta = Beta,
                                nullWhich = c(4, 3),
                                nIndicator = c(3, 4, 5, 6), 
                                loadM = .5,
                                alpha = .05, beta = .05)

# same as above, but detect that 
# the slope f1 -> f2 (of .20) differs from the slope f2 -> f3 (of .30) 
powerPath <- semPower.powerPath(type = 'a-priori',
                                Beta = Beta,
                                nullEffect = 'betaX = betaZ',
                                nullWhich = list(c(2, 1), c(3, 2)),
                                nIndicator = c(3, 4, 5, 6), 
                                loadM = .5,
                                alpha = .05, beta = .05)

# same as above, but consider a multiple group model with equally sized groups, 
# and obtain the required N to detect that the slope 
# in group 1 (of .20) differs from the one in group 2 (of .40)
Beta1 <- Beta2 <- matrix(c(
  c(.00, .00, .00, .00),       # f1
  c(.20, .00, .00, .00),       # f2
  c(.00, .30, .00, .00),       # f3
  c(.10, .00, .40, .00)        # f4
), byrow = TRUE, ncol = 4)
Beta2[2, 1] <- .40
Beta <- list(Beta1, Beta2)
powerPath <- semPower.powerPath(type = 'a-priori',
                                Beta = Beta,
                                nullEffect = 'betaA = betaB',
                                nullWhich = c(2, 1),
                                nIndicator = c(3, 4, 5, 6), 
                                loadM = .5,
                                alpha = .05, beta = .05, N = list(1, 1))
}
}
\seealso{
\code{\link[=semPower.aPriori]{semPower.aPriori()}} \code{\link[=semPower.postHoc]{semPower.postHoc()}} \code{\link[=semPower.compromise]{semPower.compromise()}}
}
