\name{monitorTrial}
\alias{monitorTrial}
\title{
Group Sequential Monitoring of Simulated Efficacy Trials for the Event of Potential Harm, Non-Efficacy, and High Efficacy
}
\description{
\code{monitorTrial} applies a group sequential monitoring procedure to data-sets generated by \code{simTrial}, which may result in modification or termination of each simulated trial.
}
\usage{
monitorTrial(dataFile, stage1, stage2, harmMonitorRange, 
             alphaPerTest = NULL, minCnt, minPct, week1, 
             minCnt2, week2, nonEffInterval, lowerVEnoneff,
             upperVEnoneff, highVE, stage1VE, 
             lowerVEuncPower = NULL, alphaNoneff, alphaHigh, 
             alphaStage1, alphaUncPower = NULL,
             estimand = c("combined", "cox", "cuminc"), 
             post6moMonitor = FALSE, VEcutoffWeek, 
             saveDir = NULL, verbose = TRUE)
}
\arguments{
  \item{dataFile}{if \code{saveDir = NULL}, a list returned by \code{simTrial}; otherwise a name (character string) of an \code{.RData} file created by \code{simTrial}}
  \item{stage1}{the final week of stage 1 in a two-stage trial}
  \item{stage2}{the final week of stage 2 in a two-stage trial, i.e., the maximum follow-up time}
  \item{harmMonitorRange}{a 2-component numeric vector specifying the range for pooled numbers of infections (pooled over the placebo and vaccine arm accruing infections the fastest) for which potential-harm stopping boundaries will be computed}
  \item{alphaPerTest}{a per-test nominal/unadjusted alpha level for potential-harm monitoring. If \code{NULL}, a per-test alpha level is calculated that yields a cumulative alpha of 0.05 at the final potential-harm analysis.}
  \item{minCnt}{a minumum number of infections (pooled over the placebo and vaccine arm accruing infections the fastest) required for the initiation of non-efficacy monitoring [criterion 1]}
  \item{minPct}{a minimum proportion of infections after \code{week1} (pooled over the placebo and vaccine arm accruing infections the fastest) required for the initiation of non-efficacy monitoring [criterion 2]}
  \item{week1}{a time point (in weeks) used, together with \code{minPct}, for defining criterion 2}
  \item{minCnt2}{a minumum number of infections after \code{week2} (pooled over the placebo and vaccine arm accruing infections the fastest) required for the initiation of non-efficacy monitoring [criterion 3]}
  \item{week2}{a time point (in weeks) used, together with \code{minCnt2}, for defining criterion 3}
  \item{nonEffInterval}{a number of infections between two adjacent non-efficacy interim analyses}  
  \item{lowerVEnoneff}{specifies criterion 1 for declaring non-efficacy: the lower bound of the two-sided (1-\code{alphaNoneff}) x 100\% confidence interval(s) for the VE estimand(s) lie(s) below \code{lowerVEnoneff} (typically set equal to 0)}
  \item{upperVEnoneff}{specifies criterion 2 for declaring non-efficacy: the upper bound of the two-sided (1-\code{alphaNoneff}) x 100\% confidence interval(s) for the VE estimand(s) lie(s) below \code{upperVEnoneff} (typically a number in the 0--0.5 range)}  
  \item{highVE}{specifies a criterion for declaring high-efficacy: the lower bound of the two-sided (1-\code{alphaHigh}) x 100\% confidence interval for the VE estimand lies above \code{highVE} (typically a number in the 0.5--1 range)}
  \item{stage1VE}{specifies a criterion for advancement of a treatment's evaluation into Stage 2: the lower bound of the two-sided (1-\code{alphaStage1}) x 100\% confidence interval for the VE estimand lies above \code{stage1VE} (typically set equal to 0)}
  \item{lowerVEuncPower}{a numeric vector with each component specifying a one-sided null hypothesis H0: VE(0--\code{stage1}) \eqn{\le} \code{lowerVEuncPower} x 100\%. Unconditional power (i.e., accounting for sequential monitoring) to reject each H0 is calculated, where the rejection region is defined by the lower bound of the two-sided (1-\code{alphaUncPower}) x 100\% confidence interval for the VE estimand being above the respective component of \code{lowerVEuncPower} (typically values in the 0--0.5 range).}
  \item{alphaNoneff}{one minus the nominal confidence level of the two-sided confidence interval used for non-efficacy monitoring}
  \item{alphaHigh}{one minus the nominal confidence level of the two-sided confidence interval used for high efficacy monitoring}
  \item{alphaStage1}{one minus the nominal confidence level of the two-sided confidence interval used for determining whether a treatment's evaluation advances into Stage 2}
  \item{alphaUncPower}{one minus the nominal confidence level of the two-sided confidence interval used to test one-sided null hypotheses H0: VE(0-\code{stage1}) \eqn{\le} \code{lowerVEuncPower} x 100\% against alternative hypotheses H1: VE(0--\code{stage1}) \eqn{>} \code{lowerVEuncPower} x 100\%. The same nominal confidence level is applied for each component of \code{lowerVEuncPower}.}  
  \item{estimand}{a character string specifying the choice of VE estimand(s) used in non- and high efficacy monitoring, advancement rule for Stage 2, and unconditional power calculations. Three options are implemented: (1) the `pure' Cox approach (\code{"cox"}), where VE is defined as 1-hazard ratio (treatment/control) and estimated by the maximum partial likelihood estimator in the Cox model; (2) the `pure' cumulative incidence-based approach (\code{"cuminc"}), where VE is defined as 1-cumulative incidence ratio (treatment/control) and estimated by the transformation of the Nelson-Aalen estimator for the cumulative hazard function; and (3) the combined approach (\code{"combined"}), where both aforementioned VE estimands are used for non-efficacy monitoring while the cumulative VE estimand is used for all other purposes. Only the first three characters are necessary.}
  \item{post6moMonitor}{a logical value indicating whether, additionally, post-6 months non-efficacy monitoring shoud be used as a more conservative non-efficacy monitoring approach. If \code{TRUE} and \code{estimand = "combined"}, the cumulative VE estimand is considered only for non-efficacy monitoring.}
  \item{VEcutoffWeek}{a time point (in weeks) defining the per-protocol VE estimand, i.e., VE(\code{VEcutoffWeek}--\code{stage1}). This VE estimand is also used in post-6 months non-efficacy monitoring. It is typically chosen as the date of the last immunization or the date of the visit following the last immunization.}
  \item{saveDir}{a character string specifying a path for \code{dataFile}. If supplied, the output is also saved as an \code{.RData} file in this directory; otherwise the output is returned as a list.}
  \item{verbose}{a logical value indicating whether information on the output directory, file name, and monitoring outcomes should be printed out (default is \code{TRUE})}
}
\details{
All time variables use week as the unit of time. Month is defined as 52/12 weeks.

Potential harm monitoring starts at the \code{harmMonitorRange[1]}-th infection pooled over the placebo group and the vaccine regimen that accrues infections the fastest. The potential harm analyses continue at each additional infection until the first interim analysis for non-efficacy. The monitoring is implemented with exact one-sided binomial tests of H0: \eqn{p \le p0} versus H1: \eqn{p > p0}, where \eqn{p} is the probability that an infected participant was assigned to the vaccine group, and \eqn{p0} is a fixed constant that represents the null hypothesis that an infection is equally likely to be assigned vaccine or placebo. Each test is performed at the same prespecified nominal/unadjusted alpha-level (\code{alphaPerTest}), chosen based on simulations such that, for each vaccine regimen, the overall type I error rate by the \code{harmMonitorRange[2]}-th arm-pooled infection (i.e., the probability that the potential harm boundary is reached when the vaccine is actually safe, \eqn{p = p0}) equals 0.05.

Non-efficacy is defined as evidence that it is highly unlikely that the vaccine has a beneficial effect measured as VE(0--\code{stage1}) of \code{upperVEnoneff} x 100\% or more. The non-efficacy analyses for each vaccine regimen will start at the first infection at or after the \code{minCnt}-th (pooled over the vaccine and placebo arm) when at least \code{minPct} x 100\% of the accumulated infections are diagnosed after \code{week1} and at least \code{minCnt2} infections are diagnosed after \code{week2}. Stopping for non-efficacy will lead to a reported two-sided (1-\code{alphaNoneff}) x 100\% CI for VE(0--\code{stage1}) with the lower confidence bound below \code{lowerVEnoneff} and the upper confidence bound below \code{upperVEnoneff}, where \code{estimand} determines the choice of the VE(0--\code{stage1}) estimand. This approach is similar to the inefficacy monitoring approach of Freidlin B, Korn EL, Gray R. (2010) A general inefficacy interim monitoring rule for randomized trials. Clinical Trials, 7:197-208. If \code{estimand = "combined"}, stopping for non-efficacy will lead to reported (1-\code{alphaNoneff}) x 100\% CIs for both VE parameters with lower confidence bounds below \code{lowerVEnoneff} and upper confidence bounds below \code{upperVEnoneff}. If \code{post6moMonitor = TRUE}, stopping for non-efficacy will lead to reported (1-\code{alphaNoneff}) x 100\% CIs for both VE(0--\code{stage1}) and VE(\code{VEcutoffWeek}--\code{stage1}) with lower confidence bounds below \code{lowerVEnoneff} and upper confidence bounds below \code{upperVEnoneff}.

High efficacy monitoring allows early detection of a highly protective vaccine if there is evidence that VE(0--\code{stage2}) \eqn{>} \code{highVE} x 100\%, based on two planned interim analyses, the first at the time of the fifth planned non-efficacy analysis, and the second at the expected mid-point between the number of infections at the first interim analysis and the number of infections observed at the end of \code{stage2}. While monitoring for potential harm and non-efficacy restricts to \code{stage1} infections, monitoring for high efficacy counts all infections during \code{stage1} or \code{stage2}, given that early stopping for high efficacy would only be warranted under evidence for durability of the efficacy.

The following principles and rules are applied in the monitoring procedure:
\itemize{
\item Exclude all follow-up data from the analysis post-unblinding (and include all data pre-unblinding).
\item The monitoring is based on modified ITT analysis, i.e., all subjects documented to be free of the study endpoint at baseline are included and analyzed according to the treatment assigned by randomization, ignoring how many vaccinations they received (only pre-unblinding follow-up included).
\item If a vaccine hits the harm boundary, immediately discontinue vaccinations and accrual into this vaccine arm, and unblind this vaccine arm (continue post-unblinded follow-up until the end of Stage 1 for this vaccine arm).  
\item If a vaccine hits the non-efficacy boundary, immediately discontinue vaccinations and accrual into this vaccine arm, keep blinded and continue follow-up until the end of Stage 1 for this vaccine arm. 
\item If and when the last vaccine arm hits the non-efficacy (or harm) boundary, discontinue vaccinations and accrual into this vaccine arm, and unblind (the trial is over, completed in Stage 1).
\item Stage 1 for the whole trial is over on the earliest date of the two events: (1) all vaccine arms have hit the harm or non-efficacy boundary; and (2) the last enrolled subject in the trial reaches the final \code{stage1} visit.
\item Continue blinded follow-up until the end of Stage 2 for each vaccine arm that reaches the end of \code{stage1} with a positive efficacy (as defined by \code{stage1VE}) or high efficacy (as defined by \code{highVE}) result.
\item If at least one vaccine arm reaches the end of \code{stage1} with a positive efficacy or high efficacy result, continue blinded follow-up in the placebo arm until the end of Stage 2.
\item Stage 2 for the whole trial is over on the earliest date of the two events: (1) all subjects in the placebo arm and each vaccine arm that registered efficacy or high efficacy in \code{stage1} have failed or been censored; and (2) all subjects in the placebo arm and each vaccine arm that registered efficacy or high efficacy in \code{stage1} have completed the final \code{stage2} visit.
}
The above rules have the following implications:
\itemize{
\item If a vaccine hits the non-efficacy boundary but Stage 1 for the whole trial is not over, then one includes in the analysis all follow-up through the final \code{stage1} visit for that vaccine regimen, including all individuals accrued up through the date of hitting the non-efficacy boundary (which will be the total number accrued to this vaccine arm).
\item If a vaccine hits the harm boundary, all follow-up information through the date of hitting the harm boundary is included for this vaccine; no follow-up data are included after this date.
\item If and when the last vaccine arm hits the non-efficacy (or harm) boundary, all follow-up information through the date of hitting the non-efficacy (or harm) boundary is included for this vaccine; no follow-up data are included after this date.
}
}
\value{
If \code{saveDir} is specified, the output list (named \code{out}) is saved as an \code{.RData} file in \code{saveDir} (the path to \code{saveDir} is printed); otherwise it is returned. The output object is a list of length equal to the number of simulated trials, each of which is a list of length equal to the number of treatment arms, each of which is a list with (at least) the following components:
\item{boundHit}{a character string stating the monitoring outcome in this treatment arm, i.e., one of \code{"Harm"}, \code{"NonEffInterim"}, \code{"NonEffFinal"}, \code{"Eff"}, or \code{"HighEff"}. The first four outcomes can occur in Stage 1, whereas the last outcome can combine data over Stage 1 and Stage 2.}
\item{stopTime}{the time of hitting a stopping boundary since the first subject enrolled in the trial}
\item{stopInfectCnt}{the pooled number of infections at \code{stopTime}}
\item{summObj}{a \code{data.frame} containing summary information from each non-/high efficacy interim analysis}
\item{finalHRci}{the final CI for the hazard ratio, available if \code{estimand!="cuminc"} and there is at least 1 infection in each arm}
\item{firstNonEffCnt}{the number of infections that triggered non-efficacy monitoring (if available)}
\item{totInfecCnt}{the total number of \code{stage1} (\code{stage2} if \code{boundHit = "HighEff"}) infections}
\item{totInfecSplit}{a table with the numbers of \code{stage1} (\code{stage2} if \code{boundHit = "HighEff"}) infections in the treatment and control arm}
\item{lastExitTime}{the time between the first subject's enrollment and the last subject's exiting from the trial}
}
\seealso{
\code{\link{simTrial}}, \code{\link{censTrial}}, and \code{\link{rankTrial}}
}
\examples{
simData <- simTrial(N=c(1000, rep(700, 2)), aveVE=seq(0, 0.4, by=0.2), 
                    VEmodel="half", vePeriods=c(1, 27, 79), enrollPeriod=78, 
                    enrollPartial=13, enrollPartialRelRate=0.5, dropoutRate=0.05, 
                    infecRate=0.04, fuTime=156, 
                    visitSchedule=c(0, (13/3)*(1:4), seq(13*6/3, 156, by=13*2/3)),
                    missVaccProb=c(0,0.05,0.1,0.15), VEcutoffWeek=26, nTrials=5, 
                    stage1=78, randomSeed=300)
                    
monitorData <- monitorTrial(dataFile=simData, stage1=78, stage2=156, 
                            harmMonitorRange=c(10,100), alphaPerTest=0.0106, 
                            minCnt=50, minPct=0.33, week1=26, minCnt2=2, week2=52, 
                            nonEffInterval=20, lowerVEnoneff=0, upperVEnoneff=0.4, 
                            highVE=0.7, stage1VE=0, lowerVEuncPower=0, alphaNoneff=0.05,
                            alphaHigh=0.05, alphaStage1=0.05, alphaUncPower=0.05,
                            estimand="cuminc", VEcutoffWeek=26)

### alternatively, to save the .RData output file (no '<-' needed):
###
### simTrial(N=c(1400, rep(1000, 2)), aveVE=seq(0, 0.4, by=0.2), VEmodel="half", 
###          vePeriods=c(1, 27, 79), enrollPeriod=78, enrollPartial=13, 
###          enrollPartialRelRate=0.5, dropoutRate=0.05, infecRate=0.04, fuTime=156, 
###          visitSchedule=c(0, (13/3)*(1:4), seq(13*6/3, 156, by=13*2/3)), 
###          missVaccProb=c(0,0.05,0.1,0.15), VEcutoffWeek=26, nTrials=30, 
###          stage1=78, saveDir="./", randomSeed=300)
###
### monitorTrial(dataFile=
###          "simTrial_nPlac=1400_nVacc=1000_1000_aveVE=0.2_0.4_infRate=0.04.RData", 
###          stage1=78, stage2=156, harmMonitorRange=c(10,100), alphaPerTest=0.0106, 
###          minCnt=50, minPct=0.33, week1=26, minCnt2=2, week2=52, nonEffInterval=20, 
###          lowerVEnoneff=0, upperVEnoneff=0.4, highVE=0.7, stage1VE=0, 
###          lowerVEuncPower=0, alphaNoneff=0.05, alphaHigh=0.05, alphaStage1=0.05, 
###          alphaUncPower=0.05, estimand="cuminc", VEcutoffWeek=26, saveDir="./")
}
