% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/poisthin.R
\name{corassign}
\alias{corassign}
\title{Group assignment that is correlated with latent factors.}
\usage{
corassign(mat, nfac = NULL, corvec = NULL, return = c("group", "full"))
}
\arguments{
\item{mat}{A matrix of count data. The rows index the individuals and
the columns index the genes.}

\item{nfac}{The number of latent factors. If \code{NULL}, then we will
use \code{\link[cate]{est.factor.num}} from the cate
package to choose the number of latent factors.}

\item{corvec}{The vector of correlations. \code{corvec[i]} is the correlation
between latent factor \code{i} and the underlying group-assignment variable.
You can think of the correlations in \code{corvec} as a kind of "tetrachoric
correlation." If \code{NULL}, then it assumes independence between
factors and group assignment. Note that the correlations of the
latent factors with the observed group-assignment vector (instead of the
latent group-assignment vector) will be \code{corvec * sqrt(2 / pi)}.}

\item{return}{What should we return? Just the group assignment
(\code{"group"}) or a list of a bunch of things (\code{"full"}).}
}
\value{
A list with some or all of the following elements:
    \describe{
      \item{\code{x}}{The vector of group assignments. \code{0L} indicates
          membership to one group and \code{1L} indicates membership to
          the other group.}
      \item{\code{nfac}}{The number of assumed latent factors.}
      \item{\code{facmat}}{A matrix, whose columns contain the latent factors.}
      \item{\code{groupfac}}{The underlying group-assignment factor.}
      \item{\code{corvec}}{The correlation vector. Note that this is the
          correlation between random variables observed in \code{groupfac}
          and \code{facmat}, }
    }
    If \code{return = "group"}, then the list only contains \code{x}.
}
\description{
We extract latent factors from the log of \code{mat} using an SVD, then
generate an underlying group-assignment variable from a conditional
normal distribution (conditional on the latent factors). This underlying
group-assignment variable is used to assign groups.
}
\details{
If \code{nfac} is provided, then \code{corvec} must be the same length as \code{nfac}.
If \code{nfac} is not provided, then it is assumed that the first \code{nfac}
elements of \code{corvec} are the underlying correlations, if \code{nfac} turns out to be
smaller than the length of \code{corvec}. If \code{nfac} turns
out to be larger than the length of \code{corvec}, then the factors without
defined correlations are assumed to have correlation 0.
}
\examples{
## Simulate data from given matrix of counts
## In practice, you would obtain Y from a real dataset, not simulate it.
set.seed(1)
nsamp <- 1000
ngene <- 10
Y <- matrix(stats::rpois(nsamp * ngene, lambda = 50), nrow = ngene)

## Set target correlation to be 0.9 and nfac to be 1
corvec <- 0.9
nfac   <- 1

## Group assignment
cout <- corassign(mat    = t(Y),
                  nfac   = nfac,
                  corvec = corvec,
                  return = "full")

## Correlation between facmat and groupfac should be about 0.9
cor(cout$facmat, cout$groupfac)

## Correlation between facmat and x should be about 0.9 * sqrt(2 / pi)
cor(cout$facmat, cout$x)
corvec * sqrt(2 / pi)


}
\references{
\itemize{
  \item{Jingshu Wang and Qingyuan Zhao (2015). cate: High Dimensional
    Factor Analysis and Confounder Adjusted Testing and Estimation.
    R package version 1.0.4. https://CRAN.R-project.org/package=cate}
}
}
\author{
David Gerard
}
