% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/permute.R
\name{permute}
\alias{permute}
\alias{permute.array}
\alias{permute.matrix}
\alias{permute.data.frame}
\alias{permute.table}
\alias{permute.numeric}
\alias{permute.character}
\alias{permute.list}
\alias{permute.dist}
\alias{permute.dendrogram}
\alias{permute.hclust}
\title{Permute the Order in Various Objects}
\usage{
permute(x, order, ...)

\method{permute}{array}(x, order, margin = NULL, ...)

\method{permute}{matrix}(x, order, margin = NULL, ...)

\method{permute}{data.frame}(x, order, margin = NULL, ...)

\method{permute}{table}(x, order, margin = NULL, ...)

\method{permute}{numeric}(x, order, ...)

\method{permute}{character}(x, order, ...)

\method{permute}{list}(x, order, ...)

\method{permute}{dist}(x, order, ...)

\method{permute}{dendrogram}(x, order, ...)

\method{permute}{hclust}(x, order, ...)
}
\arguments{
\item{x}{an object (a list, a vector, a \code{dist} object, a matrix, an
array or any other object which provides \code{dim} and standard subsetting
with \code{"["}).}

\item{order}{an object of class \link{ser_permutation} which contains
suitable permutation vectors for \code{x}. Alternatively, a character string with the
name of a seriation method appropriate for \code{x} can be specified (see \code{\link[=seriate]{seriate()}}).
This will perform seriation and permute \code{x}.}

\item{...}{if \code{order} is the name of a seriation method, then additional arguments are
passed on to \code{\link[=seriate]{seriate()}}.}

\item{margin}{specifies the dimensions to be permuted as a vector with dimension indices.
If \code{NULL}, \code{order} needs to contain a permutation for all dimensions.
If a single margin is specified, then \code{order} can also contain
a single permutation vector.
\code{margin} are ignored.}
}
\value{
A permuted object of the same class as \code{x}.
}
\description{
Provides the generic function and methods for permuting the order of various
objects including vectors, lists, dendrograms (also \code{hclust} objects),
the order of observations in a \code{dist} object, the rows and columns of a
matrix or data.frame, and all dimensions of an array given a suitable
\link{ser_permutation} object.
}
\details{
The permutation vectors in \link{ser_permutation} are suitable if the number
of permutation vectors matches the number of dimensions of \code{x} and if
the length of each permutation vector has the same length as the
corresponding dimension of \code{x}.

For 1-dimensional/1-mode data (list, vector, \code{dist}), \code{order} can
also be a single permutation vector of class \link{ser_permutation_vector}
or data which can be automatically coerced to this class (e.g. a numeric
vector).

For \code{dendrogram} and \code{hclust}, subtrees are rotated to represent
the order best possible. If the order is not achieved perfectly then the
user is warned. This behavior can be changed with the extra parameter
\code{incompatible} which can take the values \code{"warn"} (default),
\code{"stop"} or \code{"ignore"}.
}
\examples{
# List data types for permute
methods("permute")

# Permute matrix
m <- matrix(rnorm(10), 5, 2, dimnames = list(1:5, LETTERS[1:2]))
m

# Permute rows and columns
o <- ser_permutation(5:1, 2:1)
o

permute(m, o)

## permute only columns
permute(m, o, margin = 2)

## permute using PCA seriation
permute(m, "PCA")

## permute only rows using PCA
permute(m, o, margin = 1)

# Permute data.frames
df <- as.data.frame(m)
permute(df, o)

# Permute objects in a dist object
d <- dist(m)
d

permute(d, c(3, 2, 1, 4, 5))

permute(d, "Spectral")

# Permute a list
l <- list(a = 1:5, b = letters[1:3], c = 0)
l

permute(l, c(2, 3, 1))

# Permute a dendrogram
hc <- hclust(d)
plot(hc)
plot(permute(hc, 5:1))
}
\seealso{
Other permutation: 
\code{\link{get_order}()},
\code{\link{permutation_vector2matrix}()},
\code{\link{ser_dist}()},
\code{\link{ser_permutation_vector}()},
\code{\link{ser_permutation}()}
}
\author{
Michael Hahsler
}
\concept{permutation}
\keyword{manip}
