% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/monthlyWB.R
\name{monthlyWB}
\alias{monthlyWB}
\title{Monthly Water Balances}
\usage{
monthlyWB(
  AWC,
  PPT,
  PET,
  S_init = AWC,
  starting_month = 1,
  rep = 1,
  keep_last = FALSE,
  fc = 1,
  a.ss = 0.001
)
}
\arguments{
\item{AWC}{available water-holding capacity (mm), typically thickness (mm) * awc}

\item{PPT}{time-series of monthly PPT (mm), calendar year ordering}

\item{PET}{time-series of monthly PET (mm), calendar year ordering}

\item{S_init}{initial fraction of \code{AWC} filled with water}

\item{starting_month}{starting month index, 1=January, 9=September}

\item{rep}{number of cycles to run water balance}

\item{keep_last}{keep only the last iteration of the water balance}

\item{fc}{fraction of \code{AWC} representing field capacity (see details)}

\item{a.ss}{recession coefficients for subsurface flow from saturated zone, should be > 0 but very small (see details)}
}
\value{
a \code{data.frame} with the following elements:

\itemize{
\item{PPT: }{monthly PPT values}
\item{PET: }{monthly PET values}
\item{U: }{monthly U values}
\item{S: }{monthly S values}
\item{ET: }{monthly ET values}
\item{D: }{monthly D values}
\item{month: }{month number}
\item{mo: }{month label}
}
}
\description{
Perform a monthly water balance by "leaky bucket" model, provided by the \code{hydromad} package.
}
\details{
At a monthly time step, \code{fc} and \code{a.ss} have very little impact on results. See \code{?bucket.sim} for details.
}
\note{
This function depends on the \href{http://hydromad.catchment.org/}{hydromad package}.
}
\examples{

if(requireNamespace('hydromad')) {

# 4" water storage ~ 100mm

# AWC in mm
AWC <- 200

# monthly PET and PPT in mm
PET <- c(0,0,5,80,90,120,130,140,110,90,20,5)
PPT <- c(0, 150, 200, 120, 20, 0, 0, 0, 10, 20, 30, 60)

# run water balance
# start with soil AWC "empty"
(x.wb <- monthlyWB(AWC, PPT, PET, S_init = 0))

# plot the results
op <- par(no.readonly = TRUE)

par(mar=c(4,4,2,1), bg = 'white')
plotWB(WB = x.wb)

# compute fraction of AWC filled after the last month of simulation
(last.S <- x.wb$S[12] / AWC)

# re-run the water balance with this value
(x.wb <- monthlyWB(AWC, PPT, PET, S_init = last.S))

# interesting...
par(mar=c(4,4,2,1), bg = 'white')
plotWB(WB = x.wb)

# note: consider using `rep = 3, keep_last = TRUE` 
# to "warm-up" the water balance first

par(op)

}

}
\references{
Farmer, D., M. Sivapalan, Farmer, D. (2003). Climate, soil and vegetation controls upon the variability of water balance in temperate and semiarid landscapes: downward approach to water balance analysis. Water Resources Research 39(2), p 1035.
}
\author{
D.E. Beaudette
}
