test_that("List collections", {
    col <- capture_output(sits_list_collections())
    expect_true(grepl("SENTINEL", col))
    expect_true(grepl("DEAFRICA", col))
    expect_true(grepl("LANDSAT", col))
    expect_true(grepl("BDC", col))
})
test_that("api_source", {
    res_s2_b8a <- .source_bands_resolution(
        source = "MPC",
        collection = "SENTINEL-2-L2A",
        bands = "B8A"
    )
    expect_equal(res_s2_b8a[["B8A"]], 20)
    res_l8_blue <- .source_bands_resolution(
        source = "MPC",
        collection = "LANDSAT-C2-L2",
        bands = "BLUE"
    )
    expect_equal(res_l8_blue[["BLUE"]], 30)

    vls_s2_cloud <- .source_cloud_values(
        source = "MPC",
        collection = "SENTINEL-2-L2A"
    )
    expect_true(all(names(vls_s2_cloud) %in%
                        as.character(seq(from = 0, to = 11))))
    expect_equal(vls_s2_cloud[["0"]], "missing_data")
    expect_equal(vls_s2_cloud[["11"]], "snow or ice")

    open_mpc <- .source_collection_open_data(
        source = "MPC",
        collection = "SENTINEL-2-L2A"
    )
    expect_true(open_mpc)
    token_mpc <- .source_collection_open_data(
        source = "MPC",
        collection = "SENTINEL-2-L2A",
        token = TRUE
    )
    expect_false(token_mpc)

    open_bdc <- .source_collection_open_data(
        source = "BDC",
        collection = "SENTINEL-2-16D"
    )
    expect_true(open_bdc)

    token_bdc <- .source_collection_open_data(
        source = "BDC",
        collection = "SENTINEL-2-16D",
        token = TRUE
    )
    expect_true(token_bdc)

})
test_that("Reading a raster cube", {
    data_dir <- system.file("extdata/raster/mod13q1", package = "sits")
    raster_cube <- sits_cube(
            source = "BDC",
            collection = "MOD13Q1-6",
            data_dir = data_dir,
            multicores = 2,
            progress = FALSE
    )
    # get bands names
    bands <- sits_bands(raster_cube)
    expect_true(all(bands %in% c("NDVI", "EVI")))
    # test params
    params <- .raster_params_file(raster_cube$file_info[[1]]$path)
    expect_true(params$nrows == 147)
    expect_true(params$ncols == 255)
    expect_true(params$xres >= 231.5)
    # test timeline
    timeline <- sits_timeline(raster_cube)
    sub_cube <- sits_select(raster_cube,
                            start_date = timeline[1],
                            end_date = timeline[2])
    expect_equal(length(sits_timeline(sub_cube)), 2)
    params_2 <- .raster_params_file(sub_cube$file_info[[1]]$path)
    expect_true(params_2$nrows == 147)
    expect_true(params_2$ncols == 255)
    expect_true(params_2$xres >= 231.5)
})
test_that("Creating cubes from BDC - CBERS-WFI-16D", {
    # check "BDC_ACCESS_KEY" - mandatory one per user
    bdc_access_key <- Sys.getenv("BDC_ACCESS_KEY")
    testthat::skip_if(nchar(bdc_access_key) == 0,
        message = "No BDC_ACCESS_KEY defined in environment."
    )
    tiles <- c("007004", "007005")
    start_date <- "2021-09-01"
    end_date <- "2021-09-30"
    bands <- c("NDVI", "EVI", "B13", "B14", "B15", "B16", "CLOUD")
    # create a raster cube
    cbers_cube_16d <- .try({
        sits_cube(
            source = "BDC",
            collection = "CBERS-WFI-16D",
            tiles = tiles,
            start_date = start_date,
            end_date = end_date,
            progress = FALSE)
    },
    .default = NULL)
    testthat::skip_if(purrr::is_null(cbers_cube_16d),
        message = "BDC is not accessible"
    )
    # test bands and bbox
    expect_true(all(sits_bands(cbers_cube_16d) %in% bands))
    bbox <- sits_bbox(cbers_cube_16d)
    int_bbox <- .bbox_intersection(bbox, .tile_bbox(cbers_cube_16d))
    expect_true(all(int_bbox == sits_bbox(.tile(cbers_cube_16d))))
    # test timeline
    timeline <- sits_timeline(cbers_cube_16d)
    expect_true(timeline[1] <= as.Date(start_date))
    expect_true(timeline[length(timeline)] <= as.Date(end_date))
    # test raster obj
    r_obj <- .raster_open_rast(cbers_cube_16d$file_info[[1]]$path[1])
    cube_nrows <- .tile_nrows(cbers_cube_16d)
    expect_true(.raster_nrows(r_obj) == cube_nrows)
})
test_that("Creating cubes from BDC - CBERS-WFI-8D", {
    # check "BDC_ACCESS_KEY" - mandatory one per user
    bdc_access_key <- Sys.getenv("BDC_ACCESS_KEY")
    testthat::skip_if(nchar(bdc_access_key) == 0,
                      message = "No BDC_ACCESS_KEY defined in environment."
    )
    tiles <- c("007004", "007005")
    start_date <- "2022-05-01"
    end_date <- "2022-08-29"
    bands <- c("NDVI", "EVI", "B13", "B14", "B15", "B16", "CLOUD")
    # create a raster cube file from BDC
    cbers_cube_8d <- .try({
        sits_cube(
            source = "BDC",
            collection = "CBERS-WFI-8D",
            tiles = tiles,
            start_date = start_date,
            end_date = end_date,
            progress = FALSE)
    },
    .default = NULL)
    testthat::skip_if(purrr::is_null(cbers_cube_8d),
                      message = "BDC is not accessible"
    )
    expect_true(all(sits_bands(cbers_cube_8d) %in% bands))
    bbox <- sits_bbox(cbers_cube_8d)
    int_bbox <- .bbox_intersection(bbox, .tile_bbox(cbers_cube_8d))
    expect_true(all(int_bbox == sits_bbox(.tile(cbers_cube_8d))))

    timeline <- sits_timeline(cbers_cube_8d)
    expect_true(timeline[1] <= as.Date(start_date))
    expect_true(timeline[length(timeline)] <= as.Date(end_date))

    r_obj <- .raster_open_rast(cbers_cube_8d$file_info[[1]]$path[1])
    cube_nrows <- .tile_nrows(cbers_cube_8d)
    expect_true(.raster_nrows(r_obj) == cube_nrows)
})
test_that("Creating cubes from BDC - based on ROI with shapefile", {
    # check "BDC_ACCESS_KEY" - mandatory one per user
    bdc_access_key <- Sys.getenv("BDC_ACCESS_KEY")
    testthat::skip_if(nchar(bdc_access_key) == 0,
        message = "No BDC_ACCESS_KEY defined in environment."
    )
    shp_file <- system.file(
        "extdata/shapefiles/mato_grosso/mt.shp",
        package = "sits"
    )
    sf_mt <- sf::read_sf(shp_file)
    # create a raster cube file based on the information about the files
    modis_cube <- .try({
        sits_cube(
            source = "BDC",
            collection = "MOD13Q1-6",
            bands = c("NDVI", "EVI"),
            roi = sf_mt,
            start_date = "2018-09-01",
            end_date = "2019-08-29",
            progress = FALSE
        )
    },
    .default = NULL)
    testthat::skip_if(purrr::is_null(modis_cube),
        message = "BDC is not accessible"
    )
    expect_true(all(sits_bands(modis_cube) %in% c("NDVI", "EVI")))
    bbox <- sits_bbox(modis_cube, as_crs = "EPSG:4326")
    bbox_shp <- sf::st_bbox(sf_mt)
    expect_lt(bbox["xmin"], bbox_shp["xmin"])
    expect_lt(bbox["ymin"], bbox_shp["ymin"])
    expect_gt(bbox["xmax"], bbox_shp["xmax"])
    expect_gt(bbox["ymax"], bbox_shp["ymax"])
    intersects <- .cube_intersects(modis_cube, sf_mt)
    expect_true(all(intersects))

})
test_that("Creating cubes from BDC - invalid roi", {
    # check "BDC_ACCESS_KEY" - mandatory one per user
    bdc_access_key <- Sys.getenv("BDC_ACCESS_KEY")

    testthat::skip_if(nchar(bdc_access_key) == 0,
        message = "No BDC_ACCESS_KEY defined in environment."
    )
    expect_error(
        object = sits_cube(
            source = "BDC",
            collection = "MOD13Q1-6",
            bands = c("NDVI", "EVI"),
            roi = c(TRUE, FALSE),
            start_date = "2018-09-01",
            end_date = "2019-08-29",
            progress = FALSE
        )
    )
    expect_error(
        object = sits_cube(
            source = "BDC",
            collection = "MOD13Q1-6",
            bands = c("NDVI", "EVI"),
            roi = c(
                lon_min = -55.20997,
                lat_min = 15.40554,
                lon_max = -55.19883,
                lat_max = -15.39179
            ),
            tiles = "012010",
            start_date = "2018-09-01",
            end_date = "2019-08-29",
            progress = FALSE
        )
    )
})
test_that("Creating cubes from BDC - LANDSAT per tile", {
    # check "BDC_ACCESS_KEY" - mandatory one per user
    bdc_access_key <- Sys.getenv("BDC_ACCESS_KEY")
    testthat::skip_if(nchar(bdc_access_key) == 0,
                      message = "No BDC_ACCESS_KEY defined in environment."
    )
    tile <- "038046"
    start_date <- "2021-05-01"
    end_date <- "2021-09-30"
    bands <- c("NDVI", "EVI")
    # create a raster cube file based on the information about the files
    bdc_l8_cube <- .try({
        sits_cube(
            source = "BDC",
            collection = "LANDSAT-OLI-16D",
            bands = bands,
            tiles = tile,
            start_date = start_date,
            end_date = end_date,
            progress = FALSE
        )
    },
    .default = NULL)

    testthat::skip_if(purrr::is_null(bdc_l8_cube),
                      message = "BDC cube LANDSAT-OLI-16D is not accessible"
    )
    expect_equal(bdc_l8_cube$tile, tile)
    expect_true(all(sits_bands(bdc_l8_cube) %in% bands))
    # test timeline
    timeline <- sits_timeline(bdc_l8_cube)
    expect_true(timeline[1] <= as.Date(start_date))
    expect_true(timeline[length(timeline)] <= as.Date(end_date))
    # test raster obj
    r_obj <- .raster_open_rast(bdc_l8_cube$file_info[[1]]$path[1])
    cube_nrows <- .tile_nrows(bdc_l8_cube)
    expect_true(.raster_nrows(r_obj) == cube_nrows)
})
test_that("Creating cubes from BDC - LANDSAT per roi", {
    # check "BDC_ACCESS_KEY" - mandatory one per user
    bdc_access_key <- Sys.getenv("BDC_ACCESS_KEY")
    testthat::skip_if(nchar(bdc_access_key) == 0,
                      message = "No BDC_ACCESS_KEY defined in environment."
    )
    roi <- c(lon_min = -53.9311, lat_min = -13.2697,
             lon_max = -53.0595, lat_max = -12.6704)
    start_date <- "2021-05-01"
    end_date <- "2021-09-30"
    bands <- c("NDVI", "EVI")
    # create a raster cube file based on the information about the files
    bdc_l8_cube <- .try({
        sits_cube(
            source = "BDC",
            collection = "LANDSAT-OLI-16D",
            bands = bands,
            roi = roi,
            start_date = start_date,
            end_date = end_date,
            progress = FALSE
        )
    },
    .default = NULL)

    testthat::skip_if(purrr::is_null(bdc_l8_cube),
                      message = "BDC cube LANDSAT-OLI-16D is not accessible"
    )
    expect_true(all(sits_bands(bdc_l8_cube) %in% bands))
    bbox_cube <- sits_bbox(bdc_l8_cube, as_crs = "EPSG:4326")
    intersects <- .cube_intersects(bdc_l8_cube, roi)
    expect_true(all(intersects))
    # test timeline
    timeline <- sits_timeline(bdc_l8_cube)
    expect_true(timeline[1] <= as.Date(start_date))
    expect_true(timeline[length(timeline)] <= as.Date(end_date))
    # test raster obj
    r_obj <- .raster_open_rast(bdc_l8_cube$file_info[[1]]$path[1])
    cube_nrows <- .tile_nrows(bdc_l8_cube)
    expect_true(.raster_nrows(r_obj) == cube_nrows)
})
test_that("Creating cubes from BDC - SENTINEL-2 - roi", {
    # check "BDC_ACCESS_KEY" - mandatory one per user
    bdc_access_key <- Sys.getenv("BDC_ACCESS_KEY")
    testthat::skip_if(nchar(bdc_access_key) == 0,
                      message = "No BDC_ACCESS_KEY defined in environment."
    )
    roi <- c(lon_min = -53.9311, lat_min = -13.2697,
             lon_max = -53.0595, lat_max = -12.6704)
    start_date <- "2021-05-01"
    end_date <- "2021-09-30"
    bands <- c("NDVI", "EVI")
    # create a raster cube
    bdc_s2_cube <- .try({
        sits_cube(
            source = "BDC",
            collection = "SENTINEL-2-16D",
            bands = bands,
            roi = roi,
            start_date = start_date,
            end_date = end_date,
            progress = FALSE
        )
    },
    .default = NULL)
    testthat::skip_if(purrr::is_null(bdc_s2_cube),
                      message = "BDC cube SENTINEL-2-16D is not accessible"
    )
    expect_true(all(sits_bands(bdc_s2_cube) %in% c("NDVI", "EVI")))
    bbox_cube <- sits_bbox(bdc_s2_cube, as_crs = "EPSG:4326")
    intersects <- .cube_intersects(bdc_s2_cube, roi)
    expect_true(all(intersects))
    # test timeline
    timeline <- sits_timeline(bdc_s2_cube)
    expect_true(timeline[1] <= as.Date(start_date))
    expect_true(timeline[length(timeline)] <= as.Date(end_date))
    # test raster obj
    r_obj <- .raster_open_rast(bdc_s2_cube$file_info[[1]]$path[1])
    cube_nrows <- .tile_nrows(bdc_s2_cube)
    expect_true(.raster_nrows(r_obj) == cube_nrows)
})
test_that("Creating cubes from BDC - SENTINEL-2 - tile", {
    # check "BDC_ACCESS_KEY" - mandatory one per user
    bdc_access_key <- Sys.getenv("BDC_ACCESS_KEY")
    testthat::skip_if(nchar(bdc_access_key) == 0,
                      message = "No BDC_ACCESS_KEY defined in environment."
    )
    start_date <- "2021-05-01"
    end_date <- "2021-09-30"
    bands <- c("NDVI", "EVI")
    tiles <- "021019"
    # create a raster cube file
    bdc_s2_cube_t <- .try({
        sits_cube(
            source = "BDC",
            collection = "SENTINEL-2-16D",
            bands = bands,
            tiles = tiles,
            start_date = start_date,
            end_date =  end_date,
            progress = FALSE
        )
    },
    .default = NULL)

    testthat::skip_if(purrr::is_null(bdc_s2_cube_t),
                      message = "BDC cube SENTINEL-2-16D is not accessible"
    )
    expect_true(all(sits_bands(bdc_s2_cube_t) %in% c("NDVI", "EVI")))
    # test timeline
    timeline <- sits_timeline(bdc_s2_cube_t)
    expect_true(timeline[1] <= as.Date(start_date))
    expect_true(timeline[length(timeline)] <= as.Date(end_date))
    # test raster obj
    r_obj <- .raster_open_rast(bdc_s2_cube_t$file_info[[1]]$path[1])
    cube_nrows <- .tile_nrows(bdc_s2_cube_t)
    expect_true(.raster_nrows(r_obj) == cube_nrows)
})
test_that("Creating cubes from DEA", {
    # try to create a DEA cube
    dea_cube <- .try({
        sits_cube(
            source = "DEAFRICA",
            collection = "s2_l2a",
            bands = c("B01", "B04", "B05"),
            roi = c(
                lon_min = 17.379,
                lat_min = 1.1573,
                lon_max = 17.410,
                lat_max = 1.1910
            ),
            start_date = "2019-01-01",
            end_date = "2019-10-28",
            progress = FALSE
        )
    },
    .default = NULL
    )
    testthat::skip_if(purrr::is_null(dea_cube),
        message = "DEAFRICA is not accessible"
    )
    expect_true(all(sits_bands(dea_cube) %in% c("B01", "B04", "B05")))
    r <- .raster_open_rast(.tile_path(dea_cube))
    expect_equal(dea_cube$xmax[[1]], .raster_xmax(r), tolerance = 1)
    expect_equal(dea_cube$xmin[[1]], .raster_xmin(r), tolerance = 1)
})
test_that("Creating cubes from DEA - error using tiles", {
    expect_error(
        object = {
            dea_cube <-
                sits_cube(
                    source = "DEAFRICA",
                    collection = "s2_l2a",
                    bands = c("B01", "B04", "B05"),
                    tiles = "37MEP",
                    start_date = "2019-01-01",
                    end_date = "2019-10-28",
                    progress = FALSE
                )
        },
        regexp = "DEAFRICA cubes do not support searching for tiles"
    )
})
test_that("Creating Sentinel cubes from MPC", {
    s2_cube <- .try({
        sits_cube(
            source = "MPC",
            collection = "SENTINEL-2-L2A",
            tiles = "20LKP",
            bands = c("B05", "CLOUD"),
            start_date = as.Date("2018-07-18"),
            end_date = as.Date("2018-08-23"),
            progress = FALSE
        )
    },
    .default = NULL
    )
    testthat::skip_if(purrr::is_null(s2_cube),
        "MPC is not accessible"
    )
    expect_true(all(sits_bands(s2_cube) %in% c("B05", "CLOUD")))
    r <- .raster_open_rast(.tile_path(s2_cube))
    expect_equal(s2_cube$xmax[[1]], .raster_xmax(r), tolerance = 1)
    expect_equal(s2_cube$xmin[[1]], .raster_xmin(r), tolerance = 1)
    r_obj <- .raster_open_rast(s2_cube$file_info[[1]]$path[1])
    cube_nrows <- .tile_nrows(s2_cube)
    expect_true(.raster_nrows(r_obj) == cube_nrows)
})
test_that("Creating Sentinel cubes from MPC with ROI", {
    roi <- c(lon_min = -48.28579, lat_min = -16.05026,
             lon_max = -47.30839, lat_max = -15.50026)
    s2_cube_mpc <- .try({
        sits_cube(
            source = "MPC",
            collection = "SENTINEL-2-L2A",
            roi = roi,
            bands = c("B05", "CLOUD"),
            start_date = as.Date("2018-07-18"),
            end_date = as.Date("2018-08-23"),
            progress = FALSE
        )
    },
    .default = NULL
    )
    testthat::skip_if(purrr::is_null(s2_cube_mpc), "MPC is not accessible")
    expect_true(all(sits_bands(s2_cube_mpc) %in% c("B05", "CLOUD")))
    expect_equal(nrow(s2_cube_mpc), 3)
    bbox_cube <- sits_bbox(s2_cube_mpc, as_crs = "EPSG:4326")
    bbox_cube_1 <- sits_bbox(.tile(s2_cube_mpc), as_crs = "EPSG:4326")
    expect_true(bbox_cube["xmax"] >= bbox_cube_1["xmax"])
    expect_true(bbox_cube["ymax"] >= bbox_cube_1["ymax"])
    r_obj <- .raster_open_rast(s2_cube_mpc$file_info[[1]]$path[1])
    cube_nrows <- .tile_nrows(s2_cube_mpc)
    expect_true(.raster_nrows(r_obj) == cube_nrows)

})
test_that("Creating LANDSAT cubes from MPC with ROI", {
    roi <- c(lon_min = -48.28579, lat_min = -16.05026,
             lon_max = -47.30839, lat_max = -15.50026)
    l8_cube_mpc <- .try({
        sits_cube(
            source = "MPC",
            collection = "LANDSAT-C2-L2",
            roi = roi,
            bands = c("NIR08", "CLOUD"),
            start_date = as.Date("2018-07-18"),
            end_date = as.Date("2018-08-23"),
            progress = FALSE
        )
    },
    .default = NULL
    )
    testthat::skip_if(purrr::is_null(l8_cube_mpc), "MPC is not accessible")
    expect_true(all(sits_bands(l8_cube_mpc) %in% c("NIR08", "CLOUD")))
    expect_equal(nrow(l8_cube_mpc), 2)
    bbox_cube <- sits_bbox(l8_cube_mpc, as_crs = "EPSG:4326")
    bbox_cube_1 <- sits_bbox(.tile(l8_cube_mpc), as_crs = "EPSG:4326")
    expect_true(bbox_cube["xmax"] >= bbox_cube_1["xmax"])
    expect_true(bbox_cube["ymax"] >= bbox_cube_1["ymax"])
    r_obj <- .raster_open_rast(l8_cube_mpc$file_info[[1]]$path[1])
    tile_nrows <- .tile_nrows(l8_cube_mpc)[[1]]
    expect_true(.raster_nrows(r_obj) == tile_nrows)

})
test_that("Creating Harmonized Landsat Sentinel HLSS30 cubes", {
    roi <- c(lon_min = -48.28579, lat_min = -16.05026,
             lon_max = -47.30839, lat_max = -15.50026)
    hls_cube_s2 <- .try({
        sits_cube(
            source = "HLS",
            collection = "HLSS30",
            roi = roi,
            bands = c("GREEN", "NIR-NARROW", "SWIR-1", "CLOUD"),
            start_date = as.Date("2020-05-01"),
            end_date = as.Date("2020-09-01"),
            progress = FALSE
        )
    },
    .default = NULL
    )
    testthat::skip_if(
        purrr::is_null(hls_cube_s2),
        "HLSS30 collection is not accessible"
    )
    expect_true(all(sits_bands(hls_cube_s2) %in%
                        c("GREEN", "NIR-NARROW", "SWIR-1", "CLOUD")))
    expect_true(all(hls_cube_s2$satellite == "SENTINEL-2"))
    expect_true(all(hls_cube_s2$tile %in% c("23LKC", "22LHH", "22LGH")))
    expect_true(all(.fi(hls_cube_s2)$xres == 30))
    expect_true(all(.fi(hls_cube_s2)$yres == 30))
    r_obj <- .raster_open_rast(hls_cube_s2$file_info[[1]]$path[1])
    tile_nrows <- .tile_nrows(hls_cube_s2)[[1]]
    expect_true(.raster_nrows(r_obj) == tile_nrows)

    hls_cube_l8 <- .try({
        sits_cube(
            source = "HLS",
            collection = "HLSL30",
            roi = roi,
            bands = c("GREEN", "NIR-NARROW", "SWIR-1", "CLOUD"),
            start_date = as.Date("2020-05-01"),
            end_date = as.Date("2020-09-01"),
            progress = FALSE
        )
    },
    .default = NULL
    )
    testthat::skip_if(
        purrr::is_null(hls_cube_l8),
        "HLSL30 collection is not accessible"
    )
    expect_true(all(sits_bands(hls_cube_l8) %in%
                        c("GREEN", "NIR-NARROW", "SWIR-1", "CLOUD")))
    expect_true(all(hls_cube_l8$satellite == "LANDSAT-8"))
    expect_true(all(hls_cube_l8$tile %in% c("23LKC", "22LHH", "22LGH")))
    expect_true(all(.fi(hls_cube_l8)$xres == 30))
    expect_true(all(.fi(hls_cube_l8)$yres == 30))

    hls_cube_merge <- sits_merge(hls_cube_s2, hls_cube_l8)
    merge_23LKC <- dplyr::filter(hls_cube_merge, tile == "23LKC")
    s2_23LKC <- dplyr::filter(hls_cube_s2, tile == "23LKC")
    l8_23LKC <- dplyr::filter(hls_cube_l8, tile == "23LKC")
    expect_true(all(sits_timeline(merge_23LKC) %in%
                        c(sits_timeline(l8_23LKC), sits_timeline(s2_23LKC))))

})
test_that("Creating Sentinel cubes from AWS", {
    s2_cube <- .try({
        sits_cube(
            source = "AWS",
            collection = "SENTINEL-2-L2A",
            tiles = "20LKP",
            bands = c("B05", "CLOUD"),
            start_date = as.Date("2018-07-18"),
            end_date = as.Date("2018-08-23"),
            progress = FALSE
        )
    },
    .default = NULL
    )
    testthat::skip_if(
        purrr::is_null(s2_cube),
        "AWS is not accessible"
    )
    expect_true(all(sits_bands(s2_cube) %in% c("B05", "CLOUD")))
    r <- .raster_open_rast(.tile_path(s2_cube))
    expect_equal(s2_cube$xmax[[1]], .raster_xmax(r), tolerance = 1)
    expect_equal(s2_cube$xmin[[1]], .raster_xmin(r), tolerance = 1)
})
test_that("Creating LANDSAT cubes from AWS with ROI", {
    roi <- c(lon_min = -48.28579, lat_min = -16.05026,
             lon_max = -47.30839, lat_max = -15.50026)
    l8_cube_aws <- .try({
        sits_cube(
            source = "AWS",
            collection = "LANDSAT-C2-L2",
            roi = roi,
            bands = c("NIR08", "CLOUD"),
            start_date = as.Date("2018-07-18"),
            end_date = as.Date("2018-08-23"),
            progress = FALSE
        )
    },
    .default = NULL
    )
    testthat::skip_if(purrr::is_null(l8_cube_aws), "AWS is not accessible")
    expect_true(all(sits_bands(l8_cube_aws) %in% c("NIR08", "CLOUD")))
    expect_equal(nrow(l8_cube_aws ), 2)
    bbox_cube <- sits_bbox(l8_cube_aws , as_crs = "EPSG:4326")
    bbox_cube_1 <- sits_bbox(.tile(l8_cube_aws), as_crs = "EPSG:4326")
    expect_true(bbox_cube["xmax"] >= bbox_cube_1["xmax"])
    expect_true(bbox_cube["ymax"] >= bbox_cube_1["ymax"])
    r_obj <- .raster_open_rast(l8_cube_aws$file_info[[1]]$path[1])
    tile_nrows <- .tile_nrows(l8_cube_aws)[[1]]
    expect_true(.raster_nrows(r_obj) == tile_nrows)
})
