\name{sjt.frq}
\alias{sjt.frq}
\title{Show frequencies as HTML table}
\usage{
sjt.frq(data, file = NULL, weightBy = NULL, variableLabels = NULL,
  valueLabels = NULL, autoGroupAt = NULL, alternateRowColors = FALSE,
  stringValue = "value", stringCount = "N", stringPerc = "raw \%",
  stringValidPerc = "valid \%", stringCumPerc = "cumulative \%",
  stringMissingValue = "missings", highlightMedian = FALSE,
  highlightQuartiles = FALSE, skipZeroRows = FALSE, showSummary = TRUE,
  showSkew = FALSE, showKurtosis = FALSE, skewString = "&gamma;",
  kurtosisString = "&omega;", encoding = "UTF-8", CSS = NULL,
  useViewer = TRUE, no.output = FALSE)
}
\arguments{
  \item{data}{The variables which frequencies should be
  printed as table. Either use a single variable (vector)
  or a data frame where each column represents a variable
  (see examples.}

  \item{file}{The destination file, which will be in
  html-format. If no filepath is specified, the file will
  be saved as temporary file and openend either in the
  RStudio View pane or in the default web browser.}

  \item{weightBy}{A weight factor that will be applied to
  weight all cases from \code{data}.  default is
  \code{NULL}, so no weights are used.}

  \item{variableLabels}{A single character vector or a list
  of character vectors that indicate the variable names of
  those variables from \code{data} and will be used as
  variable labels in the output. Note that if multiple
  variables are supplied (as data frame), the variable
  labels must be supplied as \code{list} object (see
  examples).}

  \item{valueLabels}{A list of character vectors that
  indicate the value labels of those variables from
  \code{data}. Note that if multiple variables are supplied
  (as data frame), the value labels must be supplied as
  nested \code{list} object (see examples).}

  \item{autoGroupAt}{A value indicating at which length of
  unique values a variable from \code{data} is
  automatically grouped into smaller units (see
  \code{\link{sju.groupVar}}). Variables with large numbers
  of unique values may be too time consuming when a HTML
  table is created and R would not respond any longer.
  Hence it's recommended to group such variables. Default
  value is 50, i.e. variables with 50 and more unique
  values will be grouped using \code{\link{sju.groupVar}}
  with \code{groupsize="auto"} parameter. By default, the
  maximum group count is 30. However, if \code{autoGroupAt}
  is less than 30, \code{autoGroupAt} groups are built.
  Default value is \code{NULL}, i.e. auto-grouping is
  turned off.}

  \item{alternateRowColors}{If \code{TRUE}, alternating
  rows are highlighted with a light gray background color.}

  \item{stringValue}{String label for the very first table
  column containing the values (see \code{valueLabels}).}

  \item{stringCount}{String label for the first table data
  column containing the counts. Default is \code{"N"}.}

  \item{stringPerc}{String label for the second table data
  column containing the percentages, where the count
  percentages include missing values.}

  \item{stringValidPerc}{String label for the third data
  table column containing the valid percentages, i.e. the
  count percentage value exluding possible missing values.}

  \item{stringCumPerc}{String label for the last table data
  column containing the cumulative percentages.}

  \item{stringMissingValue}{String label for the last table
  data row containing missing values.}

  \item{highlightMedian}{If \code{TRUE}, the table row
  indicating the median value will be highlighted.}

  \item{highlightQuartiles}{If \code{TRUE}, the table row
  indicating the lower and upper quartiles will be
  highlighted.}

  \item{skipZeroRows}{If \code{TRUE}, rows with only
  zero-values are not printed. Default is \code{FALSE}.}

  \item{showSummary}{If \code{TRUE} (default), a summary
  row with total and valid N as well as mean and standard
  deviation is shown.}

  \item{showSkew}{If \code{TRUE}, the variable's skewness
  is added to the summary.  The skewness is retrieved from
  the \code{\link{describe}} function of the
  \code{\link{psych}} package.}

  \item{showKurtosis}{If \code{TRUE}, the variable's
  kurtosis is added to the summary.  The kurtosis is
  retrieved from the \code{\link{describe}} function of the
  \code{\link{psych}} package.}

  \item{skewString}{A character string, which is used as
  header for the skew column (see \code{showSkew})).
  Default is lower case Greek gamma.}

  \item{kurtosisString}{A character string, which is used
  as header for the kurtosis column (see
  \code{showKurtosis})).  Default is lower case Greek
  omega.}

  \item{encoding}{The charset encoding used for variable
  and value labels. Default is \code{"UTF-8"}. Change
  encoding if specific chars are not properly displayed
  (e.g.) German umlauts).}

  \item{CSS}{A \code{\link{list}} with user-defined
  style-sheet-definitions, according to the official CSS
  syntax (see \url{http://www.w3.org/Style/CSS/}). See
  return value \code{page.style} for details of all
  style-sheet-classnames that are used in this function.
  Parameters for this list need: \enumerate{ \item the
  class-names with \code{"css."}-prefix as parameter name
  and \item each style-definition must end with a semicolon
  } Examples: \itemize{ \item \code{css.table='border:2px
  solid red;'} for a solid 2-pixel table border in red.
  \item \code{css.summary='font-weight:bold;'} for a bold
  fontweight in the summary row.  \item
  \code{css.lasttablerow='border-bottom: 1px dotted blue;'}
  for a blue dotted border of the last table row.  } See
  further examples below.}

  \item{useViewer}{If \code{TRUE}, the function tries to
  show the HTML table in the IDE's viewer pane. If
  \code{FALSE} or no viewer available, the HTML table is
  opened in a web browser.}

  \item{no.output}{If \code{TRUE}, the html-output is
  neither opened in a browser nor shown in the viewer pane
  and not even saved to file. This option is useful when
  the html output should be used in \code{knitr} documents.
  The html output can be accessed via the return value.}
}
\value{
Invisibly returns a \code{\link{structure}} with \itemize{
\item the web page style sheet (\code{page.style}), \item
each frequency table as web page content
(\code{page.content.list}), \item the complete html-output
(\code{output.complete}) and \item the html-table with
inline-css for use with knitr (\code{knitr}) } for further
use.
}
\description{
Shows (multiple) frequency tables as HTML file, or saves
them as file.
}
\note{
The HTML tables can either be saved as file and manually
opened (specify parameter \code{file}) or they can be saved
as temporary files and will be displayed in the RStudio
Viewer pane (if working with RStudio) or opened with the
default web browser. Displaying resp. opening a temporary
file is the default behaviour (i.e. \code{file=NULL}).
}
\examples{
# load sample data
data(efc)

# retrieve value and variable labels
variables <- sji.getVariableLabels(efc)
values <- sji.getValueLabels(efc)

# show frequencies of "e42dep" in RStudio Viewer Pane
# or default web browser
\dontrun{
sjt.frq(efc$e42dep)}

# plot and show frequency table of "e42dep" with labels
\dontrun{
sjt.frq(efc$e42dep,
        variableLabels=variables['e42dep'],
        valueLabels=values[['e42dep']])}

# plot frequencies of e42dep, e16sex and c172code in one HTML file
# and show table in RStudio Viewer Pane or default web browser
\dontrun{
sjt.frq(as.data.frame(cbind(efc$e42dep, efc$e16sex, efc$c172code)),
        variableLabels=list(variables['e42dep'], variables['e16sex'], variables['c172code']),
        valueLabels=list(values[['e42dep']], values[['e16sex']], values[['c172code']]))}

# plot larger scale including zero-counts
# indicating median and quartiles
\dontrun{
sjt.frq(efc$neg_c_7,
        variableLabels=variables['neg_c_7'],
        valueLabels=values[['neg_c_7']],
        highlightMedian=TRUE,
        highlightQuartiles=TRUE)}

# -------------------------------
# auto-detection of labels
# -------------------------------
efc <- sji.setVariableLabels(efc, variables)
\dontrun{
sjt.frq(data.frame(efc$e42dep, efc$e16sex, efc$c172code))}

# --------------------------------
# User defined style sheet
# --------------------------------
\dontrun{
sjt.frq(efc$e42dep,
        variableLabels=variables['e42dep'],
        valueLabels=values[['e42dep']],
        CSS=list(css.table="border: 2px solid;",
                 css.tdata="border: 1px solid;",
                 css.firsttablecol="color:#003399; font-weight:bold;"))}
}
\references{
\itemize{ \item
\url{http://strengejacke.wordpress.com/sjplot-r-package/}
\item
\url{http://strengejacke.wordpress.com/2014/02/20/no-need-for-spss-beautiful-output-in-r-rstats/}
}
}
\seealso{
\code{\link{sjp.frq}} \cr \code{\link{sjt.xtab}}
}

