#' Gather User Data for League
#'
#' Given a league ID, grab the data concerning each user. This includes information about the users' mascot,
#' display name, avatar, if they are an owner, if they are a bot, and more.
#'
#' @return Returns a data frame containing information about the users in the league.
#' @author Nick Bultman, \email{njbultman74@@gmail.com}, September 2021
#' @keywords league users
#' @importFrom httr GET content
#' @importFrom jsonlite fromJSON
#' @importFrom dplyr rename
#' @export
#' @examples
#' \dontrun{get_league_users(688281863499907072)}
#'
#' @param league_id League ID generated by Sleeper (numeric or character)
#'
get_league_users <- function(league_id) {
  # Send request to API given league ID specified
  x <- jsonlite::fromJSON(httr::content(httr::GET(paste0("https://api.sleeper.app/v1/league/", league_id, "/users")), as = "text"))
  # Check if class of data returned from request is list
  if(inherits(x, "list")) {
    # If class is list, it is an empty list and no data was found - inform user and do not return anything
    message("No data found - was the league ID entered correctly?")
  } else {
    # If class is not list, strip out metadata nested data frame from main query
    x_metadata <- dplyr::rename(x$metadata,
                                "avatar_upload" = "avatar")
    # Drop metadata nested data frame from main query
    x$metadata <- NULL
    # Bind metadata data frame to main query so no nested data frames
    x_fin <- cbind(x, x_metadata)
    # Return final data frame
    return(x_fin)
  }
}
