#' @title SM3 Hash
#' @description
#' SM3 is a cryptographic hash function designed for digital signatures and
#' other cryptographic applications. The output of SM3 is a 256-bit hash value,
#' which is commonly represented as a 64-hexadecimal digit string.
#' @details
#' All the functions mentioned - \link{sm3_hash}, \link{sm3_hash_string}, and \link{sm3_hash_file} -
#' return a 64-character hexadecimal string representing the 256-bit hash value
#' generated by the SM3 cryptographic hash function. This hexadecimal string is
#' commonly used to represent the hash output in a human-readable format.
#' The \link{sm3_hash} function calculates the SM3 hash of a raw vector input and
#' returns a 64-character hexadecimal string. Similarly, \link{sm3_hash_string} takes
#' a string as input and also returns a 64-character hexadecimal string
#' representing the SM3 hash of the input string. The \link{sm3_hash_file} function,
#' on the other hand, takes a file path as input, reads the contents of the file,
#' calculates its SM3 hash, and returns the corresponding 64-character hexadecimal string.
#' @rdname sm3_hash
#' @param msg data to be hashed
#' @return a 64-characters hex string, which will be the sm3 hash result of the data, string or file
#' @examples
#' ## Raw vector hashing
#' msg <- charToRaw('abc')
#' sm3_hash(msg)
#'
#' ## character string hashing
#' sm3_hash_string('abc')
#'
#' ## local file hashing
#' \dontrun{
#'   sm3_hash_file('test.docx')
#' }
#' @export
#' @useDynLib smcryptoR sm3_hash_wrapper
sm3_hash <- function(msg) {
  .Call(sm3_hash_wrapper, msg)
}

#' @rdname sm3_hash
#' @param msg_string a character string to be hashed
#' @export
#' @useDynLib smcryptoR sm3_hash_string_wrapper
sm3_hash_string <- function(msg_string) {
  .Call(sm3_hash_string_wrapper, msg_string)
}

#' @rdname sm3_hash
#' @param file_path a local file to be hashed
#' @export
#' @useDynLib smcryptoR sm3_hash_file_wrapper
sm3_hash_file <- function(file_path) {
  .Call(sm3_hash_file_wrapper, file_path)
}
