% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fetchSOLUS.R
\name{fetchSOLUS}
\alias{fetchSOLUS}
\title{Fetch Soil Landscapes of the United States (SOLUS) Grids}
\usage{
fetchSOLUS(
  x = NULL,
  depth_slices = c(0, 5, 15, 30, 60, 100, 150),
  variables = c("anylithicdpt", "caco3", "cec7", "claytotal", "dbovendry", "ec", "ecec",
    "fragvol", "gypsum", "ph1to1h2o", "resdept", "sandco", "sandfine", "sandmed",
    "sandtotal", "sandvc", "sandvf", "sar", "silttotal", "soc"),
  output_type = c("prediction", "relative prediction interval",
    "95\% low prediction interval", "95\% high prediction interval"),
  grid = TRUE,
  samples = NULL,
  method = c("linear", "constant", "fmm", "natural", "monoH.FC", "step", "slice"),
  max_depth = 151,
  filename = NULL,
  overwrite = FALSE
)
}
\arguments{
\item{x}{An R spatial object (such as a \emph{SpatVector}, \emph{SpatRaster}, or \emph{sf} object) or a
\emph{SoilProfileCollection} with coordinates initialized via \verb{aqp::initSpatial<-}. Default: \code{NULL}
returns the CONUS extent as virtual raster. If \code{x} is a \emph{SpatRaster} the coordinate reference
system, extent, and resolution are used as a template for the output raster.}

\item{depth_slices}{character. One or more of: \code{"0"}, \code{"5"}, \code{"15"}, \code{"30"}, \code{"60"}, \code{"100"},
\code{"150"}. The "depth slice" \code{"all"} (used for variables such as \code{"anylithicdpt"}, and
\code{"resdept"}) is always included if any site-level variables are selected.}

\item{variables}{character. One or more of: \code{"anylithicdpt"}, \code{"caco3"}, \code{"cec7"}, \code{"claytotal"},
\code{"dbovendry"}, \code{"ec"}, \code{"ecec"}, \code{"fragvol"}, \code{"gypsum"}, \code{"ph1to1h2o"}, \code{"resdept"},
\code{"sandco"}, \code{"sandfine"}, \code{"sandmed"}, \code{"sandtotal"}, \code{"sandvc"}, \code{"sandvf"}, \code{"sar"},
\code{"silttotal"}, \code{"soc"}.}

\item{output_type}{character. One or more of: \code{"prediction"}, \code{"relative prediction interval"},
\code{"95\% low prediction interval"}, \code{"95\% high prediction interval"}}

\item{grid}{logical. Default \code{TRUE} returns a \emph{SpatRaster} object for an extent. \code{FALSE} returns
a \emph{SoilProfileCollection}. Any other value returns a \emph{list} object with names \code{"grid"} and
\code{"spc"} containing both result objects.}

\item{samples}{integer. Number of regular samples to return for \emph{SoilProfileCollection} output.
Default \code{NULL} will convert all grid cells to a unique profile. Note that for a large extent,
this can produce large objects with a very large number of layers (especially with \code{method}
other than \code{"step"}).}

\item{method}{character. Used to determine depth interpolation method for \emph{SoilProfileCollection}
output. Default: \code{"linear"}. Options include any \code{method} allowed for \code{approxfun()} or
\code{splinefun()} plus \code{"step"} and \code{"slice"}. \code{"step"} uses the prediction depths as the top and
bottom of each interval to create a piecewise continuous profile to maximum of 200 cm depth
(for 150 cm upper prediction depth). \code{"slice"} returns a discontinuous profile with 1 cm thick
slices at the predicted depths. Both \code{"step"} and \code{"slice"} return a number of layers equal to
length of \code{depth_slices}, and all other methods return data in interpolated 1cm slices.}

\item{max_depth}{integer. Maximum depth to interpolate 150 cm slice data to. Default: \code{151}.
Interpolation deeper than 151 cm is not possible for methods other than \code{"step"} and will
result in missing values.}

\item{filename}{character. Path to write output raster file. Default: \code{NULL} will keep result in
memory (or store in temporary file if memory threshold is exceeded)}

\item{overwrite}{Overwrite \code{filename} if it exists? Default: \code{FALSE}}
}
\value{
A \emph{SpatRaster} object containing SOLUS grids for specified extent, depths, variables, and
product types.
}
\description{
This tool creates a virtual raster or downloads data for an extent from Cloud Optimized GeoTIFFs
(COGs) from the \href{https://agdatacommons.nal.usda.gov/articles/dataset/Data_from_Soil_Landscapes_of_the_United_States_100-meter_SOLUS100_soil_property_maps_project_repository/25033856}{Soil Landscapes of the United States 100-meter (SOLUS100) soil property maps project repository}.
}
\details{
If the input object \code{x} is not specified (\code{NULL} or missing), a \emph{SpatRaster} object using the
virtual URLs is returned. The full extent and resolution data set can be then downloaded and
written to file using \code{terra::writeRaster()} (or any other processing step specifying an output
file name). When input object \code{x} is specified, a \emph{SpatRaster} object using in memory or local
(temporary file or \code{filename}) resources is returned after downloading the data only for the
target extent. In the case where \code{x} is a \emph{SoilProfileCollection} or an \emph{sf} or \emph{SpatVector}
object containing point geometries, the result will be a \emph{SoilProfileCollection} for values
extracted at the point locations. To return both the \emph{SpatRaster} and \emph{SoilProfileCollection}
object output in a \emph{list}, use \code{grid = NULL}.
}
\examples{
\dontshow{if (curl::has_internet()  && requireNamespace("httr", quietly = TRUE) && requireNamespace("sf") && requireNamespace("terra")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}

\dontrun{
b <- c(-119.747629, -119.67935, 36.912019, 36.944987)

bbox.sp <- sf::st_as_sf(wk::rct(
  xmin = b[1], xmax = b[2], ymin = b[3], ymax = b[4],
  crs = sf::st_crs(4326)
))

ssurgo.geom <- soilDB::SDA_spatialQuery(
  bbox.sp,
  what = 'mupolygon',
  db = 'SSURGO',
  geomIntersection = TRUE
)

# grid output
res <- fetchSOLUS(
  ssurgo.geom,
  depth_slices = "0",
  variables = c("sandtotal", "silttotal", "claytotal", "cec7"),
  output_type = "prediction"
)

terra::plot(res)

# SoilProfileCollection output, using linear interpolation for 1cm slices
# site-level variables (e.g. resdept) added to site data.frame of SPC
res <- fetchSOLUS(
  ssurgo.geom,
  depth_slices = c("0", "5", "15", "30", "60", "100", "150"),
  variables = c("sandtotal", "silttotal", "claytotal", "cec7", "resdept"),
  output_type = "prediction",
  method = "linear",
  grid = FALSE,
  samples = 10
)

# plot, truncating each profile to the predicted restriction depth
aqp::plotSPC(trunc(res, 0, res$resdept_p), color = "claytotal_p", divide.hz = FALSE)
}
\dontshow{\}) # examplesIf}
}
\references{
Nauman, T. W., Kienast-Brown, S., Roecker, S. M., Brungard, C., White, D., Philippe,
J., & Thompson, J. A. (2024). Soil landscapes of the United States (SOLUS): developing
predictive soil property maps of the conterminous United States using hybrid training sets.
Soil Science Society of America Journal, 88, 2046–2065. \doi{https://doi.org/10.1002/saj2.20769}
}
\author{
Andrew G. Brown
}
