% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MGC.R, R/MGC_Function.R, R/MGC_Linear.R,
%   R/MGC_Noise.R, R/MGC_Random.R, R/MGC_Shape.R, R/MGC_Static.R
\name{MGC}
\alias{MGC}
\alias{MGC_Function}
\alias{MGC_Linear}
\alias{keyframe}
\alias{add_keyframe}
\alias{get_keyframes}
\alias{remove_keyframe}
\alias{MGC_Noise}
\alias{MGC_Random}
\alias{Shape_Gaussian}
\alias{Shape_Block}
\alias{MGC_Static}
\title{Moving Generator Cluster}
\usage{
MGC(...)

MGC_Function(density, center, parameter, shape = Shape_Gaussian)

MGC_Linear(dimension = 2, keyframelist = NULL, shape = Shape_Gaussian)

keyframe(time, density, center, parameter, reset = FALSE)

add_keyframe(x, time, density, center, parameter, reset = FALSE)

get_keyframes(x)

remove_keyframe(x, time)

MGC_Noise(density, range)

MGC_Random(density, center, parameter, randomness = 1, shape = Shape_Gaussian)

Shape_Gaussian(center, parameter)

Shape_Block(center, parameter)

MGC_Static(density = 1, center, parameter, shape = Shape_Gaussian)
}
\arguments{
\item{...}{Further arguments.}

\item{density}{The density of the cluster. For `MGC_Function, this
attribute is a function and defines the density of a cluster (i.e., how many points it creates)
at each given timestamp.}

\item{center}{A list that defines the center of the cluster. The list should
have a length equal to the dimensionality. For \code{MGC_Function}, this
list consists of functions that define the movement of the cluster. For
\code{MGC_Random}, this attribute defines the beginning location for the
\code{MGC} before it begins moving.}

\item{parameter}{Parameters for the shape. For the default shape
\code{Shape_Gaussian} the parameter is the standard deviation, one per
dimension. If a single value is specified then it is recycled for all
dimensions.}

\item{shape}{A function creating the shape of the cluster. It gets passed on
the parameters argument from above. Available functions are
\code{Shape_Gaussian} (the parameters are a vector containing standard
deviations) and \code{Shape_Block} (parameters are the dimensions of the
uniform block).}

\item{dimension}{Dimensionality of the data stream.}

\item{keyframelist}{a list of keyframes to initialize the \code{MGC_Linear}
object with.}

\item{time}{The time stamp the keyframe should be located or which keyframe
should be removed.}

\item{reset}{Should the cluster reset to the first keyframe (time 0) after
this keyframe is finished?}

\item{x}{An object of class \code{MGC_Linear}.}

\item{range}{The area in which the noise should appear.}

\item{randomness}{The maximum amount the cluster will move during one time
step.}
}
\description{
Creates an evolving cluster for use as a component of a \link{DSD_MG} data stream.
}
\details{
An \code{MGC} describes a single cluster for use as a component in a \link{DSD_MG}.
Different \link{MGC}s  allow the user to express
different cluster behaviors within a single data stream. Static, (i.e., not moving)
clusters are defined as:
\itemize{
\item \code{MGC_Static} cluster positions are fixed
\item \code{MGC_Noise} allows to add random noise.
}

Moving (evolving) clusters are defined as:
\itemize{
\item \code{MGC_Linear} creates an evolving cluster for a who's behavior is determined by
keyframes. Several keyframe
functions are provided to create, add and remove keyframes.
See Examples section for details.
\item \code{MGC_Function} allows to specify \code{density}, \code{center}, and \code{parameter}
as a function of time.
\item \code{MGC_Random} allows for a creation of a cluster that follows a random walk.
}

Cluster shapes can be specified using the functions:
\itemize{
\item \code{Shape_Gaussian}
\item \code{Shape_Block}
}

New
shapes can be defined as a function with parameters \code{center} and \code{parameter} that return a single new
point. Here is an example:

\if{html}{\out{<div class="sourceCode">}}\preformatted{Shape_Gaussian <- function(center, parameter)
   rnorm(length(center), mean = center, sd = parameter)
}\if{html}{\out{</div>}}
}
\examples{
MGC()

### Two static clusters (Gaussian with sd of .1 and a Block with width .4)
###   with added noise
stream <- DSD_MG(dim = 2,
  MGC_Static(den = .45, center = c(1, 0), par = .1, shape = Shape_Gaussian),
  MGC_Static(den = .45, center = c(2, 0), par = .4, shape = Shape_Block),
  MGC_Noise( den = .1, range = rbind(c(0, 3), c(-1,1)))
)
stream

plot(stream)

### Example of several MGC_Randoms which define clusters that randomly move.
stream <- DSD_MG(dim = 2,
  MGC_Random(den = 100, center=c(1, 0), par = .1, rand = .2),
  MGC_Random(den = 100, center=c(2, 0), par = .4, shape = Shape_Block, rand = .2)
)

\dontrun{
  animate_data(stream, 2500, xlim = c(0,3), ylim = c(-1,1), horizon = 100)
}


### Example of several MGC_Functions

### a block-shaped cluster moving from bottom-left to top-right increasing size
c1 <- MGC_Function(
  density = function(t){ 100 },
  parameter = function(t){ 1 * t },
  center = function(t) c(t, t),
  shape = Shape_Block
  )

### a cluster moving in a circle (default shape is Gaussian)
c2 <- MGC_Function(
  density = function(t){ 25 },
  parameter = function(t){ 5 },
  center= function(t) c(sin(t / 10) * 50 + 50, cos(t / 10) * 50 + 50)
)

stream <- DSD_MG(dim = 2, c1, c2)

## adding noise after the stream was created
add_cluster(stream, MGC_Noise(den = 10, range = rbind(c(-20, 120), c(-20, 120))))

stream

\dontrun{
animate_data(stream, 10000, xlim = c(-20, 120), ylim = c(-20, 120), horizon = 100)
}

### Example of several MGC_Linear: A single cluster splits at time 50 into two.
### Note that c2 starts at time = 50!
stream <- DSD_MG(dim = 2)
c1 <- MGC_Linear(dim = 2)
add_keyframe(c1, time = 1,  dens = 50, par = 5, center = c(0, 0))
add_keyframe(c1, time = 50, dens = 50, par = 5, center = c(50, 50))
add_keyframe(c1, time = 100,dens = 50, par = 5, center = c(50, 100))
add_cluster(stream, c1)

c2 <- MGC_Linear(dim = 2, shape = Shape_Block)
add_keyframe(c2, time = 50, dens = 25, par = c(10, 10), center = c(50, 50))
add_keyframe(c2, time = 100,dens = 25, par = c(30, 30), center = c(100, 50))
add_cluster(stream, c2)

\dontrun{
animate_data(stream, 5000, xlim = c(0, 100), ylim = c(0, 100), horiz = 100)
}

### two fixed and a moving cluster
stream <- DSD_MG(dim = 2,
  MGC_Static(dens = 1, par = .1, center = c(0, 0)),
  MGC_Static(dens = 1, par = .1, center = c(1, 1)),
  MGC_Linear(dim = 2, list(
    keyframe(time = 0,    dens = 1, par = .1, center = c(0, 0)),
    keyframe(time = 1000, dens = 1, par = .1, center = c(1, 1)),
    keyframe(time = 2000, dens = 1, par = .1, center = c(0, 0), reset = TRUE)
  )))

noise <- MGC_Noise(dens = .1, range = rbind(c(-.2, 1.2), c(-.2, 1.2)))
add_cluster(stream, noise)

\dontrun{
animate_data(stream, n = 2000 * 3.1, xlim = c(-.2, 1.2), ylim = c(-.2, 1.2), horiz = 200)
}

}
\seealso{
\link{DSD_MG} for details on how to use an \code{MGC} within
a \link{DSD}.
}
\author{
Matthew Bolanos
}
