% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/samplePoisMixture.R
\name{samplePoisMixture}
\alias{samplePoisMixture}
\title{Telescoping sampling of a Bayesian finite Poisson mixture with a
prior on the number of components K.}
\usage{
samplePoisMixture(
  y,
  S,
  mu,
  eta,
  a0,
  b0,
  h0,
  H0,
  M,
  burnin,
  thin,
  Kmax,
  G = c("MixDynamic", "MixStatic"),
  priorOnK,
  priorOnWeights,
  verbose = FALSE
)
}
\arguments{
\item{y}{A numeric matrix; containing the data.}

\item{S}{A numeric matrix; containing the initial cluster
assignments.}

\item{mu}{A numeric matrix; containing the initial cluster-specific
rate values.}

\item{eta}{A numeric vector; containing the initial cluster sizes.}

\item{a0}{A numeric vector; hyperparameter of the prior on the rate \eqn{\mu}.}

\item{b0}{A numeric vector; hyperparameter of the prior on the rate \eqn{\mu}.}

\item{h0}{A numeric vector; hyperparameter of the prior on the rate \eqn{\mu}.}

\item{H0}{A numeric vector; hyperparameter of the prior on the rate \eqn{\mu}.}

\item{M}{A numeric scalar; specifying the number of recorded
iterations.}

\item{burnin}{A numeric scalar; specifying the number of burn-in
iterations.}

\item{thin}{A numeric scalar; specifying the thinning used for the
iterations.}

\item{Kmax}{A numeric scalar; the maximum number of components.}

\item{G}{A character string; either \code{"MixDynamic"} or \code{"MixStatic"}.}

\item{priorOnK}{A named list; providing the prior on the number of components K, see \code{\link[=priorOnK_spec]{priorOnK_spec()}}.}

\item{priorOnWeights}{A named list; providing the prior on the mixture weights.}

\item{verbose}{A logical; indicating if some intermediate clustering
results should be printed.}
}
\value{
A named list containing:
\itemize{
\item \code{"Mu"}: sampled rate \eqn{\mu}.
\item \code{"Eta"}: sampled weights.
\item \code{"S"}: sampled assignments.
\item \code{"Nk"}: number of observations assigned to the different components, for each iteration.
\item \code{"K"}: sampled number of components.
\item \code{"Kplus"}: number of filled, i.e., non-empty components, for each iteration.
\item \code{"e0"}: sampled Dirichlet parameter of the prior on the weights (if \eqn{e_0} is random).
\item \code{"alpha"}: sampled Dirichlet parameter of the prior on the weights (if \eqn{\alpha} is random).
\item \code{"acc"}: logical vector indicating acceptance in the Metropolis-Hastings step when sampling either e0 or \eqn{\alpha}.
}
}
\description{
\itemize{
\item The MCMC scheme is implemented as suggested in Frühwirth-Schnatter et al (2021).
\item The priors on the model parameters are specified as in
Frühwirth-Schnatter et al (2021) and Früwirth-Schnatter and
Malsiner-Walli (2019), see the vignette for details and notation.
}
}
\examples{
N <- 200
z <- sample(1:2, N, prob = c(0.5, 0.5), replace = TRUE)
y <- rpois(N, c(1, 6)[z])

M <- 200
thin <- 1
burnin <- 100

Kmax <- 50  
Kinit <- 10

G <- "MixDynamic"
priorOnAlpha <- priorOnAlpha_spec("gam_1_2")
priorOnK <- priorOnK_spec("BNB_143")

a0 <- 0.1 
h0 <- 0.5 
b0 <- a0/mean(y) 
H0 <- h0/b0

cl_y <- kmeans(y, centers = Kinit, nstart = 100)
S_0 <- cl_y$cluster
mu_0 <- t(cl_y$centers)
eta_0 <- rep(1/Kinit, Kinit)

result <- samplePoisMixture(
  y, S_0, mu_0, eta_0, 
  a0, b0, h0, H0,
  M, burnin, thin, Kmax, 
  G, priorOnK, priorOnAlpha)

K <- result$K
Kplus <- result$Kplus

plot(K, type = "l", ylim = c(0, max(K)), 
     xlab = "iteration", main = "",
     ylab = expression("K" ~ "/" ~ K["+"]), col = 1)
lines(Kplus, col = 2)
legend("topright", legend = c("K", expression(K["+"])),
       col = 1:2, lty = 1, box.lwd = 0)

}
