% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/delaunay.R
\name{delaunay}
\alias{delaunay}
\title{Delaunay triangulation}
\usage{
delaunay(
  points,
  atinfinity = FALSE,
  degenerate = FALSE,
  exteriorEdges = FALSE,
  elevation = FALSE
)
}
\arguments{
\item{points}{the points given as a matrix, one point per row}

\item{atinfinity}{Boolean, whether to include a point at infinity}

\item{degenerate}{Boolean, whether to include degenerate tiles}

\item{exteriorEdges}{Boolean, for dimension 3 only, whether to return
the exterior edges (see below)}

\item{elevation}{Boolean, only for three-dimensional points; if \code{TRUE},
the function performs an elevated Delaunay tessellation, using the
third coordinate of a point for its elevation; see the example}
}
\value{
If the function performs an elevated Delaunay tessellation, then
  the returned value is a list with four fields: \code{mesh}, \code{edges},
  \code{volume}, and \code{surface}. The \code{mesh} field is an object of
  class \code{mesh3d}, ready for plotting with the \strong{rgl} package. The
  \code{edges} field provides the indices of the vertices of the edges, and
  others informations; see \code{\link[Rvcg]{vcgGetEdge}}.
  The \code{volume} field provides the sum of the
  volumes under the Delaunay triangles, that is to say the total volume
  under the triangulated surface. Finally, the \code{surface} field provides
  the sum of the areas of the Delaunay triangles, thus this an approximate
  value of the area of the surface that is triangulated.
  The elevated Delaunay tessellation is built with the help of the
  \strong{interp} package.

Otherwise, the function returns the Delaunay tessellation with many details,
  in a list. This list contains three fields:
\describe{
  \item{\emph{vertices}}{the vertices (or sites) of the tessellation; these
  are the points passed to the function}
  \item{\emph{tiles}}{the tiles of the tessellation (triangles in dimension 2,
  tetrahedra in dimension 3)}
  \item{\emph{tilefacets}}{the facets of the tiles of the tessellation}
}
In dimension 3, the list contains an additional field \emph{exteriorEdges}
  if you set \code{exteriorEdges = TRUE}. This is the list of the exterior
  edges, represented as \code{\link{Edge3}} objects. This field is involved
  in the function \code{\link{plotDelaunay3D}}.

The \strong{vertices} field is a list with the following fields:
\describe{
  \item{\emph{id}}{the id of the vertex; this is nothing but the index of
  the corresponding point passed to the function}
  \item{\emph{neighvertices}}{the ids of the vertices of the tessellation
  connected to this vertex by an edge}
  \item{\emph{neightilefacets}}{the ids of the tile facets this vertex
  belongs to}
  \item{\emph{neightiles}}{the ids of the tiles this vertex belongs to}
}
The \strong{tiles} field is a list with the following fields:
\describe{
  \item{\emph{id}}{the id of the tile}
  \item{\emph{simplex}}{a list describing the simplex (that is, the tile);
  this list contains four fields: \emph{vertices}, a
  \code{\link[hash]{hash}} giving the simplex vertices and their id,
  \emph{circumcenter}, the circumcenter of the simplex, \emph{circumradius},
  the circumradius of the simplex, and \emph{volume}, the volume of the
  simplex}
  \item{\emph{facets}}{the ids of the facets of this tile}
  \item{\emph{neighbors}}{the ids of the tiles adjacent to this tile}
  \item{\emph{family}}{two tiles have the same family if they share the
  same circumcenter; in this case the family is an integer, and the family is
  \code{NA} for tiles which do not share their circumcenter with any other
  tile}
  \item{\emph{orientation}}{\code{1} or \code{-1}, an indicator of the
  orientation of the tile}
}
The \strong{tilefacets} field is a list with the following fields:
\describe{
  \item{\emph{id}}{the id of this tile facet}
  \item{\emph{subsimplex}}{a list describing the subsimplex (that is, the
  tile facet); this list is similar to the \emph{simplex} list of
  \strong{tiles}}
  \item{\emph{facetOf}}{one or two ids, the id(s) of the tile this facet
  belongs to}
  \item{\emph{normal}}{a vector, the normal of the tile facet}
  \item{\emph{offset}}{a number, the offset of the tile facet}
}
}
\description{
Delaunay triangulation (or tessellation) of a set of points.
}
\note{
The package provides the functions \code{\link{plotDelaunay2D}} to
  plot a 2D Delaunay tessellation and \code{\link{plotDelaunay3D}} to
  plot a 3D Delaunay tessellation. But there is no function to plot an
  elevated Delaunay tessellation; the examples show how to plot such a
  Delaunay tessellation.
}
\examples{
library(tessellation)
points <- rbind(
 c(0.5,0.5,0.5),
 c(0,0,0),
 c(0,0,1),
 c(0,1,0),
 c(0,1,1),
 c(1,0,0),
 c(1,0,1),
 c(1,1,0),
 c(1,1,1)
)
del <- delaunay(points)
del$vertices[[1]]
del$tiles[[1]]
del$tilefacets[[1]]

# an elevated Delaunay tessellation ####
f <- function(x, y){
  dnorm(x) * dnorm(y)
}
x <- y <- seq(-5, 5, length.out = 50)
grd <- expand.grid(x = x, y = y) # grid on the xy-plane
points <- as.matrix(transform( # data (x_i, y_i, z_i)
  grd, z = f(x, y)
))
del <- delaunay(points, elevation = TRUE)
del[["volume"]] # close to 1, as expected
# plotting
library(rgl)
mesh <- del[["mesh"]]
open3d(windowRect = c(100, 100, 612, 356), zoom = 0.6)
aspect3d(1, 1, 20)
shade3d(mesh, color = "limegreen")
wire3d(mesh)
}
\seealso{
\code{\link{getDelaunaySimplicies}}
}
