#' Visualize the CWL workflow
#'
#' @param g Graph generated by \code{\link{get_graph}}.
#' @param hierarchical Enable the hierarchical layout? Default is \code{TRUE}.
#' @param direction Direction of the hierarchical layout.
#' Options include \code{"LR"}, \code{"RL"}, \code{"UD"}, and \code{"DU"}
#' (up-down, down-up, left-right, right-left). Default is \code{"LR"}.
#' @param separation Level separation parameter from
#' \code{\link[visNetwork]{visHierarchicalLayout}}.
#' @param palette Three-color palette for inputs, outputs, and steps.
#' @param width Canvas width, see \code{\link[visNetwork]{visNetwork}}.
#' Default is \code{"100\%"}.
#' @param height Canvas height, see \code{\link[visNetwork]{visNetwork}}.
#' Default is \code{600}.
#'
#' @return A \code{\link[visNetwork]{visNetwork}} output.
#'
#' @importFrom visNetwork visNetwork visNodes visEdges visGroups visHierarchicalLayout
#'
#' @export visualize_graph
#'
#' @examples
#' flow <- system.file("cwl/sbg/workflow/gatk4-wgs.json", package = "tidycwl") %>% read_cwl_json()
#' get_graph(
#'   flow %>% parse_inputs(),
#'   flow %>% parse_outputs(),
#'   flow %>% parse_steps()
#' ) %>% visualize_graph()
visualize_graph <-
  function(g, hierarchical = TRUE, direction = "LR", separation = 300,
           palette = c("#C3C3C3", "#FF8F00", "#00AAA8"),
           width = "100%", height = 600) {
    if (length(palette) < 3L) stop("Color palette must contain at least three colors")

    nodes <- g$nodes
    edges <- g$edges

    visNetwork(nodes, edges, width = width, height = height) %>%
      visNodes(borderWidth = 2) %>%
      visEdges(
        arrows = list(to = list(enabled = TRUE, scaleFactor = 0.5)),
        smooth = list(type = "cubicBezier", roundness = 0.6)
      ) %>%
      visGroups(groupname = "input", color = palette[1], shadow = list(enabled = TRUE)) %>%
      visGroups(groupname = "output", color = palette[2], shadow = list(enabled = TRUE)) %>%
      visGroups(groupname = "step", color = palette[3], shadow = list(enabled = TRUE)) %>%
      visHierarchicalLayout(
        enabled = hierarchical,
        direction = direction, levelSeparation = separation,
        sortMethod = "directed",
      )
  }
