setup(options(lifecycle_verbosity = "quiet"))

test_that("mutate_if() works for numeric columns", {
  df <- data.table(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_if.(is.numeric, ~ .x + 1)

  expect_equal(df$x, c(2,2,2))
  expect_equal(df$y, c(3,3,3))
})

test_that("mutate_at(): .vars works with select helpers in c()", {
  df <- data.table(x_start = c(1,1,1), end_x = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_at.(c(starts_with("x")), ~ .x + 1)

  expect_equal(df$x_start, c(2,2,2))
  expect_equal(df$end_x, c(2,2,2))
})

test_that("mutate_at() works for range selection", {
  df <- data.table(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_at.(c(x:y), ~ .x + 1)

  expect_equal(df$x, c(2,2,2))
  expect_equal(df$y, c(3,3,3))
})

test_that("mutate_at(): .vars works with is.numeric", {
  df <- data.table(x_start = c(1,1,1), end_x = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_at.(where(is.numeric), function(.x) .x + 1)

  expect_equal(df$x_start, c(2,2,2))
  expect_equal(df$end_x, c(3,3,3))
})

test_that("mutate_at(): .vars works with is.character", {
  df <- data.table(x_start = c(1,1,1), end_x = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_at.(where(is.character), function(.x) paste0(.x, "_append"))

  expect_equal(df$x_start, c(1,1,1))
  expect_equal(df$z, c("a_append", "a_append", "b_append"))
})

test_that("mutate_at() works with newly named columns", {
  df <- data.table(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_at.(c(x:y), list(new = function(.x) .x + 1))

  expect_named(df, c("x","y","z","x_new","y_new"))
  expect_equal(df$x_new, c(2,2,2))
  expect_equal(df$y_new, c(3,3,3))
})

test_that("mutate_all() works for all variables", {
  df <- data.table(x = c(1,1,1), y = c(2,2,2))
  df <- df %>%
    mutate_all.(function(.x) .x + 1)

  expect_equal(df$x, c(2,2,2))
  expect_equal(df$y, c(3,3,3))
})

test_that("mutate_all() works for all variables - bare function", {
  df <- data.table(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_all.(as.character)

  expect_equal(df$x, c("1","1","1"))
  expect_equal(df$y, c("2","2","2"))
})

test_that("mutate_all() works with data.frame", {
  df <- data.frame(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_all.(as.character)

  expect_equal(df$x, c("1","1","1"))
  expect_equal(df$y, c("2","2","2"))
})

test_that("mutate_across.(): .cols works with is.numeric", {
  df <- data.table(x_start = c(1,1,1), end_x = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_across.(where(is.numeric), function(.x) .x + 1)

  expect_equal(df$x_start, c(2,2,2))
  expect_equal(df$end_x, c(3,3,3))
})

test_that("mutate_across.(): modify-by-reference doesn't occur", {
  df <- data.table(x_start = c(1,1,1), end_x = c(2,2,2), z = c("a", "a", "b"))
  df %>%
    mutate_across.(where(is.numeric), ~ .x + 1)

  expect_equal(df$x_start, c(1,1,1))
  expect_equal(df$end_x, c(2,2,2))
})

test_that("mutate_across(): .cols works with is.numeric with data.frame", {
  df <- data.frame(x_start = c(1,1,1), end_x = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_across.(where(is.numeric), function(.x) .x + 1)

  expect_equal(df$x_start, c(2,2,2))
  expect_equal(df$end_x, c(3,3,3))
})

test_that("mutate_across(): .cols works with is.character", {
  df <- data.table(x_start = c(1,1,1), end_x = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_across.(where(is.character), function(.x) paste0(.x, "_append"))

  expect_equal(df$x_start, c(1,1,1))
  expect_equal(df$z, c("a_append", "a_append", "b_append"))
})

test_that("mutate_across() works with newly named columns", {
  df <- data.table(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_across.(c(x:y), list(new = function(.x) .x + 1))

  expect_named(df, c("x","y","z","x_new","y_new"))
  expect_equal(df$x_new, c(2,2,2))
  expect_equal(df$y_new, c(3,3,3))
})

test_that("mutate_across() works with newly named columns using .names with single .fn", {
  df <- data.table(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_across.(c(x:y), ~ .x + 1, .names = "new_{.col}")

  expect_named(df, c("x","y","z","new_x","new_y"))
  expect_equal(df$new_x, c(2,2,2))
  expect_equal(df$new_y, c(3,3,3))
})

test_that("mutate_across() works with newly named columns using .names with single .fn using col", {
  # This test will need to be removed when {col} is deprecated
  df <- data.table(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_across.(c(x:y), ~ .x + 1, .names = "new_{col}")

  expect_named(df, c("x","y","z","new_x","new_y"))
  expect_equal(df$new_x, c(2,2,2))
  expect_equal(df$new_y, c(3,3,3))
})

test_that("mutate_across() works with newly named columns using .names", {
  df <- data.table(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_across.(c(x:y), list(new = function(.x) .x + 1), .names = "{.fn}_{.col}")

  expect_named(df, c("x","y","z","new_x","new_y"))
  expect_equal(df$new_x, c(2,2,2))
  expect_equal(df$new_y, c(3,3,3))
})

test_that("mutate_across() works with newly named columns using .names with using fn and col", {
  # This test will need to be removed when {col} and {fn} is deprecated
  df <- data.table(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_across.(c(x:y), list(new = function(.x) .x + 1), .names = "{fn}_{col}")

  expect_named(df, c("x","y","z","new_x","new_y"))
  expect_equal(df$new_x, c(2,2,2))
  expect_equal(df$new_y, c(3,3,3))
})

test_that("mutate_across() works with newly named columns using .names w/ autonaming", {
  df <- data.table(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  df <- df %>%
    mutate_across.(c(x:y), list(new = ~ .x + 1, ~ .x + 2), .names = "{.col}_{.fn}_stuff")

  expect_named(df, c("x","y","z","x_new_stuff","y_new_stuff", "x_2_stuff", "y_2_stuff"))
  expect_equal(df$x_new_stuff, c(2,2,2))
  expect_equal(df$y_new_stuff, c(3,3,3))
  expect_equal(df$x_2_stuff, c(3,3,3))
  expect_equal(df$y_2_stuff, c(4,4,4))
})

# twiddle testing ----------------------------
test_that("mutate_if.() works with twiddle", {
  df <- data.table(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  anon_df <- df %>%
    mutate_if.(is.numeric, function(.x) .x + 1)

  twiddle_df <- df %>%
    mutate_if.(is.numeric, ~ .x + 1)

  expect_equal(anon_df, twiddle_df)
})

test_that("mutate_at.() works with twiddle", {
  df <- data.table(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  anon_df <- df %>%
    mutate_at.(c(x:y), function(.x) .x + 1)

  twiddle_df <- df %>%
    mutate_at.(c(x:y), ~ .x + 1)

  expect_equal(anon_df, twiddle_df)
})

test_that("mutate_across.() works with twiddle", {
  df <- data.table(x = c(1,1,1), y = c(2,2,2), z = c("a", "a", "b"))
  anon_df <- df %>%
    mutate_across.(c(x:y), function(.x) .x + 1)

  twiddle_df <- df %>%
    mutate_across.(c(x:y), ~ .x + 1)

  expect_equal(anon_df, twiddle_df)
})

test_that("mutate_across() works with .by", {
  test_df <- data.table::data.table(
    x = c(1,2,3),
    y = c(4,5,6),
    z = c("a","a","b"))

  results_df <- test_df %>%
    mutate_across.(c(x, y), ~ mean(.x), .by = z)

  expect_named(results_df, c("x", "y", "z"))
  expect_equal(results_df$x, c(1.5, 1.5, 3))
  expect_equal(results_df$y, c(4.5, 4.5, 6))
})

test_that("mutate_across.() works with .by enhanced selection", {
  test_df <- data.table::data.table(
    x = c(1,2,3),
    y = c(4,5,6),
    z = c("a","a","b"))

  results_df <- test_df %>%
    mutate_across.(c(x, y), ~ mean(.x), .by = where(is.character))

  expect_named(results_df, c("x", "y", "z"))
  expect_equal(results_df$x, c(1.5, 1.5, 3))
  expect_equal(results_df$y, c(4.5, 4.5, 6))
})
