/*
 *  tikzDevice, (C) 2009-2011 Charlie Sharpsteen and Cameron Bracken
 *
 *  A graphics device for R :
 *    A Computer Language for Statistical Data Analysis
 *
 *  Copyright (C) 1995, 1996  Robert Gentleman and Ross Ihaka
 *  Copyright (C) 2001-8  The R Development Core Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, a copy is available at
 *  http://www.r-project.org/Licenses/
 *
 *  The C code in this project started as a fork of:
 *    A PicTeX Device, (C) 1996 Valerio Aimale
 *
 *
 *  "If I have seen further, it is only by standing on
 *   the shoulders of giants."
 *
 *   -I. Newton
 *
*/

/********************************************************************/

/*
 * NOTE:
 *   This is the first effort of dyed-in-the-wool Fortran programmers
 *   to write C code. Hence the comments in this file will make many
 *   observations that may seem obvious or inane. There also may be a
 *   generous amount of snide comments concerning the syntax of the 
 *   C language.
*/

/* 
 * Function prototypes are defined in here. Apparently in C
 * it is absolutely necessary for function definitions to appear 
 * BEFORE they are called by other functions. Hence many source code
 * files do not present code in the order in which that code
 * is used. Using a header file with function declarations allows
 * the programmer to order the code in any sequence they choose.
*/

/*
 * This header also includes other header files describing functions 
 * provided by the R language.
*/
#include "tikzDevice.h"

// We are writing to files so we need stdio.h
#include <stdio.h>

/*
 * Main entry point from the R environment, called by the R function
 * tikz() to open a new TikZ graphics device.
*/
SEXP TikZ_StartDevice ( SEXP args ){

  /*
   * Make sure the version number of the R running this
   * routine is compatible with the version number of 
   * the R that compiled this routine.
  */
  R_GE_checkVersionOrDie(R_GE_version);

  /* Declare local variabls for holding the components of the args SEXP */
  const char *fileName;
  const char *bg, *fg;
  double width, height;
  Rboolean standAlone, bareBones;
  const char *documentDeclaration, *packages, *footer;
  double baseSize;
  Rboolean console, sanitize, onefile;

  /* 
   * pGEDevDesc is a variable provided by the R Graphics Engine
   * that represents a graphics device to the rest of the R system.
   * It contains one important componant of type pDevDesc
   * which contains information specific to the implementation of
   * the TikZ Device. The creation and initialization of this component
   * is the main task of this routine.
  */
  pGEDevDesc tikzDev;


  /* Retrieve function arguments from input SEXP. */

  
  /*
   * Skip first argument. It holds the name of the R function
   * that called this C routine.
  */ 
  args = CDR(args);

  /* Recover file name. */
  fileName = translateChar(asChar(CAR(args)));
  /* Advance to next argument stored in the args SEXP. */
  args = CDR(args);

  /* Recover figure dimensions. */
  /* For now these are assumed to be in inches. */
  width = asReal(CAR(args)); args = CDR(args);
  height = asReal(CAR(args)); args = CDR(args);

  onefile = asLogical(CAR(args)); args = CDR(args);
  
  /* Recover initial background and foreground colors. */
  bg = CHAR(asChar(CAR(args))); args = CDR(args);
  fg = CHAR(asChar(CAR(args))); args = CDR(args);

  /* Recover the base fontsize */
  baseSize = asReal(CAR(args)); args = CDR(args);

  /* 
   * Set the standAlone parameter which specifies if the TikZ
   * pictures generated by this device should be wrapped in their
   * own LaTeX Document
  */
  standAlone = asLogical(CAR(args)); args = CDR(args);

  /* 
   * Set the bareBones parameter which specifies if TikZ code 
   * should be output directly without wrapping it a LaTeX document
   * or the tikzpicture environment.
  */
  bareBones = asLogical(CAR(args)); args = CDR(args);
  
  /* Grab the latex header and footers*/
  documentDeclaration = CHAR(asChar(CAR(args))); args = CDR(args);
  packages = CHAR(asChar(CAR(args))); args = CDR(args);
  footer = CHAR(asChar(CAR(args))); args = CDR(args);

  /*
   * Should the output be sent to the R console? An null file name also
   * indicates console output.
   */
  console = asLogical(CAR(args)); args = CDR(args);
  if ( fileName[0] == '\0' )
    console = TRUE;

  /*
   * Should text strings passed to the plotting device be sent
   * to a sanitization function- i.e. to provide automatic
   * escaping of TeX special characters such as %,_,\, etc?
  */ 
  sanitize = asLogical(CAR(args)); args = CDR(args);

  /*
   * See the definition of tikz_engine in tikzDevice.h
   */
  int engine = asInteger(CAR(args));

  /* Ensure there is an empty slot avaliable for a new device. */
  R_CheckDeviceAvailable();

  BEGIN_SUSPEND_INTERRUPTS{

    /* 
     * The pDevDesc variable specifies the funtions and components 
     * that describe the specifics of this graphics device. After
     * setup, this information will be incorporated into the pGEDevDesc
     * variable tikzDev.
    */ 
    pDevDesc deviceInfo;

    /* 
     * Create the deviceInfo variable. If this operation fails, 
     * a 0 is returned in order to cause R to shut down due to the
     * possibility of corrupted memory.
    */
    if( !( deviceInfo = (pDevDesc) calloc(1, sizeof(DevDesc))) ) {
      return 0;
    }

    /*
     * Call setup routine to initialize deviceInfo and associate
     * R graphics function hooks with the appropriate C routines
     * in this file.
    */
    if( !TikZ_Setup( deviceInfo, fileName, width, height, onefile, bg, fg, baseSize,
        standAlone, bareBones, documentDeclaration, packages,
        footer, console, sanitize, engine ) ){
      /* 
       * If setup was unsuccessful, destroy the device and return
       * an error message.
      */
      free( deviceInfo );
      error("TikZ device setup was unsuccessful!");
    }

    /* Create tikzDev as a Graphics Engine device using deviceInfo. */
    tikzDev = GEcreateDevDesc( deviceInfo );

    /*
     * Register the device as an avaiable graphics device in the R
     * Session.  The user will now see a device labeled "tikz output"
     * when running functions such as dev.list().
    */ 
    GEaddDevice2( tikzDev, "tikz output" );

  } END_SUSPEND_INTERRUPTS;


  return R_NilValue;

}


/*
 * This function is responsible for initializing device parameters
 * contained in the variable deviceInfo. It returns a true or false
 * value depending on the success of initialization operations. The
 * static keyword means this function can only be seen by other functions
 * in this file. A better choice for the keyword might have been something
 * like "private"... 
*/

static Rboolean TikZ_Setup(
  pDevDesc deviceInfo,
  const char *fileName,
  double width, double height, Rboolean onefile,
  const char *bg, const char *fg, double baseSize,
  Rboolean standAlone, Rboolean bareBones,
  const char *documentDeclaration,
  const char *packages, const char *footer, 
  Rboolean console, Rboolean sanitize, int engine ){

  /* 
   * Create tikzInfo, this variable contains information which is
   * unique to the implementation of the TikZ Device. The deviceInfo
   * variable contains a slot into which tikzInfo can be placed so that
   * this information persists and is retrievable during the lifespan
   * of this device.
   *
   * More information on the components of the deviceInfo structure,
   * which is a pointer to a DevDesc variable, can be found under
   * struct _DevDesc in the R header file GraphicsDevice.h
   *
   * tikzInfo is a structure which is defined in the file tikzDevice.h
  */
  tikzDevDesc *tikzInfo;

  /* 
   * Initialize tikzInfo, return false if this fails. A false return
   * value will cause the whole device initialization routine to fail.
  */
  if( !( tikzInfo = (tikzDevDesc *) malloc(sizeof(tikzDevDesc)) ) ){
    return FALSE;
  }

  /* Copy TikZ-specific information to the tikzInfo variable. */
  if ( onefile == FALSE ) {
    /*
     * Hopefully 10 extra digits will be enough for storing incrementing file
     * numbers.
     */
    tikzInfo->outFileName = calloc_x_strlen(fileName, 10);
    tikzInfo->originalFileName = calloc_strcpy(fileName);
  } else {
    tikzInfo->outFileName = calloc_strcpy(fileName);
  }
  tikzInfo->engine = engine;
  tikzInfo->rasterFileCount = 1;
  tikzInfo->debug = DEBUG;
  tikzInfo->standAlone = standAlone;
  tikzInfo->bareBones = bareBones;
  tikzInfo->oldFillColor = 0;
  tikzInfo->oldDrawColor = 0;
  tikzInfo->stringWidthCalls = 0;

  tikzInfo->documentDeclaration = calloc_strcpy(documentDeclaration);
  tikzInfo->packages = calloc_strcpy(packages);
  tikzInfo->footer = calloc_strcpy(footer);

  tikzInfo->console = console;
  tikzInfo->sanitize = sanitize;
  tikzInfo->clipState = TIKZ_NO_CLIP;
  tikzInfo->pageState = TIKZ_NO_PAGE;
  tikzInfo->onefile = onefile;
  tikzInfo->pageNum = 1;

  /* Incorporate tikzInfo into deviceInfo. */
  deviceInfo->deviceSpecific = (void *) tikzInfo;

  /* 
   * These next statements define the capabilities of the device.
   * These capabilities include:
   *  -Device/user interaction
   *  -Gamma correction
   *  -Clipping abilities
   *  -UTF8 support
   *  -Text justification/alignment abilities
  */

  /* 
   * Define the gamma factor- used to adjust the luminosity of an image. 
   * Set to 1 since there is no gamma correction in the TikZ device. Also,
   * canChangeGamma is set to FALSE to disallow user adjustment of this
   * default.
  */
  deviceInfo->startgamma = 1;
  deviceInfo->canChangeGamma = FALSE;

  /*
   * canHAdj is an integer specifying the level of horizontal adjustment
   * or justification provided by this device. Currently set to 1 as this
   * is implemented by having the device insert /raggedleft, /raggedright
   * and /centering directives.
   *
   * Level 2 represents support for continuous variation between left aligned 
   * and right aligned- this is certainly possible in TeX but would take some
   * thought to implement.
  */
  deviceInfo->canHAdj = 1;

  /*
   * useRotatedTextInContour specifies if the text function along with
   * rotation parameters should be used over Hershey fonts when printing
   * contour plot labels. As one of the primary goals of this device
   * is to unify font choices, this value is set to true.
  */
  deviceInfo->useRotatedTextInContour = TRUE; 

  /*
   * canClip specifies whether the device implements routines for trimming
   * plotting output such that it falls within a rectangular clipping area.
  */
  deviceInfo->canClip = TRUE;

  /*
   * These next parameters speficy if the device reacts to keyboard and 
   * mouse events. Since this device outputs to a file, not a screen window, 
   * these actions are disabled.
  */
  deviceInfo->canGenMouseDown = FALSE;
  deviceInfo->canGenMouseMove = FALSE;
  deviceInfo->canGenMouseUp = FALSE;
  deviceInfo->canGenKeybd = FALSE;

  /* 
   * This parameter specifies whether the device is set up to handle UTF8
   * characters. This makes a difference in the complexity of the text
   * handling functions that must be built into the device. If set to true
   * both hook functions textUTF8 and strWidthUTF8 must be implemented.
   * Compared to ASCII, which only has 128 character values, UTF8 has
   * thousands.
   *
   * Version 0.6.0 of tikzDevice gained the ability to calculate metrics for
   * UTF8 encoded strings and characters. Those calculations are not done here
   * in the C code but implemented through the magical callback to R. On the R
   * level, we determine automatically is a string contains multibyte UTF8
   * characters and then use XeLaTeX.  Bottom line is, even though hasTextUTF8
   * is FALSE we can still print UTF8 characters and we dont need a separate
   * text handling function for UTF8 characters (thank god).
   * 
   * wantSymbolUTF8 indicates if mathematical symbols should be sent to
   * the device as UTF8 characters.  These can be handled in the same way as
   * normal UTF8 text and so wantSymbolUTF8 is TRUE.
  */
  deviceInfo->hasTextUTF8 = FALSE;
  switch (tikzInfo->engine) {
    case pdftex:
      deviceInfo->wantSymbolUTF8 = FALSE;
      break;
    case xetex:
    case luatex:
      deviceInfo->wantSymbolUTF8 = TRUE;
      break;
  }

#if R_GE_version >= 9
  /* Added in 2.14.0 for `dev.capabilities`. In all cases 0 means NA (unset). */
  deviceInfo->haveTransparency = 2;  /* 1 = no, 2 = yes */
  deviceInfo->haveTransparentBg = 2; /* 1 = no, 2 = fully, 3 = semi */
  deviceInfo->haveRaster = 2;        /* 1 = no, 2 = yes, 3 = except for missing values */
  deviceInfo->haveCapture = 1;       /* 1 = no, 2 = yes */
  deviceInfo->haveLocator = 1;       /* 1 = no, 2 = yes */
#endif

  /*
   * Initialize device parameters. These concern properties such as the 
   * plotting canvas size, the initial foreground and background colors and 
   * the initial clipping area. Other parameters related to fonts and text 
   * output are also included.
  */

  /*
   * Set canvas size. The bottom left corner is considered the origin and 
   * assigned the value of 0pt, 0pt. The upper right corner is assigned by 
   * converting the specified height and width of the device to points.
  */
  deviceInfo->bottom = 0;
  deviceInfo->left = 0;
  deviceInfo->top = dim2dev( height );
  deviceInfo->right = dim2dev( width );

  /* Set default character size in pixels. */
  deviceInfo->cra[0] = 0.9 * baseSize;
  deviceInfo->cra[1] = 1.2 * baseSize;

  /* Set initial font. */
  deviceInfo->startfont = 1;

  /* Set base font size. */
  deviceInfo->startps = baseSize;

  /*
   * Apparently these are supposed to center text strings over the points at
   * which they are plotted.
   *
   * Values cribbed from devPS.c in the R source. In paticular, setting
   * `yLineBias` to 0 causes text in the margins of an x axis to recieve more
   * leading that text in the margins of a y axis.
  */
  deviceInfo->xCharOffset = 0.4900;
  deviceInfo->yCharOffset = 0.3333;
  deviceInfo->yLineBias = 0.2;

  /* Specify the number of inches per pixel in the x and y directions. */
  deviceInfo->ipr[0] = 1/dim2dev(1);
  deviceInfo->ipr[1] = 1/dim2dev(1);

  /* Set initial foreground and background colors. */
  deviceInfo->startfill = R_GE_str2col( bg );
  deviceInfo->startcol = R_GE_str2col( fg );

  /* Set initial line type. */
  deviceInfo->startlty = 0;


  /* 
   * Connect R graphic function hooks to TikZ Routines implemented in this
   * file. Each routine performs a specific function such as adding text, 
   * drawing a line or reporting/adjusting the status of the device.
  */

  /* Utility routines. */
  deviceInfo->close = TikZ_Close;
  deviceInfo->newPage = TikZ_NewPage;
  deviceInfo->clip = TikZ_Clip;
  deviceInfo->size = TikZ_Size;

  /* Text routines. */
  deviceInfo->metricInfo = TikZ_MetricInfo;
  deviceInfo->strWidth = TikZ_StrWidth;
  deviceInfo->text = TikZ_Text;

  /* Drawing routines. */
  deviceInfo->line = TikZ_Line;
  deviceInfo->circle = TikZ_Circle;
  deviceInfo->rect = TikZ_Rectangle;
  deviceInfo->polyline = TikZ_Polyline;
  deviceInfo->polygon = TikZ_Polygon;
  deviceInfo->path = TikZ_Path;

  /* 
   * Raster Routines.  Currently implemented as stub functions to
   * avoid nasty crashes. 
  */
  deviceInfo->raster = TikZ_Raster;
  deviceInfo->cap = TikZ_Cap;

  /* Dummy routines. These are mainly used by GUI graphics devices. */
  deviceInfo->activate = TikZ_Activate;
  deviceInfo->deactivate = TikZ_Deactivate;
  deviceInfo->locator = TikZ_Locator;
  deviceInfo->mode = TikZ_Mode;

  /*
   * If outputting to a single file, call TikZ_Open to create and initialize
   * the output. For multiple files, each call to TikZ_NewPage will set up a
   * new file.
   */
  if( tikzInfo->onefile )
    if( !TikZ_Open(deviceInfo) )
      return FALSE;

  return TRUE;
}


/*==============================================================================
                            Core Graphics Routines
             Implementaion of an R Graphics Device as Defined by:
                               GraphicsDevice.h
==============================================================================*/

/*
 * Routines for handling device state:
 *
 * - Open
 * - Close
 * - Newpage
 * - Clip
 * - Size
 */
static Rboolean TikZ_Open( pDevDesc deviceInfo )
{
  /*
   * Shortcut pointers to variables of interest.  It seems like there HAS to be
   * a more elegent way of accesing these...
  */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;

  /* If creating multiple files, add the page number to the filename. */
  if ( !tikzInfo->onefile )
    sprintf(tikzInfo->outFileName, tikzInfo->originalFileName, tikzInfo->pageNum);

  if ( !tikzInfo->console )
    if ( !(tikzInfo->outputFile = fopen(R_ExpandFileName(tikzInfo->outFileName), "w")) )
      return FALSE;

  /* Print header comment */
  Print_TikZ_Header( tikzInfo );

  /* Header for a standalone LaTeX document*/
  if(tikzInfo->standAlone == TRUE){
    printOutput(tikzInfo,"%s",tikzInfo->documentDeclaration);
    printOutput(tikzInfo,"%s",tikzInfo->packages);
    printOutput(tikzInfo,"\\begin{document}\n\n");
  }

  return TRUE;
}

static void TikZ_Close( pDevDesc deviceInfo){

  /* Shortcut pointers to variables of interest. */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;

  if ( tikzInfo->clipState == TIKZ_FINISH_CLIP ) {
    printOutput(tikzInfo, "\\end{scope}\n");
    tikzInfo->clipState = TIKZ_NO_CLIP;
  }

  /* End the tikz environment if we're not doing a bare bones plot. */
  if( tikzInfo->bareBones != TRUE && tikzInfo->pageState == TIKZ_FINISH_PAGE ) {
    printOutput(tikzInfo, "\\end{tikzpicture}\n");
    tikzInfo->pageState = TIKZ_NO_PAGE;
  }

  /* Close off the standalone document*/
  if ( tikzInfo->standAlone == TRUE ) {
    printOutput(tikzInfo, tikzInfo->footer);
    printOutput(tikzInfo,"\n\\end{document}\n");
  }

  if(tikzInfo->debug == TRUE) 
    printOutput(tikzInfo,
      "%% Calculated string width %d times\n",
      tikzInfo->stringWidthCalls);

  /* Close the file and destroy the tikzInfo structure. */
  if(tikzInfo->console == FALSE)
    fclose(tikzInfo->outputFile);

  /* Deallocate pointers */
  free(tikzInfo->outFileName);
  if ( !tikzInfo->onefile )
    free(tikzInfo->originalFileName);

  const_free(tikzInfo->documentDeclaration);
  const_free(tikzInfo->packages);
  const_free(tikzInfo->footer);

  free(tikzInfo);
}

static void TikZ_NewPage( const pGEcontext plotParams, pDevDesc deviceInfo )
{
  /* Shortcut pointers to variables of interest. */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;

  if ( tikzInfo->clipState == TIKZ_FINISH_CLIP ) {
    printOutput(tikzInfo, "\\end{scope}\n");
    tikzInfo->clipState = TIKZ_NO_CLIP;
  }

  if ( tikzInfo->pageState == TIKZ_FINISH_PAGE ) {
    if ( !tikzInfo->bareBones )
      printOutput(tikzInfo, "\\end{tikzpicture}\n");

    if ( !tikzInfo->onefile ) {
      if( tikzInfo->standAlone )
        printOutput(tikzInfo,"\n\\end{document}\n");

      if( !tikzInfo->console )
        fclose(tikzInfo->outputFile);
    }
  }

  /*
   * Color definitions do not persist accross tikzpicture environments. Set the
   * cached colors to "impossible" values so that the first drawing operation
   * inside the next environment will trigger a re-definition of colors.
   */
  tikzInfo->oldFillColor = -999;
  tikzInfo->oldDrawColor = -999;

  /*
   * Setting this flag will cause the `TikZ_CheckState` function to emit the
   * code required to begin a new `tikzpicture` enviornment. `TikZ_CheckState`
   * is called by every graphics function that generates visible output.
   */
  tikzInfo->pageState = TIKZ_START_PAGE;
}

static void TikZ_Clip( double x0, double x1,
    double y0, double y1, pDevDesc deviceInfo )
{
  /* Shortcut pointers to variables of interest. */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;

  deviceInfo->clipBottom = y0;
  deviceInfo->clipLeft = x0;
  deviceInfo->clipTop = y1;
  deviceInfo->clipRight = x1;

  if ( tikzInfo->clipState == TIKZ_FINISH_CLIP )
    printOutput(tikzInfo, "\\end{scope}\n");

  /*
   * Color definitions do not persist accross scopes. Set the cached colors to
   * "impossible" values so that the first drawing operation inside the scope
   * will trigger a re-definition of colors.
   */
  tikzInfo->oldFillColor = -999;
  tikzInfo->oldDrawColor = -999;

  /*
   * Setting this flag will cause the `TikZ_CheckState` function to emit the
   * code required to begin a new clipping scope. `TikZ_CheckState` is called
   * by every graphics function that generates visible output.
   */
  tikzInfo->clipState = TIKZ_START_CLIP;
}

static void TikZ_Size( double *left, double *right,
    double *bottom, double *top, pDevDesc deviceInfo){
  
  /* Return canvas size. */
  *bottom = deviceInfo->bottom;
  *left = deviceInfo->left;
  *top = deviceInfo->top;
  *right = deviceInfo->right;

}


/*
 * Routines for calculating text metrics:
 *
 * - MetricInfo
 * - StrWidth
 */

/*
 * This function is supposed to calculate character metrics (such as raised 
 * letters, stretched letters, ect). Currently the TikZ device does not 
 * perform such functions, so this function returns the default metrics
 * the Quartz device uses when it can't think of anything else.
 * 
 * The fact that this function is not implemented is the most likely cause
 * for the *vertical* alignment of text strings being off. This shortcoming
 * is most obvious when plot legends are created.
 *
*/ 
static void TikZ_MetricInfo(int c, const pGEcontext plotParams,
    double *ascent, double *descent, double *width, pDevDesc deviceInfo ){


  /* Shortcut pointers to variables of interest. */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;

  if (tikzInfo->engine == pdftex) {
    /*
     * PdfTeX can only deal with ASCII characters, check the character code c
     * to see if it falls outside the range of printable characters which are:
     * 32-126
     */
    if( c < 32 || c > 126 ){
      /* Non-printable character. Set metrics to zero and return. */
      *ascent = 0.0;
      *descent = 0.0;
      *width = 0.0;
      return;
    }
  }

  // Calculate font scaling factor.
  double fontScale = ScaleFont( plotParams, deviceInfo );

  // Prepare to call back to R in order to retrieve character metrics.
  SEXP namespace;
  PROTECT( namespace = TIKZ_NAMESPACE );

  // Call out to R to retrieve the latexParseCharForMetrics function.
  // Note: this code will eventually call a different function that provides
  // caching of the results. Right now we're directly calling the function
  // that activates LaTeX.
  SEXP metricFun = findFun(install("getLatexCharMetrics"), namespace);

  SEXP RCallBack;
  PROTECT( RCallBack = allocVector(LANGSXP,7) );

  // Place the function into the first slot of the SEXP.
  SETCAR( RCallBack, metricFun );

  // Place the character code into the second slot of the SEXP.
  SETCADR( RCallBack, ScalarInteger( c ) );
  SET_TAG( CDR( RCallBack ), install("charCode") );

  // Pass graphics parameters cex and fontface.
  SETCADDR( RCallBack,  ScalarReal( fontScale ) );
  SET_TAG( CDDR( RCallBack ), install("cex") );
  SETCADDDR( RCallBack,  ScalarInteger( plotParams->fontface ) );
  SET_TAG( CDR(CDDR( RCallBack )), install("face") );

  /*
   * Set the TeX engine based on tikzInfo
   */
  switch (tikzInfo->engine) {
    case pdftex:
      SETCAD4R(RCallBack, mkString("pdftex"));
      break;
    case xetex:
      SETCAD4R(RCallBack, mkString("xetex"));
      break;
    case luatex:
      SETCAD4R(RCallBack, mkString("luatex"));
      break;
  }
  SET_TAG(CDDR(CDDR(RCallBack)), install("engine"));

  SETCAD4R(CDR(RCallBack), mkString(tikzInfo->documentDeclaration));
  SET_TAG(CDR(CDDR(CDDR(RCallBack))), install("documentDeclaration"));

  SETCAD4R(CDDR(RCallBack), mkString(tikzInfo->packages));
  SET_TAG(CDDR(CDDR(CDDR(RCallBack))), install("packages"));

  SEXP RMetrics;
  PROTECT( RMetrics = eval(RCallBack, namespace) );

  // Recover the metrics.
  *ascent = REAL(RMetrics)[0];
  *descent = REAL(RMetrics)[1];
  *width = REAL(RMetrics)[2];

  if( tikzInfo->debug == TRUE )
  printOutput( tikzInfo, "%% Calculated character metrics. ascent: %f, descent: %f, width: %f\n",
    *ascent, *descent, *width);

  UNPROTECT(3);

  return;

}

/*
 * This function is supposed to calculate the plotted with, in device raster
 * units of an arbitrary string. This is perhaps the most difficult function
 * that a device needs to implement. Calculating the exact with of a string 
 * is especially tricky because this device is designed to print characters 
 * in whatever font is being used in the the TeX document. The end font that
 * the user decides to typeset their document in may also be unknown to the
 * device. The problem is further complicated by the fact that TeX strings 
 * can be used directly in annotations.  For example the string \textit{x} 
 * literaly has 10 characters but when it is actually typeset it only has
 * one. Given this difficulty the function currently writes the string
 * to a temporary file and calls LaTeX in order to obtain an authoratative
 * measure of the string width.
 *
 * There is a rediculous amount of overhead involved with this process and
 * the number of calls required to obtion widths for common things such as
 * all the number s on a plot axes can easily add up to several seconds.
 *
 * However, if we do not perform string width calculation R is unable to
 * properly align text in the plots R. This is something that LaTeX and
 * TikZ should actually be taking care of by themselves but the current
 * graphics system does not allow for this.
 *
 * Given that we need text strings to be aligned for good output, we are
 * stuck using this inefficient hybrid system untill we think of something
 * better.
 *
*/
static double TikZ_StrWidth( const char *str,
    const pGEcontext plotParams, pDevDesc deviceInfo ){
      
  /* Shortcut pointers to variables of interest. */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;

  // Calculate font scaling factor.
  double fontScale = ScaleFont( plotParams, deviceInfo );

  /*
   * New string width calculation method: call back to R
   * and run the R function getLatexStrWidth.
   *
   * This used to be implemented as a C function, but
   * the nuts and bolts were re-implemented back
   * on the R side of this package. There seems to
   * have been no major performance penalty associated
   * with doing this.
   *
   * Why was it done?
   *
   * - Windows and Linux did not suppress the output
   *   of the C system call to LaTeX which resulted 
   *   in spam and lag. In the case of Windows, a
   *   whole mess of CMD windows were spawned which
   *   eventually crashed the system.
   *
   * - Using R's system() call we gain a level of
   *   abstraction that works accross all platforms.
   *   We can also use functions like tempdir() to
   *   do the dirty work somewhere where the user
   *   won't have to clean it up.
   *
   * - If a LaTeX parser ever gets implemented, it
   *   will probably be easiest to implement it in
   *   R. If a LaTeX parser ever gets stolen from
   *   something like python's matplotlib, R will
   *   probably provide the interface. Therefore
   *   a callback to R may be necessary anyway.
   *
   * - Having C code called by R call R code is 
   *   fucking wicked.
   *
  */

  /*
   * Find the namespace of the TikZ package.
   */
  SEXP namespace;
  PROTECT( namespace = TIKZ_NAMESPACE );

  // Call out to R to retrieve the getLatexStrWidth function.
  SEXP widthFun = findFun(install("getLatexStrWidth"), namespace);

  /*
   * Create a SEXP that will be the R function call. The SEXP will have five
   * components- the R function being called, the string being passed, the
   * current value of the graphics parameters cex and fontface and the TeX
   * engine to be used. Therefore it is allocated as a  LANGSXP vector of
   * length 5. This is done inside a PROTECT() function to keep the R garbage
   * collector from saying "Hmmm... what's this? Looks like noone is using it
   * so I guess I will nuke it."
  */
  SEXP RCallBack;
  PROTECT( RCallBack = allocVector(LANGSXP, 7) );

  // Place the function into the first slot of the SEXP.
  SETCAR( RCallBack, widthFun );

  //If using the sanitize option call back to R for the sanitized string
  char *cleanString = NULL;
  if(tikzInfo->sanitize == TRUE){
    cleanString = Sanitize( str );
    // Place the sanitized string into the second slot of the SEXP.
    SETCADR( RCallBack, mkString( cleanString ) );
    
  }else{
    
    // Place the string into the second slot of the SEXP.
    SETCADR( RCallBack, mkString( str ) );
    
  }
  // Tag the string with a name, this name coressponds to the
  // dummy argument of the R function getLatexStringWidth.
  SET_TAG( CDR( RCallBack ), install("texString") );

  // Pass graphics parameters cex and fontface.
  SETCADDR( RCallBack,  ScalarReal( fontScale ) );
  SET_TAG( CDDR( RCallBack ), install("cex") );
  SETCADDDR( RCallBack,  ScalarInteger( plotParams->fontface ) );
  SET_TAG( CDR(CDDR( RCallBack )), install("face") );

  /*
   * Set the TeX engine based on tikzInfo
   */
  switch (tikzInfo->engine) {
    case pdftex:
      SETCAD4R(RCallBack, mkString("pdftex"));
      break;
    case xetex:
      SETCAD4R(RCallBack, mkString("xetex"));
      break;
    case luatex:
      SETCAD4R(RCallBack, mkString("luatex"));
      break;
  }
  SET_TAG(CDDR(CDDR(RCallBack)), install("engine"));

  SETCAD4R(CDR(RCallBack), mkString(tikzInfo->documentDeclaration));
  SET_TAG(CDR(CDDR(CDDR(RCallBack))), install("documentDeclaration"));

  SETCAD4R(CDDR(RCallBack), mkString(tikzInfo->packages));
  SET_TAG(CDDR(CDDR(CDDR(RCallBack))), install("packages"));

  /*
   * Call the R function, capture the result.
   * PROTECT may not be necessary here, but I'm doing
   * it just in case the SEXP holds a pointer to an
   * R function return value that the garbage collector
   * decides to nuke.
  */
  SEXP RStrWidth;
  PROTECT( RStrWidth = eval(RCallBack, namespace) );

  /*
   * Why REAL()[0] instead of asReal(CAR())? I have no fucking
   * clue...
   *
   * After browsing Rinternals.h, the location where SEXPs and
   * their access functions are defined, I have an explanation
   * that seems plausible.
   *
   * Since getLatexStrWidth returns a single variable of a single
   * type, it is returned as a vector SEXP. The value can be
   * extracted to a C variable by coercing the SEXP to real and
   * then accessing the first element of the resulting array.
   *
   * When a R function returns, or passes in the case of the 
   * .External call that leads into all of this code, a collection
   * of *different* objects they are passed as a list instead of
   * a vector. Therefore CAR is needed to access the list followed
   * by coercion using asReal().
   *
   * Seems like this explains what is going on here, although it
   * is just a wild guess on my part. Maybe I should post to
   * r-devel and ask for clarification...
   *
  */
  double width = REAL(RStrWidth)[0];

  /*
   * Since we called PROTECT thrice, we must call UNPROTECT
   * and pass the number 3.
   */
  UNPROTECT(3);
  if(tikzInfo->sanitize == TRUE){ free(cleanString); }
  
  /*Show only for debugging*/
  if(tikzInfo->debug == TRUE) 
    printOutput(tikzInfo,
      "%% Calculated string width of %s as %f\n",str,width);
  
  /*
   * Increment the number of times this function has been called
   * Used for debugging purposes.
   *
  */
  tikzInfo->stringWidthCalls++;

  return(width);
    
}


/*
 * Output routines
 *
 * - Text
 * - Circle
 * - Rectangle
 * - Line
 * - Polyline
 * - Polygon
 * - Path
 * - Raster
 */

/*
 * This function should plot a string of text at coordinates x and y with
 * a rotation value specified by rot and horizontal alignment specified by
 * hadj. Additional parameters such as color, font type, font style, line
 * height and font size are specified in the pGEcontext variable plotParams.
 *
 * The rotation value is given in degrees.
*/
static void TikZ_Text( double x, double y, const char *str,
    double rot, double hadj, const pGEcontext plotParams, 
    pDevDesc deviceInfo){
  
  /* Shortcut pointers to variables of interest. */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;
  
  double tol = 0.01;
  
  // Append font face commands depending on which font R is using.
  char *tikzString = calloc_x_strlen(str, 20);

  switch( plotParams->fontface ){
  
    case 2:
      // R is requesting bold font.
      strcat( tikzString, "\\bfseries " );
      break;

    case 3:
      // R is requesting italic font.
      strcat( tikzString, "\\itshape " );
      break;

    case 4:
      // R is requesting bold italic font.
      strcat( tikzString, "\\bfseries\\itshape " );
      break;

  } // End font face switch.

  // Form final output string.
  strcat( tikzString, str );

  // Calculate font scaling factor.
  double fontScale = ScaleFont( plotParams, deviceInfo );
  
  /*Show only for debugging*/
  if(tikzInfo->debug == TRUE) 
    printOutput(tikzInfo,
      "%% Drawing node at x = %f, y = %f\n",
      x,y);

  TikZ_CheckState(deviceInfo);
  TikZ_DefineColors(plotParams, deviceInfo, DRAWOP_DRAW);

  /* Start a node for the text, open an options bracket. */
  printOutput(tikzInfo,"\n\\node[text=drawColor");
  /* FIXME: Should bail out of this function early if text is fully transparent */
  if( !R_OPAQUE(plotParams->col) )
    printOutput(tikzInfo, ",text opacity=%4.2f", R_ALPHA(plotParams->col)/255.0);

  /* Rotate the text if desired. */
  if( rot != 0 )
    printOutput(tikzInfo, ",rotate=%6.2f", rot );

  /* End options, print coordinates and string. */
  printOutput(tikzInfo, ",anchor=");

  //Justify the text
  if(fabs(hadj - 0.0) < tol){
    //Left Justified
    printOutput(tikzInfo, "base west");
  }
  if(fabs(hadj - 0.5) < tol){
    //Center Justified
    printOutput(tikzInfo, "base");
  }
  if(fabs(hadj - 1) < tol){
    //Right Justified
    printOutput(tikzInfo, "base east");
  }

  printOutput(tikzInfo,
    ",inner sep=0pt, outer sep=0pt, scale=%6.2f] at (%6.2f,%6.2f) {",
    fontScale, x, y);

  char *cleanString = NULL;
  if(tikzInfo->sanitize == TRUE){
    //If using the sanitize option call back to R for the sanitized string
    cleanString = Sanitize( tikzString );
  	if(tikzInfo->debug == TRUE)
    	printOutput(tikzInfo,"\n%% Sanatized %s to %s\n",tikzString,cleanString);
    printOutput(tikzInfo, "%s};\n", cleanString);
  }else{
    printOutput(tikzInfo, "%s};\n", tikzString);
  }

  /* 
   * Since we no longer need tikzString, 
   * we should free the memory that it is being stored in.
  */
  free( tikzString );
  if(tikzInfo->sanitize == TRUE){ free( cleanString ); }

  /* 
   * Add a small red marker to indicate the 
   * point the text string is being aligned to.
  */
  if( DEBUG == TRUE )
    printOutput(tikzInfo, 
      "\n\\draw[color=red, fill=red] (%6.2f,%6.2f) circle (0.5pt);\n", 
      x, y);

}


static void TikZ_Circle( double x, double y, double r,
    const pGEcontext plotParams, pDevDesc deviceInfo){

  /* Shortcut pointers to variables of interest. */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;
  TikZ_DrawOps ops = TikZ_GetDrawOps(plotParams);

  /*Show only for debugging*/
  if(tikzInfo->debug == TRUE) 
    printOutput(tikzInfo,
      "%% Drawing Circle at x = %f, y = %f, r = %f\n",
      x,y,r);

  TikZ_CheckState(deviceInfo);
  TikZ_DefineColors(plotParams, deviceInfo, ops);

  /* Start drawing, open an options bracket. */
  printOutput(tikzInfo,"\n\\path[");
  TikZ_WriteDrawOptions(plotParams, deviceInfo, ops);

  /* End options, print coordinates. */
  printOutput(tikzInfo, "] (%6.2f,%6.2f) circle (%6.2f);\n",
    x,y,r);
}

static void TikZ_Rectangle( double x0, double y0,
    double x1, double y1, const pGEcontext plotParams, pDevDesc deviceInfo){

  /* Shortcut pointers to variables of interest. */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;
  TikZ_DrawOps ops = TikZ_GetDrawOps(plotParams);

  /*Show only for debugging*/
  if(tikzInfo->debug == TRUE) 
    printOutput(tikzInfo,
      "%% Drawing Rectangle from x0 = %f, y0 = %f to x1 = %f, y1 = %f\n",
      x0,y0,x1,y1);

  TikZ_CheckState(deviceInfo);
  TikZ_DefineColors(plotParams, deviceInfo, ops);

  /* Start drawing, open an options bracket. */
  printOutput(tikzInfo,"\n\\path[");
  TikZ_WriteDrawOptions(plotParams, deviceInfo, ops);

  /* End options, print coordinates. */
  printOutput(tikzInfo, 
    "] (%6.2f,%6.2f) rectangle (%6.2f,%6.2f);\n",
    x0,y0,x1,y1);

}


static void TikZ_Line( double x1, double y1,
    double x2, double y2, const pGEcontext plotParams, pDevDesc deviceInfo){

  /* Shortcut pointers to variables of interest. */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;
  TikZ_DrawOps ops = TikZ_GetDrawOps(plotParams);

  /*Show only for debugging*/
  if(tikzInfo->debug == TRUE) 
    printOutput(tikzInfo,
      "%% Drawing line from x1 = %10.4f, y1 = %10.4f to x2 = %10.4f, y2 = %10.4f\n",
      x1,y1,x2,y2);

  TikZ_CheckState(deviceInfo);
  TikZ_DefineColors(plotParams, deviceInfo, ops);

  /* Start drawing a line, open an options bracket. */
  printOutput(tikzInfo,"\n\\path[");
  TikZ_WriteDrawOptions(plotParams, deviceInfo, ops);

  /* End options, print coordinates. */
  printOutput(tikzInfo, "] (%6.2f,%6.2f) -- (%6.2f,%6.2f);\n",
    x1,y1,x2,y2);

}


static void TikZ_Polyline( int n, double *x, double *y,
    pGEcontext plotParams, pDevDesc deviceInfo ){

  /* Shortcut pointers to variables of interest. */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;
  /*
   * FIXME:
   * Any fill operations returned by TikZ_GetDrawOps are removed by
   * applying a bitwise and with `DRAWOP_DRAW`. This is because the old
   * StyleDef-based code had an ugly hack in it that explicitly disabled
   * filling for polypaths.
   *
   * This fixme is here because we have no tests that detect this supposed bug.
   */
  TikZ_DrawOps ops = TikZ_GetDrawOps(plotParams) & DRAWOP_DRAW;

  /*Show only for debugging*/
  if(tikzInfo->debug == TRUE) 
    printOutput(tikzInfo,
      "%% Starting Polyline\n");

  TikZ_CheckState(deviceInfo);
  TikZ_DefineColors(plotParams, deviceInfo, ops);

  /* Start drawing, open an options bracket. */
  printOutput(tikzInfo,"\n\\path[");
  TikZ_WriteDrawOptions(plotParams, deviceInfo, ops);

  /* End options, print first set of coordinates. */
  printOutput(tikzInfo, "] (%6.2f,%6.2f) --\n",
    x[0],y[0]);
  
  /* Print coordinates for the middle segments of the line. */
  int i;
  for ( i = 1; i < n-1; i++ ){
    
    printOutput(tikzInfo, "\t(%6.2f,%6.2f) --\n",
      x[i],y[i]);

  }

  /* Print last set of coordinates. End path. */
  printOutput(tikzInfo, "\t(%6.2f,%6.2f);\n",
    x[n-1],y[n-1]);
    
  /*Show only for debugging*/
  if(tikzInfo->debug == TRUE) 
    printOutput(tikzInfo,
      "%% End Polyline\n");

}

static void TikZ_Polygon( int n, double *x, double *y,
    pGEcontext plotParams, pDevDesc deviceInfo ){

  /* Shortcut pointers to variables of interest. */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;
  TikZ_DrawOps ops = TikZ_GetDrawOps(plotParams);

  /*Show only for debugging*/
  if(tikzInfo->debug == TRUE) 
    printOutput(tikzInfo,
      "%% Starting Polygon\n");

  TikZ_CheckState(deviceInfo);
  TikZ_DefineColors(plotParams, deviceInfo, ops);

  /* Start drawing, open an options bracket. */
  printOutput(tikzInfo,"\n\\path[");
  TikZ_WriteDrawOptions(plotParams, deviceInfo, ops);

  /* End options, print first set of coordinates. */
  printOutput(tikzInfo, "] (%6.2f,%6.2f) --\n",
    x[0],y[0]);
  
  /* Print coordinates for the middle segments of the line. */
  int i;
  for ( i = 1; i < n; i++ ){
    
    printOutput(tikzInfo, "\t(%6.2f,%6.2f) --\n",
      x[i],y[i]);

  }

  /* End path by cycling to first set of coordinates. */
  printOutput(tikzInfo, "\tcycle;\n" );

  /*Show only for debugging*/
  if(tikzInfo->debug == TRUE) 
    printOutput(tikzInfo,
      "%% End Polyline\n");

}


static void
TikZ_Path( double *x, double *y,
  int npoly, int *nper,
  Rboolean winding,
  const pGEcontext plotParams, pDevDesc deviceInfo
){

  int i, j, index;
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;
  TikZ_DrawOps ops = TikZ_GetDrawOps(plotParams);

  if(tikzInfo->debug) { printOutput(tikzInfo, "%% Drawing polypath with %i subpaths\n", npoly); }

  TikZ_CheckState(deviceInfo);
  TikZ_DefineColors(plotParams, deviceInfo, ops);

  /*
   * Start drawing, open an options bracket.
   *
   * TikZ has built-in support for handling rule-based filling of operlaping
   * polygons as R expects.
   *
   * Thank you TikZ!
   */
  printOutput(tikzInfo,"\n\\path[");
  TikZ_WriteDrawOptions(plotParams, deviceInfo, ops);

  /*
   * Select rule to be used for overlapping fills as specified by the 'winding'
   * parameter. See the "Graphic Parameters: Interior Rules" section of the PGF
   * manual for details.
   */
  if (winding) {
    printOutput(tikzInfo, ",nonzero rule");
  } else {
    printOutput(tikzInfo, ",even odd rule");
  }

  printOutput(tikzInfo, "]");


  /* Draw polygons */
  index = 0;
  for (i = 0; i < npoly; i++) {

    if(tikzInfo->debug) { printOutput(tikzInfo, "\n%% Drawing subpath: %i\n", i); }

    printOutput(tikzInfo, "\n\t(%6.2f,%6.2f) --\n", x[index],y[index]);
    index++;

    for (j = 1; j < nper[i]; j++) {
      printOutput(tikzInfo, "\t(%6.2f,%6.2f) --\n", x[index],y[index]);
      index++;
    }

    printOutput(tikzInfo, "\tcycle" );

  }

  /* Close the \filldraw command */
  printOutput(tikzInfo, ";\n");

}


/*
 * Creates a raster image whose lower left corner is centered at the
 * coordinates given by x and y.
 *
 * This is currently a stub function which displayes a message stating
 * that raster creation is not yet implemented.  Without this function,
 * R would crash if the user attempts to print a raster.
 *
 * This could probably be implemented by writing the raster to an image file,
 * say PNG, and then dropping a node in the TikZ output that contains
 * an \includegraphics directive.
*/
static void TikZ_Raster(
  unsigned int *raster,
  int w, int h,
  double x, double y,
  double width, double height,
  double rot,
  Rboolean interpolate,
  const pGEcontext plotParams, pDevDesc deviceInfo
){

  /* Shortcut pointer to device information. */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;

  /*
   * Recover package namespace as the raster output function is not exported
   * into the global environment.
  */
  SEXP namespace;
  PROTECT( namespace = TIKZ_NAMESPACE );

  /*
   * Prepare callback to R for creation of a PNG from raster data.  Seven
   * parameters will be passed:
   *
   * - The name of the current output file.
   *
   * - The number of rasters that have been output so far.
   *
   * - The raster data.
   *
   * - The number of rows and columns in the raster data.
   *
   * - The desired dimensions of the final image, in inches.
   *
   * - The value of the interpolate variable.
  */
  SEXP RCallBack;
  PROTECT( RCallBack = allocVector(LANGSXP, 8) );
  SETCAR( RCallBack, install("tikz_writeRaster") );

  SETCADR( RCallBack, mkString( tikzInfo->outFileName ) );
  SET_TAG( CDR(RCallBack), install("fileName") );

  SETCADDR( RCallBack, ScalarInteger( tikzInfo->rasterFileCount ) );
  SET_TAG( CDDR(RCallBack), install("rasterCount") );

  /*
   * The raster values are stored as a 32 bit unsigned integer.  Every 8 bits
   * contains an red, green, blue or alpha value (actual order is ABGR).  This
   * is the tricky bit of dealing with the raster-- there is no easy way to send
   * unsigned integers back into the R environment.  So... I gues we'll split
   * things back to RBGA values, send back a list of four vectors and regenrate
   * the whole shbang on the R side... there should be an easier way to deal
   * with this.
  */
  SEXP red_vec, blue_vec, green_vec, alpha_vec;
  PROTECT( red_vec = allocVector( INTSXP, w * h ) );
  PROTECT( blue_vec = allocVector( INTSXP, w * h ) );
  PROTECT( green_vec = allocVector( INTSXP, w * h ) );
  PROTECT( alpha_vec = allocVector( INTSXP, w * h ) );

  /*
   * Use the R_<color component> macros defined in GraphicsDevice.h to generate
   * RGBA components from the raster data.  These macros are basically shorthand
   * notation for C bitwise operators that extract 8 bit chunks from the 32 bit
   * unsigned integers contained in the raster vector.
   *
   * NOTE:
   *
   * There is some funny business that happens below.
   *
   * In the definition of device_Raster from GraphicsDevice.h, the byte order
   * of the colors entering this routine in the `raster` argument are specified
   * to be ABGR. The color extraction macros assume the order is RGBA.
   *
   * In practice, it appears the byte order in `raster` is RBGA--hence the use
   * of R_GREEN and R_BLUE are swapped below.
   *
   * If the width or height arguments to this function are negative, we
   * interpret this as a sign that the raster should be flipped along the x or
   * y matrix. For efficiency, these transformations are done in the extraction
   * loop so that the data only has to be transformed once.
   */
  int i, j, index, target, row_offset = 0, col_offset = 0, row_trans = 1, col_trans = 1;
  if ( height < 0 ) {
    /* Using these parameters, one can cause a loop to "count backwards" */
    row_trans = -1;
    row_offset = h - 1;
    /*
     * If a dimension is negative, the (x,y) coordinate no longer references
     * the lower left corner of the image. We correct for this and then make
     * sure the dimension is positive.
     */
    y += height;
    height = fabs(height);
  }

  if ( width < 0 ) {
    col_trans = -1;
    col_offset = w - 1;
    x += width;
    width = fabs(width);
  }

  for ( i = 0; i < h; ++i ) {
    for ( j = 0; j < w; ++j ) {
      target = i*w + j;
      index = (row_trans*i + row_offset)*w + (col_trans*j + col_offset);

      INTEGER(red_vec)[target] = R_RED(raster[index]);
      INTEGER(green_vec)[target] = R_BLUE(raster[index]);
      INTEGER(blue_vec)[target] = R_GREEN(raster[index]);
      INTEGER(alpha_vec)[target] = R_ALPHA(raster[index]);
    }
  }

  /*
   * We will store all the vectors generated above in an R list named colors,
   * this will make it easier to pass back into the R environment as an argument
   * to an R function
  */
  SEXP colors;
  PROTECT( colors =  allocVector( VECSXP, 4 ) );
  SET_VECTOR_ELT( colors, 0, red_vec  );
  SET_VECTOR_ELT( colors, 1, blue_vec );
  SET_VECTOR_ELT( colors, 2, green_vec );
  SET_VECTOR_ELT( colors, 3, alpha_vec );

  /* We will also make this a named list. */
  SEXP color_names;
  PROTECT( color_names = allocVector( STRSXP, 4 ) );
  SET_STRING_ELT( color_names, 0, mkChar("red") );
  SET_STRING_ELT( color_names, 1, mkChar("green") );
  SET_STRING_ELT( color_names, 2, mkChar("blue") );
  SET_STRING_ELT( color_names, 3, mkChar("alpha") );

  /* Apply the names to the list. */
  setAttrib( colors, R_NamesSymbol, color_names );


  SETCADDDR( RCallBack, colors );
  SET_TAG( CDR(CDDR(RCallBack)), install("rasterData") );

  SETCAD4R( RCallBack, ScalarInteger(h) );
  SET_TAG( CDDR(CDDR(RCallBack)), install("nrows") );

  SETCAD4R( CDR(RCallBack), ScalarInteger(w) );
  SET_TAG( CDR(CDDR(CDDR(RCallBack))), install("ncols") );

  /* Create a list containing the final width and height of the image */
  SEXP final_dims, dim_names;

  PROTECT( final_dims = allocVector(VECSXP, 2) );
  SET_VECTOR_ELT(final_dims, 0, ScalarReal(width/dim2dev(1.0)));
  SET_VECTOR_ELT(final_dims, 1, ScalarReal(height/dim2dev(1.0)));

  PROTECT( dim_names = allocVector(STRSXP, 2) );
  SET_STRING_ELT(dim_names, 0, mkChar("width"));
  SET_STRING_ELT(dim_names, 1, mkChar("height"));

  setAttrib(final_dims, R_NamesSymbol, dim_names);

  SETCAD4R(CDDR(RCallBack), final_dims);
  SET_TAG(CDDR(CDDR(CDDR(RCallBack))), install("finalDims"));

  SETCAD4R(CDR(CDDR(RCallBack)), ScalarLogical(interpolate));
  SET_TAG(CDR(CDDR(CDDR(CDDR(RCallBack)))), install("interpolate"));


  SEXP rasterFile;
  PROTECT( rasterFile = eval(RCallBack, namespace) );

  TikZ_CheckState(deviceInfo);

  /* Position the image using a node */
  printOutput(tikzInfo, "\\node[inner sep=0pt,outer sep=0pt,anchor=south west,rotate=%6.2f] at (%6.2f, %6.2f) {\n",
    rot, x, y);
  /* Include the image using PGF's native image handling */
  printOutput(tikzInfo, "\t\\pgfimage[width=%6.2fpt,height=%6.2fpt,",
      width, height);
  /* Set PDF interpolation (not all viewers respect this, but they should) */
  if (interpolate) {
    printOutput(tikzInfo, "interpolate=true]");
  } else {
    printOutput(tikzInfo, "interpolate=false]");
  }
  /* Slap in the file name */
  printOutput(tikzInfo, "{%s}", translateChar(asChar(rasterFile)));
  printOutput(tikzInfo, "};\n");

  if (tikzInfo->debug) { printOutput(tikzInfo, "\\draw[fill=red] (%6.2f, %6.2f) circle (1pt);", x, y); }

  /*
   * Increment the number of raster files we have created with this device.
   * This is used to provide unique file names for each raster.
  */
  tikzInfo->rasterFileCount++;

  UNPROTECT(11);
  return;

}


/*
 * From what little documentation exists in GraphicsDevice.h, it is
 * assumed that this function is intended to support capturing a
 * "screen shot" of the current device output and returning it
 * as a raster image.
 *
 * Implementing this functionality would require some careful thought
 * and probably won't happen unless a serious need arises.
 *
 * Argument for implementation: could be useful for "previewing" the 
 * current* state of the tikzDevice output.
*/
static SEXP TikZ_Cap( pDevDesc deviceInfo ){

  warning( "The tikzDevice does not currently support capturing device output to a raster image." );
  return R_NilValue;

}


/* 
 * Activate and deactivate execute commands when the active R device is 
 * changed. For devices using plotting windows, these routines usually change 
 * the window title to something like "Active" or "Inactive". Locator is a 
 * routine that is determines coordinates on the plotting canvas corresponding 
 * to a mouse click. For devices plotting to files these functions can be left 
 * as dummy routines.
*/
static void TikZ_Activate( pDevDesc deviceInfo ){}
static void TikZ_Deactivate( pDevDesc deviceInfo ){}
static Rboolean TikZ_Locator( double *x, double *y, pDevDesc deviceInfo ){
  return FALSE;
}

/*
 * The mode function is called when R begins drawing and ends drawing using
 * a device. Currently there are no actions necessary under these conditions
 * so this function is a dummy routine. Conciveably this function could be
 * used to wrap TikZ graphics in \begin{scope} and \end{scope} directives.
*/
static void TikZ_Mode( int mode, pDevDesc deviceInfo ){}

/*==============================================================================

                          End Core Graphics Routines

==============================================================================*/


/*==============================================================================

                           Style Definition Routines

==============================================================================*/

/*
 * This function constructs a value that can be tested using a bitwise and to
 * determine if a given plotting opertion will result in a visible fill or
 * stroke.
 */
static TikZ_DrawOps TikZ_GetDrawOps(pGEcontext plotParams)
{
  TikZ_DrawOps ops = DRAWOP_NOOP;

  /*
   * NOTE:
   *
   * Should also check that `plotParams.lty > 0` as a line type of 0 means
   * "blank". However, R does not seem to set this parameter consistently.
   */
  if( !R_TRANSPARENT(plotParams->col) && (plotParams->lwd > 0) )
    ops |= DRAWOP_DRAW;

  if( !R_TRANSPARENT(plotParams->fill) )
    ops |= DRAWOP_FILL;

  return ops;
}

static void TikZ_DefineColors(pGEcontext plotParams, pDevDesc deviceInfo, TikZ_DrawOps ops)
{
  int color;

  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;

  if ( ops & DRAWOP_DRAW ) {
    color = plotParams->col;
    if ( color != tikzInfo->oldDrawColor ) {
      tikzInfo->oldDrawColor = color;
      printOutput(tikzInfo,
        "\\definecolor[named]{drawColor}{rgb}{%4.2f,%4.2f,%4.2f}\n",
        R_RED(color)/255.0,
        R_GREEN(color)/255.0,
        R_BLUE(color)/255.0);
    }
  }

  if ( ops & DRAWOP_FILL ) {
    color = plotParams->fill;
    if( color != tikzInfo->oldFillColor ) {
      tikzInfo->oldFillColor = color;
      printOutput(tikzInfo,
        "\\definecolor[named]{fillColor}{rgb}{%4.2f,%4.2f,%4.2f}\n",
        R_RED(color)/255.0,
        R_GREEN(color)/255.0,
        R_BLUE(color)/255.0);
    }
  }

}

/*
 * NOTE: This function operates under the assumption that no other functions
 * have written into the options bracket for a path. Custom path options should
 * be added after the call to `TikZ_WriteDrawOptions` and should remember to
 * bring their own commas.
 */
static void TikZ_WriteDrawOptions(const pGEcontext plotParams, pDevDesc deviceInfo,
    TikZ_DrawOps ops)
{
  /* Bail out if there is nothing to do */
  if ( ops == DRAWOP_NOOP )
    return;

  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;

  if ( ops & DRAWOP_DRAW ) {
    printOutput(tikzInfo, "draw=drawColor");
    if( !R_OPAQUE(plotParams->col) )
      printOutput(tikzInfo, ",draw opacity=%4.2f", R_ALPHA(plotParams->col)/255.0);

    TikZ_WriteLineStyle(plotParams, tikzInfo);
  }

  if ( ops & DRAWOP_FILL ) {
    /* Toss in a comma if we printed draw options */
    if ( ops & DRAWOP_DRAW )
      printOutput(tikzInfo, ",");

    printOutput(tikzInfo, "fill=fillColor");
    if( !R_OPAQUE(plotParams->fill) )
      printOutput(tikzInfo, ",fill opacity=%4.2f", R_ALPHA(plotParams->fill)/255.0);
  }

}

static void TikZ_WriteLineStyle(pGEcontext plotParams, tikzDevDesc *tikzInfo)
{

  /*
   * Set the line width, 0.4pt is the TikZ default so scale lwd=1 relative to
   * that
   */
  printOutput(tikzInfo,",line width=%4.1fpt", 0.4*plotParams->lwd);

  if ( plotParams->lty > 1 ) {
    char dashlist[8];
    int i, nlty, lty = plotParams->lty;

    /*
     * From ?par :
     *
     * Line types can either be specified by giving an index into a small
     * built-in table of line types (1 = solid, 2 = dashed, etc, see lty above)
     * or directly as the lengths of on/off stretches of line. This is done
     * with a string of an even number (up to eight) of characters, namely
     * non-zero (hexadecimal) digits which give the lengths in consecutive
     * positions in the string. For example, the string "33" specifies three
     * units on followed by three off and "3313" specifies three units on
     * followed by three off followed by one on and finally three off. The
     * ‘units’ here are (on most devices) proportional to lwd, and with lwd = 1
     * are in pixels or points or 1/96 inch.
     *
     * The five standard dash-dot line types (lty = 2:6) correspond to:
     *  c("44", "13", "1343", "73", "2262")
     *
     * (0=blank, 1=solid (default), 2=dashed, 3=dotted, 4=dotdash, 5=longdash,
     * 6=twodash)
     */

    /*Retrieve the line type pattern*/
    for ( i = 0; i < 8 && lty & 15 ; i++ ) {
      dashlist[i] = lty & 15;
      lty = lty >> 4;
    }
    nlty = i; i = 0;

    printOutput(tikzInfo, ",dash pattern=");

    /*Set the dash pattern*/
    while( i < nlty ){
      if( (i % 2) == 0 ){
        printOutput(tikzInfo, "on %dpt ", dashlist[i]);
      }else{
        printOutput(tikzInfo, "off %dpt ", dashlist[i]);
      }
      i++;
    }
  }

  switch ( plotParams->ljoin ) {
    case GE_ROUND_JOIN:
      printOutput(tikzInfo, ",line join=round");
      break;
    case GE_MITRE_JOIN:
      /* Default if nothing is specified */
      if(plotParams->lmitre != 10)
        printOutput(tikzInfo, ",mitre limit=%4.2f",plotParams->lmitre);
      break;
    case GE_BEVEL_JOIN:
      printOutput(tikzInfo, ",line join=bevel");
  }

  switch ( plotParams->lend ) {
    case GE_ROUND_CAP:
      printOutput(tikzInfo, ",line cap=round");
      break;
    case GE_BUTT_CAP:
      /* Default if nothing is specified */
      break;
    case GE_SQUARE_CAP:
      printOutput(tikzInfo, ",line cap=rect");
  }

}

/*
 * This function calculates an appropriate scaling factor for text by
 * first calculating the ratio of the requested font size to the LaTeX
 * base font size. The ratio is then further scaled by the value of
 * the character expansion factor cex.
*/
static double
ScaleFont( const pGEcontext plotParams, pDevDesc deviceInfo ){

  // These parameters all affect the font size.
  double baseSize = deviceInfo->startps;
  double fontSize = plotParams->ps;
  double cex = plotParams->cex;

  double fontScale = ( fontSize / baseSize ) * cex;

  return( fontScale );

}


/*==============================================================================

                         Other User Callable Routines

==============================================================================*/

void TikZ_Annotate(const char **annotation, int *size){
  
  //1. Get values of tikzInfo and deviceInfo
  //2. Print out annotation 
  pDevDesc deviceInfo = GEcurrentDevice()->dev;
  
  /* Shortcut pointers to variables of interest. */
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;
    
  int i = 0;
    
  if(tikzInfo->debug == TRUE)
    printOutput(tikzInfo,"\n%% Annotating Graphic\n");
  
  for(i = 0; i < size[0]; ++i)
    printOutput(tikzInfo, "%s\n", annotation[i] );
}


/*
 * Returns information stored in the tikzDevDesc structure of a given device.
 */
SEXP TikZ_DeviceInfo(SEXP device_num){

  int dev_index = asInteger(device_num);
  pDevDesc deviceInfo = GEgetDevice(dev_index - 1)->dev;
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;

  SEXP info, names;
  PROTECT( info = allocVector(VECSXP, 2) );
  PROTECT( names = allocVector(STRSXP, 2) );

  SET_VECTOR_ELT(info, 0, mkString(tikzInfo->outFileName));
  SET_STRING_ELT(names, 0, mkChar("output_file"));

  switch( tikzInfo->engine ){
    case pdftex:
      SET_VECTOR_ELT(info, 1, mkString("pdftex"));
      break;
    case xetex:
      SET_VECTOR_ELT(info, 1, mkString("xetex"));
      break;
    case luatex:
      SET_VECTOR_ELT(info, 1, mkString("luatex"));
      break;
  }
  SET_STRING_ELT(names, 1, mkChar("engine"));


  setAttrib(info, R_NamesSymbol, names);

  UNPROTECT(2);
  return(info);

}


/* Run R evaluations inside a context protected from things like CTRL-C */
SEXP TikZ_EvalWithoutInterrupts(SEXP expr, SEXP envir){
  SEXP result;

  BEGIN_SUSPEND_INTERRUPTS{
    result = eval(expr, envir);
  }END_SUSPEND_INTERRUPTS;

  return result;
}


/*==============================================================================

                               Utility Routines

==============================================================================*/

static void printOutput(tikzDevDesc *tikzInfo, const char *format, ...){
  
  va_list(ap);
  va_start(ap, format);
  
  if(tikzInfo->console == TRUE)
    Rvprintf(format, ap);
  else
    vfprintf(tikzInfo->outputFile, format, ap);
  
  va_end(ap);
  
}


/*
 * This function is responsible for writing header information
 * to the output file. Currently this header information includes:
 *
 *   - The current version number of TikZ device.
 *   - The date on which the graphic was created.
 *
*/
static void Print_TikZ_Header( tikzDevDesc *tikzInfo ){

  /* Call back to R to retrieve current date and version num*/

  /*
   * Recover package namespace as the date formatting function
   * is not exported
  */
  SEXP namespace;
  PROTECT( namespace = TIKZ_NAMESPACE );


  SEXP currentDate;
  PROTECT(
    currentDate = eval(lang1( install("getDateStampForTikz") ),
      namespace )
  );

  SEXP currentVersion;
  PROTECT(
    currentVersion = eval(lang1( install("getTikzDeviceVersion") ),
      namespace )
  );

  printOutput( tikzInfo, "%% Created by tikzDevice version %s on %s\n",
    CHAR(STRING_ELT(currentVersion,0)), CHAR(STRING_ELT(currentDate,0)) );

	//Specifically for TeXShop, force it to open the file with UTF-8 encoding
	printOutput(tikzInfo, "%% !TEX encoding = UTF-8 Unicode\n");

  UNPROTECT(3);

}

static char *Sanitize(const char *str){

  
  //Splice in escaped charaters via a callback to R
  
  //Call out to R to retrieve the sanitizeTexString function.
  SEXP sanitizeFun = findFun( install("sanitizeTexString"), R_GlobalEnv );

  /*
   * Create a SEXP that will be the R function call. The SEXP will
   * have four components- the R function being calledand the string 
   * being passed. Therefore it is allocated as a  LANGSXP
   * vector of length 2. This is done inside a PROTECT() function
   * to keep the R garbage collector from saying "Hmmm... what's
   * this? Looks like noone is using it so I guess I will nuke it."
  */
  SEXP RCallBack;
  PROTECT( RCallBack = allocVector(LANGSXP,2) );

  // Place the function into the first slot of the SEXP.
  SETCAR( RCallBack, sanitizeFun );
  
  // Place the string into the second slot of the SEXP.
  SETCADR( RCallBack, mkString( str ) );
  // Tag the string with a name, this name coressponds to the
  // dummy argument of the R function sanitizeTexString.
  SET_TAG( CDR( RCallBack ), install("string") );

  /*
   * Call the R function, capture the result.
  */
  SEXP RSanitizedString;
  PROTECT( RSanitizedString = eval( RCallBack, R_GlobalEnv ) );

  const char *cleanString = CHAR(asChar(RSanitizedString));

  //if(DEBUG)
  //  printf("Clean String: %s\n",cleanString);

  /* 
   * cleanString is a pointer to data derived from an R object.  Once UNPROTECT
   * is called, this object may be eaten by the R garbage collector.  Therefore,
   * we need to copy the data we care about into a new string.
  */
  char *cleanStringCP = calloc_strcpy(cleanString);

  // Since we called PROTECT twice, we must call UNPROTECT
  // and pass the number 2.
  UNPROTECT(2);
  
  return cleanStringCP;
}

#if 0
static Rboolean contains_multibyte_chars(const char *str){
  /*
   * Recover package namespace as the multibyte check function
   * is not exported
  */
  SEXP namespace;
  PROTECT( namespace = TIKZ_NAMESPACE );

  SEXP multibyte_check_fun = findFun(
      install("anyMultibyteUTF8Characters"), namespace);

  SEXP RCallBack;
  PROTECT( RCallBack = allocVector(LANGSXP,2) );

  // Place the function into the first slot of the SEXP.
  SETCAR( RCallBack, multibyte_check_fun );

  // Place the string into the second slot of the SEXP.
  SETCADR( RCallBack, mkString( str ) );
  SET_TAG( CDR( RCallBack ), install("string") );

  /*
   * Call the R function, capture the result.
  */
  SEXP result;
  PROTECT( result = eval(RCallBack, namespace) );

  UNPROTECT(3);

  return(asLogical(result));
}
#endif


/*
 * This function is responsible for converting lengths given in page
 * dimensions (ie. inches, cm, etc.) to device dimensions (currenty
 * points- 1/72.27 of an inch). However, due to the flexability of TeX
 * and TikZ, any combination of device and user dimensions could
 * theoretically be supported.
*/
static double dim2dev( double length ){
  return length*72.27;
}


/*
 * This function checks to see if a new page or clipping scope needs to be
 * started and is called by every function that produces visible output. Having
 * this function allows the creation of new pages or clipping scopes to be
 * deferred until there is actually output to place in the environment.
 */
static void TikZ_CheckState(pDevDesc deviceInfo)
{
  tikzDevDesc *tikzInfo = (tikzDevDesc *) deviceInfo->deviceSpecific;

  if( tikzInfo->pageState == TIKZ_START_PAGE ) {
    /*
     * Start a new file if we are outputting to multiple files.
     *
     * FIXME:
     * Need better error handling. If we can't open a file, we are basically
     * fucked and the whole device should implode. Instead, calling `error` will
     * jump control to R without allowing for any sort of teardown and the
     * program will be left in an indeterminate state.
     */
    if( !tikzInfo->onefile )
      if( !TikZ_Open(deviceInfo) )
        error("Unable to open output file: %s", tikzInfo->outputFile);

    if ( tikzInfo->debug == TRUE )
      printOutput(tikzInfo,
        "%% Beginning new tikzpicture 'page'\n");

    if ( tikzInfo->bareBones != TRUE )
      printOutput(tikzInfo, "\\begin{tikzpicture}[x=1pt,y=1pt]\n");

    /*
     * Emit a path that encloses the entire canvas area in order to ensure that
     * the final typeset plot is the size the user specified. Adding the `use as
     * bounding box` key to the path options should save TikZ some work when it
     * comes to calculating the bounding of the graphic from its contents.
     */
    int color = deviceInfo->startfill;
    tikzInfo->oldFillColor = color;
    printOutput(tikzInfo,
      "\\definecolor[named]{fillColor}{rgb}{%4.2f,%4.2f,%4.2f}\n",
      R_RED(color)/255.0,
      R_GREEN(color)/255.0,
      R_BLUE(color)/255.0);

    printOutput(tikzInfo, "\\path[use as bounding box");

    /* TODO: Consider only filling when the color is not transparent. */
    printOutput(tikzInfo, ",fill=fillColor");
    if( !R_OPAQUE(color) )
      printOutput(tikzInfo, ",fill opacity=%4.2f", R_ALPHA(color)/255.0);

    printOutput(tikzInfo, "] (0,0) rectangle (%6.2f,%6.2f);\n",
      deviceInfo->right,deviceInfo->top);

    tikzInfo->pageState = TIKZ_FINISH_PAGE;
    tikzInfo->pageNum++;
  } /* End if pageState == TIKZ_START_PAGE */


  if ( tikzInfo->clipState == TIKZ_START_CLIP ) {
    printOutput(tikzInfo, "\\begin{scope}\n");
    printOutput(tikzInfo,
      "\\path[clip] (%6.2f,%6.2f) rectangle (%6.2f,%6.2f);\n",
      deviceInfo->clipLeft, deviceInfo->clipBottom,
      deviceInfo->clipRight, deviceInfo->clipTop);

    if ( tikzInfo->debug == TRUE )
      printOutput(tikzInfo,
        "\\path[draw=red,very thick,dashed] (%6.2f,%6.2f) rectangle (%6.2f,%6.2f);\n",
        deviceInfo->clipLeft, deviceInfo->clipBottom,
        deviceInfo->clipRight, deviceInfo->clipTop);

    tikzInfo->clipState = TIKZ_FINISH_CLIP;
  } /* End if clipState == TIKZ_START_CLIP */

}

static char *calloc_strcpy(const char *str){
  return calloc_x_strcpy(str, 0);
}

static char *calloc_x_strcpy(const char *str, size_t extra){
  char *ret;

  ret = calloc_x_strlen(str, extra);
  strcpy(ret, str);
  return ret;
}

static char *calloc_x_strlen(const char *str, size_t extra){
  return (char*) calloc(strlen(str) + 1 + extra, sizeof(char));
}

static void const_free(const void *ptr){
  free((void*)ptr);
}
