% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/recipes-step_roll_apply.R
\name{step_roll_apply}
\alias{step_roll_apply}
\alias{tidy.step_roll_apply}
\title{Rolling Window Transformation}
\usage{
step_roll_apply(
  recipe,
  ...,
  period,
  .f,
  align = c("center", "left", "right"),
  names = NULL,
  role = "predictor",
  trained = FALSE,
  columns = NULL,
  f_name = NULL,
  skip = FALSE,
  id = rand_id("roll_apply")
)

\method{tidy}{step_roll_apply}(x, ...)
}
\arguments{
\item{recipe}{A recipe object. The step will be added to the
sequence of operations for this recipe.}

\item{...}{One or more numeric columns to be smoothed.
See \code{\link[recipes:selections]{recipes::selections()}} for more details.
For the \code{tidy} method, these are not currently used.}

\item{period}{The number of periods to include in the local rolling window.
This is effectively the "window size".}

\item{.f}{A summary \strong{formula} in one of the following formats:
\itemize{
\item \code{mean} with no arguments
\item \code{function(x) mean(x, na.rm = TRUE)}
\item \code{~ mean(.x, na.rm = TRUE)}, it is converted to a function.
}}

\item{align}{Rolling functions generate \code{period - 1} fewer values than the incoming vector.
Thus, the vector needs to be aligned. Alignment of the vector follows 3 types:
\itemize{
\item \strong{Center:} \code{NA} or \code{.partial} values are divided and added to the beginning and
end of the series to "Center" the moving average.
This is common for de-noising operations. See also \verb{[smooth_vec()]} for LOESS without NA values.
\item \strong{Left:} \code{NA} or \code{.partial} values are added to the end to shift the series to the Left.
\item \strong{Right:} \code{NA} or \code{.partial} values are added to the beginning to shif the series to the Right. This is common in
Financial Applications such as moving average cross-overs.
}}

\item{names}{An optional character string that is the same
length of the number of terms selected by \code{terms}. These will be
the names of the \strong{new columns} created by the step.
\itemize{
\item If \code{NULL}, existing columns are transformed.
\item If not \code{NULL}, new columns will be created.
}}

\item{role}{For model terms created by this step, what analysis
role should they be assigned?. By default, the function assumes
that the new variable columns created by the original variables
will be used as predictors in a model.}

\item{trained}{A logical to indicate if the quantities for
preprocessing have been estimated.}

\item{columns}{A character string of variables that will be
used as inputs. This field is a placeholder and will be
populated once \code{\link[recipes:prep.recipe]{recipes::prep.recipe()}} is used.}

\item{f_name}{A character string for the function being applied.
This field is a placeholder and will be populated during the \code{tidy()} step.}

\item{skip}{A logical. Should the step be skipped when the recipe is
baked by bake.recipe()? While all operations are baked when prep.recipe()
is run, some operations may not be able to be conducted on new data
(e.g. processing the outcome variable(s)). Care should be taken when
using skip = TRUE as it may affect the computations for subsequent operations.}

\item{id}{A character string that is unique to this step to identify it.}

\item{x}{A \code{step_roll_apply} object.}
}
\value{
For \code{step_roll_apply}, an updated version of recipe with
the new step added to the sequence of existing steps (if any).
For the \code{tidy} method, a tibble with columns \code{terms}
(the selectors or variables selected), \code{value} (the feature
names).
}
\description{
\code{step_roll_apply} creates a a \emph{specification} of a recipe
step that will apply a function
to one or more a Numeric column(s).
}
\details{
\strong{Alignment}

Rolling functions generate \code{period - 1} fewer values than the incoming vector.
Thus, the vector needs to be aligned. Alignment of the vector follows 3 types:
\itemize{
\item \strong{Center:} \code{NA} or \code{partial} values are divided and added to the beginning and
end of the series to "Center" the moving average.
This is common for de-noising operations. See also \verb{[smooth_vec()]} for LOESS without NA values.
\item \strong{Left:} \code{NA} or \code{partial} values are added to the end to shift the series to the Left.
\item \strong{Right:} \code{NA} or \code{partial} values are added to the beginning to shif the series to the Right. This is common in
Financial Applications such as moving average cross-overs.
}

\strong{Partial Values}
\itemize{
\item The advantage to using \code{partial} values vs \code{NA} padding is that
the series can be filled (good for time-series de-noising operations).
\item The downside to partial values is that the partials can become less stable
at the regions where incomplete windows are used.
}

If instability is not desirable for de-noising operations, a suitable alternative
is \code{\link[=step_smooth]{step_smooth()}}, which implements local polynomial regression.
}
\examples{
library(recipes)
library(tidyverse)
library(tidyquant)
library(timetk)

# Training Data
FB_tbl <- FANG \%>\%
    filter(symbol == "FB") \%>\%
    select(symbol, date, adjusted)

# New Data
new_data <- FB_tbl \%>\%
    tk_index() \%>\%
    tk_make_future_timeseries(n_future = 90) \%>\%
    tibble(date = .)  \%>\%
    mutate(date = date) \%>\%
    bind_cols(FB_tbl \%>\% slice((n() - 90 + 1):n()))


# Create a recipe object with a step_roll_apply
rec_ma_50 <- recipe(adjusted ~ ., data = FB_tbl) \%>\%
    step_roll_apply(adjusted, period = 50, .f = ~ AVERAGE(.x))

# Bake the recipe object - Applies the Moving Average Transformation
training_data_baked <- bake(prep(rec_ma_50), FB_tbl)

# Apply to New Data
new_data_baked <- bake(prep(rec_ma_50), new_data)

# Visualize effect
training_data_baked \%>\%
    ggplot(aes(date, adjusted)) +
    geom_line() +
    geom_line(color = "red", data = new_data_baked)

# ---- NEW COLUMNS ----
# Use the `names` argument to create new columns instead of overwriting existing

rec_ma_30_names <- recipe(adjusted ~ ., data = FB_tbl) \%>\%
    step_roll_apply(adjusted, period = 30, .f = AVERAGE, names = "adjusted_ma_30")

bake(prep(rec_ma_30_names), FB_tbl) \%>\%
    ggplot(aes(date, adjusted)) +
    geom_line(alpha = 0.5) +
    geom_line(aes(y = adjusted_ma_30), color = "red", size = 1)



}
\seealso{
Time Series Analysis:
\itemize{
\item Engineered Features: \code{\link[=step_timeseries_signature]{step_timeseries_signature()}}, \code{\link[=step_holiday_signature]{step_holiday_signature()}}, \code{\link[=step_fourier]{step_fourier()}}
\item Diffs & Lags \code{\link[=step_diff]{step_diff()}}, \code{\link[recipes:step_lag]{recipes::step_lag()}}
\item Smoothing: \code{\link[=step_roll_apply]{step_roll_apply()}}, \code{\link[=step_smooth]{step_smooth()}}
\item Variance Reduction: \code{\link[=step_box_cox]{step_box_cox()}}
\item Imputation: \code{\link[=step_impute_ts]{step_impute_ts()}}
}

Main Recipe Functions:
\itemize{
\item \code{\link[recipes:recipe]{recipes::recipe()}}
\item \code{\link[recipes:prep.recipe]{recipes::prep.recipe()}}
\item \code{\link[recipes:bake.recipe]{recipes::bake.recipe()}}
}
}
\concept{moving_windows}
\concept{preprocessing}
\keyword{datagen}
