% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ipe.R
\name{ipe}
\alias{ipe}
\title{Iterative Parameter Estimation (IPE) for Treatment Switching}
\usage{
ipe(
  data,
  id = "id",
  stratum = "",
  time = "time",
  event = "event",
  treat = "treat",
  rx = "rx",
  censor_time = "censor_time",
  base_cov = "",
  aft_dist = "weibull",
  low_psi = -2,
  hi_psi = 2,
  strata_main_effect_only = 1,
  treat_modifier = 1,
  recensor = TRUE,
  admin_recensor_only = TRUE,
  autoswitch = TRUE,
  root_finding = "brent",
  alpha = 0.05,
  ties = "efron",
  tol = 1e-06,
  boot = FALSE,
  n_boot = 1000,
  seed = NA
)
}
\arguments{
\item{data}{The input data frame that contains the following variables:
\itemize{
\item \code{id}: The subject id.
\item \code{stratum}: The stratum.
\item \code{time}: The survival time for right censored data.
\item \code{event}: The event indicator, 1=event, 0=no event.
\item \code{treat}: The randomized treatment indicator, 1=treatment,
0=control.
\item \code{rx}: The proportion of time on active treatment.
\item \code{censor_time}: The administrative censoring time. It should
be provided for all subjects including those who had events.
\item \code{base_cov}: The baseline covariates (excluding treat).
}}

\item{id}{The name of the id variable in the input data.}

\item{stratum}{The name(s) of the stratum variable(s) in the input data.}

\item{time}{The name of the time variable in the input data.}

\item{event}{The name of the event variable in the input data.}

\item{treat}{The name of the treatment variable in the input data.}

\item{rx}{The name of the rx variable in the input data.}

\item{censor_time}{The name of the censor_time variable in the input data.}

\item{base_cov}{The names of baseline covariates (excluding
treat) in the input data for the causal AFT model and the outcome
Cox model.}

\item{aft_dist}{The assumed distribution for time to event for the AFT
model. Options include "exponential", "weibull" (default),
"loglogistic", and "lognormal".}

\item{low_psi}{The lower limit of the causal parameter.}

\item{hi_psi}{The upper limit of the causal parameter.}

\item{strata_main_effect_only}{Whether to only include the strata main
effects in the AFT model. Defaults to \code{TRUE}, otherwise all
possible strata combinations will be considered in the AFT model.}

\item{treat_modifier}{The optional sensitivity parameter for the
constant treatment effect assumption.}

\item{recensor}{Whether to apply recensoring to counterfactual
survival times. Defaults to \code{TRUE}.}

\item{admin_recensor_only}{Whether to apply recensoring to administrative
censoring times only. Defaults to \code{TRUE}. If \code{FALSE},
recensoring will be applied to the actual censoring times for dropouts.}

\item{autoswitch}{Whether to exclude recensoring for treatment arms
with no switching. Defaults to \code{TRUE}.}

\item{root_finding}{Character string specifying the univariate
root-finding algorithm to use. Options are \code{"brent"} (default)
for Brent's method, or \code{"bisection"} for the bisection method.}

\item{alpha}{The significance level to calculate confidence intervals.}

\item{ties}{The method for handling ties in the Cox model, either
"breslow" or "efron" (default).}

\item{tol}{The desired accuracy (convergence tolerance) for \code{psi}
for the root finding algorithm.}

\item{boot}{Whether to use bootstrap to obtain the confidence
interval for hazard ratio. Defaults to \code{FALSE}, in which case,
the confidence interval will be constructed to match the log-rank
test p-value.}

\item{n_boot}{The number of bootstrap samples.}

\item{seed}{The seed to reproduce the bootstrap results. The default is
missing, in which case, the seed from the environment will be used.}
}
\value{
A list with the following components:
\itemize{
\item \code{psi}: The estimated causal parameter.
\item \code{psi_CI}: The confidence interval for \code{psi}.
\item \code{psi_CI_type}: The type of confidence interval for \code{psi},
i.e., "log-rank p-value" or "bootstrap".
\item \code{logrank_pvalue}: The two-sided p-value of the log-rank test
for the ITT analysis.
\item \code{cox_pvalue}: The two-sided p-value for treatment effect based on
the Cox model applied to counterfactual unswitched survival times.
If \code{boot} is \code{TRUE}, this value represents the
bootstrap p-value.
\item \code{hr}: The estimated hazard ratio from the Cox model.
\item \code{hr_CI}: The confidence interval for hazard ratio.
\item \code{hr_CI_type}: The type of confidence interval for hazard ratio,
either "log-rank p-value" or "bootstrap".
\item \code{Sstar}: A data frame containing the counterfactual untreated
survival times and event indicators for each treatment group.
The variables include \code{id}, \code{stratum},
\code{"t_star"}, \code{"d_star"}, \code{"treated"}, \code{base_cov},
and \code{treat}.
\item \code{kmstar}: A data frame containing the Kaplan-Meier estimates
based on the counterfactual untreated survival times by treatment arm.
\item \code{data_aft}: The input data for the AFT model for
estimating \code{psi}. The variables include \code{id}, \code{stratum},
\code{"t_star"}, \code{"d_star"}, \code{"treated"}, \code{base_cov},
and \code{treat}.
\item \code{fit_aft}: The fitted AFT model for estimating \code{psi}.
\item \code{data_outcome}: The input data for the outcome Cox model of
counterfactual unswitched survival times.
The variables include \code{id}, \code{stratum}, \code{"t_star"},
\code{"d_star"}, \code{"treated"}, \code{base_cov}, and \code{treat}.
\item \code{fit_outcome}: The fitted outcome Cox model.
\item \code{fail}: Whether a model fails to converge.
\item \code{psimissing}: Whether the psi parameter cannot be estimated.
\item \code{settings}: A list with the following components:
\itemize{
\item \code{aft_dist}: The distribution for time to event for the AFT
model.
\item \code{strata_main_effect_only}: Whether to only include the strata
main effects in the AFT model.
\item \code{low_psi}: The lower limit of the causal parameter.
\item \code{hi_psi}: The upper limit of the causal parameter.
\item \code{treat_modifier}: The sensitivity parameter for the constant
treatment effect assumption.
\item \code{recensor}: Whether to apply recensoring to counterfactual
survival times.
\item \code{admin_recensor_only}: Whether to apply recensoring to
administrative censoring times only.
\item \code{autoswitch}: Whether to exclude recensoring for treatment
arms with no switching.
\item \code{root_finding}: The univariate root-finding algorithm to use.
\item \code{alpha}: The significance level to calculate confidence
intervals.
\item \code{ties}: The method for handling ties in the Cox model.
\item \code{tol}: The desired accuracy (convergence tolerance)
for \code{psi}.
\item \code{boot}: Whether to use bootstrap to obtain the confidence
interval for hazard ratio.
\item \code{n_boot}: The number of bootstrap samples.
\item \code{seed}: The seed to reproduce the bootstrap results.
}
\item \code{fail_boots}: The indicators for failed bootstrap samples
if \code{boot} is \code{TRUE}.
\item \code{fail_boots_data}: The data for failed bootstrap samples
if \code{boot} is \code{TRUE}.
\item \code{hr_boots}: The bootstrap hazard ratio estimates if \code{boot} is
\code{TRUE}.
\item \code{psi_boots}: The bootstrap \code{psi} estimates if \code{boot} is
\code{TRUE}.
}
}
\description{
Obtains the causal parameter estimate from the
accelerated failure-time (AFT) model and the hazard ratio estimate
from the Cox model to adjust for treatment switching.
}
\details{
We use the following steps to obtain the hazard ratio estimate
and confidence interval had there been no treatment switching:
\itemize{
\item Use IPE to estimate the causal parameter \eqn{\psi} based on the AFT
model for the observed survival times for the experimental arm and
the counterfactual survival times for the control arm,
\deqn{U_{i,\psi} = T_{C_i} + e^{\psi}T_{E_i}}
\item Fit the Cox proportional hazards model to the observed survival times
for the experimental group and the counterfactual survival times
for the control group to obtain the hazard ratio estimate.
\item Use either the log-rank test p-value for the intention-to-treat (ITT)
analysis or bootstrap to construct the confidence interval for
hazard ratio. If bootstrapping is used, the confidence interval
and corresponding p-value are calculated based on a t-distribution with
\code{n_boot - 1} degrees of freedom.
}
}
\examples{

library(dplyr)

# Example 1: one-way treatment switching (control to active)

data <- immdef \%>\% mutate(rx = 1-xoyrs/progyrs)

fit1 <- ipe(
  data, id = "id", time = "progyrs", event = "prog", treat = "imm", 
  rx = "rx", censor_time = "censyrs", aft_dist = "weibull",
  boot = FALSE)

c(fit1$hr, fit1$hr_CI)

# Example 2: two-way treatment switching (illustration only)

# the eventual survival time
shilong1 <- shilong \%>\%
  arrange(bras.f, id, tstop) \%>\%
  group_by(bras.f, id) \%>\%
  slice(n()) \%>\%
  select(-c("ps", "ttc", "tran"))

shilong2 <- shilong1 \%>\%
  mutate(rx = ifelse(co, ifelse(bras.f == "MTA", dco/ady, 
                                1 - dco/ady),
                     ifelse(bras.f == "MTA", 1, 0)))

fit2 <- ipe(
  shilong2, id = "id", time = "tstop", event = "event",
  treat = "bras.f", rx = "rx", censor_time = "dcut",
  base_cov = c("agerand", "sex.f", "tt_Lnum", "rmh_alea.c",
               "pathway.f"),
  aft_dist = "weibull", boot = FALSE)

c(fit2$hr, fit2$hr_CI)

}
\references{
Michael Branson and John Whitehead.
Estimating a treatment effect in survival studies in which patients
switch treatment.
Statistics in Medicine. 2002;21(17):2449-2463.

Ian R White.
Letter to the Editor: Estimating treatment effects in randomized
trials with treatment switching.
Statistics in Medicine. 2006;25(9):1619-1622.
}
\author{
Kaifeng Lu, \email{kaifenglu@gmail.com}
}
