#' Interactive visualization of voxelwise RLRT results
#' 
#' This function uses interactive graphics tools, provided by the \pkg{rpanel}
#' package, to visualize the results of massively parallel RLRT results as
#' produced by \code{\link{rlrt4d}}.
#' 
#' RLRT results are shown for a cross-section (slice), and a slider allows
#' viewing of different slices.  Clicking on a voxel produces a scatterplot of
#' the data at that voxel.
#' 
#' @param rlrtobj an RLRT object as produced by \code{\link{rlrt4d}}.
#' @param array4d a 4-dimensional array containing the raw data that were
#' smoothed at each point.  The first 3 dimensions refer to x, y, and z
#' coordinates and the last dimension corresponds to different images.
#' @param predictor a vector or matrix of covariates.
#' @param disp values from the RLRT object to be displayed: either RLRT
#' statistics, p-values, or FDR values.
#' @param titl title of the panel.
#' @param xlab,ylab x- and y-axis labels.
#' @param slice index of the default slice to be shown in the panel.
#' @param ylim.scatter the y limits of the scatterplots.
#' @param col.image a list of colors such as that generated by rainbow,
#' heat.colors, topo.colors, terrain.colors or similar functions.
#' @param neglog10 logical; if \code{TRUE}, negative base \code{10} logarithm
#' (of the quantity specified by \code{disp}) is displayed.
#' @param threshold the upper limit of the values to be plotted. All larger
#' values will be replaced by the threshold value.
#' @author Lei Huang \email{huangracer@@gmail.com}
#' @seealso \code{\link{rlrt4d}}
#' @references Bowman, A., Crawford, E., Alexander, G., and Bowman, R. (2007).
#' rpanel: Simple interactive controls for R functions using the tcltk package.
#' \emph{Journal of Statistical Software}, 17(9).
#' @examples
#' 
#' # Please see example for rlrt4d
#' @export
rlrtpanel <-
function(rlrtobj, array4d, predictor, disp = c("stat","p","fdr"), titl="", xlab="Age", ylab="", slice=(dim(array4d)[3]%/%2), ylim.scatter=NULL, col.image = shape::femmecol(100), neglog10=FALSE, threshold=NULL)  {
    tkrp = tkrp1 = NULL
    disp = match.arg(disp)
    coord = attributes(array4d)$coord
    has.data = attributes(array4d)$has.data
    
    x.coord = coord[[1]]
    y.coord = coord[[2]]
    z.coord = coord[[3]]
    axis.flag = TRUE
    ttl = "z ="
    xlb="x"; ylb="y"
    if (!(disp %in% c("stat", "p", "fdr"))) stop("You must choose rlrt statistics, p-value or fdr to display!")
    arr = array(NA, dim=dim(has.data))
    if (disp=="stat") arr[has.data] = rlrtobj$stat
    if (disp=="p") arr[has.data] = rlrtobj$p
    if (disp=="fdr") arr[has.data] = rlrtobj$fdr
    if (neglog10) {
    	arr[has.data] = -log10(arr[has.data])
        if (disp=="stat") warning("Do you really want to take the negative base-10 log of the RLR statistic?")
    }
    if (!is.null(threshold))    arr[!is.na(arr) & (arr>threshold)] = threshold
    zlim = range(arr[arr<Inf], na.rm=TRUE)    
    
    draw <- function(panel) {
        with(panel,image(x = x.coord, y=y.coord, z=arr[ , , slice],
              col=col.image, main=paste(ttl, z.coord[slice]), xlab=xlb, ylab=ylb, zlim=zlim, axes=axis.flag))
        panel
    }
    
    redraw <- function(panel) {   
        rpanel::rp.tkrreplot(panel, tkrp)
        panel
    }
    
    scatterplot <- function(panel,x,y)  {
      with(panel, plot(predictor, array4d[which.min(abs(x.coord-x)), which.min(abs(y.coord-y)), slice, ],
           xlab=xlab, ylab=ylab, ylim=ylim.scatter, main=paste("Coordinates", round(x), round(y), z.coord[slice])))
      return(panel)
   }
   
    legend.draw <- function(panel) {
         shape::emptyplot(main="    ")
         shape::colorlegend(posx=c(0.6,0.7), col=col.image,
               zlim=zlim, zval = seq(min(zlim), max(zlim), length.out=5),main="", left=FALSE, digit=2)
         panel
    }
    
    imgplot <- rpanel::rp.control(title = titl, slice=slice)
    rpanel::rp.tkrplot(imgplot, tkrp, draw, action=scatterplot, hscale = 0.8, vscale = 1.2, pos=list(row=0, column=0))
    rpanel::rp.tkrplot(imgplot, tkrp1, legend.draw, hscale = 0.8, vscale = 1.2, pos=list(row=0, column=1))
    rpanel::rp.slider(panel = imgplot, variable=slice, action = redraw, from = 1, to = dim(arr)[3], resolution = 1, title=ttl, showvalue=is.null(coord), pos=list(row=1, column=1))
}

