% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/genAlgo.R
\name{genAlgo}
\alias{genAlgo}
\title{Start a Genetic Algorithm for a wind farm layout optimization}
\usage{
genAlgo(Polygon1, GridMethod, Rotor, n, fcrR, referenceHeight, RotorHeight,
  SurfaceRoughness, Proportionality, iteration, mutr, vdirspe, topograp,
  elitism, nelit, selstate, crossPart1, trimForce, Projection, sourceCCL,
  sourceCCLRoughness, weibull, weibullsrc, Parallel, numCluster,
  verbose = FALSE, plotit = FALSE)
}
\arguments{
\item{Polygon1}{The considered area as SpatialPolygon, SimpleFeature Polygon
or coordinates as matrix/data.frame}

\item{GridMethod}{Should the polygon be divided into rectangular or
hexagonal grid cells? The default is "Rectangular" grid cells and hexagonal
grid cells are computed when assigning "h" or "hexagon" to this input
variable.}

\item{Rotor}{A numeric value that gives the rotor radius in meter}

\item{n}{A numeric value indicating the required amount of turbines}

\item{fcrR}{A numeric value that is used for grid spacing. Default is 5}

\item{referenceHeight}{The height at which the incoming
wind speeds were measured. Default is the RotorHeight.}

\item{RotorHeight}{The desired height of the turbine.}

\item{SurfaceRoughness}{A surface roughness length of the
considered area in m.  If the terrain effect model is activated, a
surface roughness will be calculated for every grid cell with the
elevation and land cover information. Default is 0.3}

\item{Proportionality}{A numeric value used for grid calculation.
Determines the percentage a grid has to overlay. Default is 1}

\item{iteration}{A numeric value indicating the desired amount
of iterations of the algorithm. Default is 20}

\item{mutr}{A numeric mutation rate with low default value of 0.008}

\item{vdirspe}{A data.frame containing the incoming wind speeds,
wind directions and probabilities}

\item{topograp}{Logical value, which indicates if the terrain effect model
should be activated or not. Default is FALSE}

\item{elitism}{Boolean value, which indicates whether elitism should
be included or not. Default is TRUE}

\item{nelit}{If \code{elitism} is TRUE, then this input variable
determines the amount of individuals in the elite group. Default is 7}

\item{selstate}{Determines which selection method is used,
"FIX" selects a constant percentage and "VAR" selects a variable percentage,
depending on the development of the fitness values. Default is "FIX"}

\item{crossPart1}{Determines which crossover method is used,
"EQU" divides the genetic code at equal intervals and
"RAN" divides the genetic code at random locations. Default is "EQU"}

\item{trimForce}{If activated (\code{trimForce == TRUE}),
the algorithm will take a probabilistic approach to trim the windfarms
to the desired amount of turbines. If deactivated
(\code{trimForce == FALSE}) the adjustment will be random.
Default is FALSE}

\item{Projection}{A desired Projection can be used instead
of the default Lambert Azimuthal Equal Area Projection (EPSG:3035).}

\item{sourceCCL}{The path to the Corine Land Cover raster (.tif). Only
required when the terrain effect model is activated. If nothing is assign,
it will try to download a version from the EEA-website.}

\item{sourceCCLRoughness}{The source to the adapted
Corine Land Cover legend as .csv file. Only required when terrain
effect model is activated. As default a .csv file within this
package (\file{~/extdata}) is taken that was already adapted
manually. To use your own .csv legend this variable has to be assigned.}

\item{weibull}{A logical value that specifies whether to take Weibull
parameters into account. If weibull==TRUE, the wind speed values from the
'dirSpeed' data frame are ignored. The algorithm will calculate the mean
wind speed for every wind turbine according to the Weibull parameters.
Default is FALSE}

\item{weibullsrc}{A list of Weibull parameter rasters, where the first list
item must be the shape parameter raster k and the second item must be the
scale parameter raster a of the Weibull distribution. If no list is given,
then rasters included in the package are used instead, which currently
only cover Austria. This variable is only used if weibull==TRUE.}

\item{Parallel}{Boolean value, indicating whether parallel processing
should be used. The parallel and doParallel packages are used for parallel
processing. Default is FALSE}

\item{numCluster}{If Parallel is TRUE, this variable defines the number
of clusters to be used}

\item{verbose}{If TRUE, will print out information for every generation 
Default is FALSE}

\item{plotit}{If TRUE, will plot the best windfarm of a generation. Default
is FALSE}
}
\value{
The result is a matrix with aggregated values per generation,
the best individual regarding energy and efficiency per generation,
some fuzzy control variables per generation, a list of all fitness values
per generation, the amount of individuals after each process, a matrix of
all energy, efficiency and fitness values per generation, the selection and 
crossover paramters, a matrix with the generational difference in maximum
and mean energy output, a matrix with the given inputs, a dataframe with
the wind information, the mutation rate per generation and matrix with all 
tested wind farm layouts.
}
\description{
Run a Genetic Algorithm to optimize the layout of wind turbines
on a given area. The algorithm works with a fixed amount of turbines, a 
fixed rotor radius and a mean wind speed value for every incoming wind 
direction.
}
\details{
A terrain effect model can be included in the optimization process.
Therefore, an SRTM elevation model will be downloaded automatically via the
\code{raster::getData} function. A land cover raster can also be downloaded
automatically from the EEA-website, or the path to a raster file can be 
passed to \code{sourceCCL}. The algorithm uses an adapted version of the
Raster legend ("clc_legend.csv"), which is stored in the package directory 
\file{~/inst/extdata}. To use other values for the land cover roughness 
lengths, insert a column named \strong{"Rauhigkeit_z"} to the .csv file, 
assign a surface roughness length to all land cover types. 
Be sure that all rows are filled with numeric values and save the file
with \strong{";"} separation. Assign the path of the file to the 
input variable \code{sourceCCLRoughness} of this function.
}
\examples{
\donttest{
## Create a random rectangular shapefile
library(sp)
Polygon1 <- Polygon(rbind(c(4498482, 2668272), c(4498482, 2669343),
                          c(4499991, 2669343), c(4499991, 2668272)))
Polygon1 <- Polygons(list(Polygon1), 1);
Polygon1 <- SpatialPolygons(list(Polygon1))
Projection <- "+proj=laea +lat_0=52 +lon_0=10 +x_0=4321000 +y_0=3210000
+ellps=GRS80 +towgs84=0,0,0,0,0,0,0 +units=m +no_defs"
proj4string(Polygon1) <- CRS(Projection)
plot(Polygon1,axes=TRUE)

## Create a uniform and unidirectional wind data.frame and plot the
## resulting wind rose
data.in <- data.frame(ws = 12, wd = 0)
windrosePlot <- plotWindrose(data = data.in, spd = data.in$ws,
                dir = data.in$wd, dirres=10, spdmax=20)

## Runs an optimization run for 20 iterations with the
## given shapefile (Polygon1), the wind data.frame (data.in),
## 12 turbines (n) with rotor radii of 30m (Rotor) and rotor height of 100m.
result <- genAlgo(Polygon1 = Polygon1,
                  n = 12,
                  vdirspe = data.in,
                  Rotor = 30,
                  RotorHeight = 100)
PlotWindfarmGA(result = result, Polygon1 = Polygon1)

## Runs the same optimization, but with parallel processing and 3 cores.
result_par <- genAlgo(Polygon1 = Polygon1, GridMethod ="h", n=12, Rotor=30,
                 fcrR=5,iteration=10, vdirspe = data.in,crossPart1 = "EQU",
                 selstate="FIX",mutr=0.8, Proportionality = 1,
                 SurfaceRoughness = 0.3, topograp = FALSE,
                 elitism=TRUE, nelit = 7, trimForce = TRUE,
                 referenceHeight = 50,RotorHeight = 100, 
                 Parallel = TRUE, numCluster = 3)
PlotWindfarmGA(result = result_par, GridMethod = "h", Polygon1 = Polygon1)

## Runs the same optimization, this time with hexagonal grids.
result_hex <- genAlgo(Polygon1 = Polygon1, GridMethod ="h", n=12, Rotor=30,
                 fcrR=5,iteration=10, vdirspe = data.in,crossPart1 = "EQU",
                 selstate="FIX",mutr=0.8, Proportionality = 1,
                 SurfaceRoughness = 0.3, topograp = FALSE,
                 elitism=TRUE, nelit = 7, trimForce = TRUE,
                 referenceHeight = 50,RotorHeight = 100)
PlotWindfarmGA(result = result_hex, GridMethod = "h", Polygon1 = Polygon1)

## Run an optimization with the Weibull parameters included in the package.
result_weibull <- genAlgo(Polygon1 = Polygon1, GridMethod ="h", n=12,
                 fcrR=5,iteration=10, vdirspe = data.in,crossPart1 = "EQU",
                 selstate="FIX",mutr=0.8, Proportionality = 1, Rotor=30,
                 SurfaceRoughness = 0.3, topograp = FALSE,
                 elitism=TRUE, nelit = 7, trimForce = TRUE,
                 referenceHeight = 50,RotorHeight = 100,
                 weibull = TRUE)
PlotWindfarmGA(result = result_weibull, GridMethod= "h", Polygon1 = Polygon1)

## Run an optimization with given Weibull parameter rasters.
#araster <- "/..pathto../a_param_raster.tif"
#kraster <- "/..pathto../k_param_raster.tif"
#weibullrasters <- list(raster(kraster), raster(araster))
#result_weibull <- genAlgo(Polygon1 = Polygon1, GridMethod ="h", n=12,
#                  fcrR=5,iteration=10, vdirspe = data.in,crossPart1 = "EQU",
#                  selstate="FIX",mutr=0.8, Proportionality = 1, Rotor=30,
#                  SurfaceRoughness = 0.3, topograp = FALSE,
#                  elitism=TRUE, nelit = 7, trimForce = TRUE,
#                  referenceHeight = 50,RotorHeight = 100,
#                  weibull = TRUE, weibullsrc = weibullrasters)
#PlotWindfarmGA(result = result_weibull, GridMethod= "h", Polygon1 = Polygon1)
}
}
\author{
Sebastian Gatscha
}
