% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/module5_mst.R
\name{mst}
\alias{mst}
\alias{mst_assemble}
\alias{mst_constraint}
\alias{mst_get_items}
\alias{mst_module_mininfo}
\alias{mst_obj}
\alias{mst_rdp}
\alias{mst_route}
\alias{mst_stage_length}
\alias{plot.mst}
\alias{print.mst}
\title{Computerized Multistage Testing (MST)}
\usage{
mst(pool, design, npanel, method = c("topdown", "bottomup"), len = NULL,
  maxselect = NULL)

mst_route(x, route, op = c("+", "-"))

mst_obj(x, theta, indices = NULL, target = NULL, flatten = NULL)

mst_constraint(x, coef, min = NA, max = NA, level = NULL,
  indices = NULL)

mst_stage_length(x, stages, min = NA, max = NA)

mst_rdp(x, theta, indices, tol)

mst_module_mininfo(x, theta, mininfo, indices)

mst_assemble(x, ...)

\method{print}{mst}(x, ...)

\method{plot}{mst}(x, ...)

mst_get_items(x, panel, stage = NULL, module = NULL, route = NULL)
}
\arguments{
\item{pool}{the item pool (data.frame)}

\item{design}{the MST design (string): e.g., "1-2-3", "1-2-2", etc.}

\item{npanel}{the number of panels}

\item{method}{the design method: 'topdown' or 'bottomup'}

\item{len}{the module/route length}

\item{maxselect}{the maximum selection of items}

\item{x}{a MST object}

\item{route}{a MST route represented by a vector of module indices}

\item{op}{\code{"+"} for adding a route and \code{"-"} for removing a route}

\item{theta}{a theta point or interval for optimization}

\item{indices}{the index of the route (topdown) or the module (bottomup) for adding objectives}

\item{target}{the target values of the TIF objectives. \code{NULL} for maximization}

\item{flatten}{the parameter for getting a flat TIF}

\item{coef}{the coefficients of the constraint}

\item{min}{the lower bound of the constraints}

\item{max}{the upper bound of the constraints}

\item{level}{the constrained level, \code{NA} for continuous variable}

\item{stages}{the stage index}

\item{tol}{tolerance parameter}

\item{mininfo}{the minimum information threshold}

\item{...}{further arguments}

\item{panel}{the panel index}

\item{stage}{the stage index}

\item{module}{the module index}
}
\description{
\code{mst} creates a multistage (MST) object

\code{mst_route} adds and removes routes in the MST

\code{mst_obj} adds objective functions to the MST

\code{mst_constraint} adds constraints to the MST

\code{mst_stage_length} sets the minimum and maximum length for a stage

\code{mst_rdp} anchors the routing decision point (rdp) between adjacent modules

\code{mst_module_mininfo} sets the minimum information for modules

\code{mst_assemble} assembles the mst

\code{mst_get_items} extracts items from results
}
\details{
The \code{mst} object contains: item pool (\code{pool}), ATA (\code{assembler}), 
route map (\code{route}), module map (\code{module}), design method (\code{method}), 
and several constants such as \code{npanel}, \code{nstage}, \code{nmodule}, 
\code{nroute}.\cr
Two identifiers are used to index modules/testlets. \code{form} is a unique 
id for all modules (for ATA), and \code{index} is a panel-wise unique id (for MST).\cr
There are two methods for designing MST: \code{'bottomup'} and \code{'topdown'}. 
The bottomup approach adds objectives and constraints on individual modules, whereas
the topdown approach adds objectives and constraints directly on routes.\cr

\code{plot.mst} draws module information functions if \code{byroute=FALSE}
and route information functions if \code{byroute=TRUE}
}
\examples{
\dontrun{
library(dplyr)
set.seed(10001)
## generate item pool
pool <- model_3pl()$gendata(1, 300)$items
pool$content <- sample(1:3, nrow(pool), replace=TRUE)
pool$time <- round(exp(rnorm(nrow(pool), log(60), .2)))

## ex. 1: 1-2-2 MST, 2 panels, topdown
## 20 items in total, 10 items in content area 1
## maximize info. at -1 and 1 for easy and hard routes
x <- mst(pool, "1-2-2", 2, 'topdown', len=20, maxselect=1)
x <- mst_obj(x, theta=-1, indices=1:2)
x <- mst_obj(x, theta=1, indices=3:4)
x <- mst_constraint(x, "content", 10, 10, level=1)
x <- mst_assemble(x, timeout=10)
plot(x)
plot(x, byroute=TRUE)
freq(mst_get_items(x, panel=1, route=1)$content, 1:3)$freq
freq(mst_get_items(x, panel=2, route=4)$content, 1:3)$freq

## ex. 2: 1-2-3 MST, 2 panels, bottomup, 
## 10 items in total and 4 items in content area 1 in each module
## maximize info. at -1, 0 and 1 for easy, medium, and hard modules
x <- mst(pool, "1-2-3", 2, 'bottomup', len=10, maxselect=1) \%>\%
  mst_route(c(1, 2, 6), "-") \%>\%
  mst_route(c(1, 3, 4), "-") \%>\%
  mst_obj(theta= 0, indices=c(1, 5)) \%>\%
  mst_obj(theta=-1, indices=c(2, 4)) \%>\%
  mst_obj(theta= 1, indices=c(3, 6)) \%>\%
  mst_constraint("content", 4, 4, level=1)
x <- mst_assemble(x, timeout=10)
group_by(x$items, panel, index) \%>\% summarise(n=sum(content==1))
}
}

