% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/phm_random_a0.R
\name{phm.random.a0}
\alias{phm.random.a0}
\title{Model fitting for the proportional hazards model with piecewise constant hazard and random a0}
\usage{
phm.random.a0(
  time,
  event,
  X,
  S,
  historical,
  n.intervals,
  change.points = NULL,
  prior.beta.mvn = NULL,
  prior.beta.mean = rep(0, 50),
  prior.beta.sd = rep(1000, 50),
  prior.lambda0.hp1 = rep(10^(-5), 50),
  prior.lambda0.hp2 = rep(10^(-5), 50),
  prior.a0.shape1 = rep(1, 10),
  prior.a0.shape2 = rep(1, 10),
  prior.lambda.hp1 = rep(10^(-5), 50),
  prior.lambda.hp2 = rep(10^(-5), 50),
  lower.limits = NULL,
  upper.limits = rep(100, 50),
  slice.widths = rep(0.1, 50),
  nMC = 10000,
  nBI = 250
)
}
\arguments{
\item{time}{Vector of follow up times.}

\item{event}{Vector of status indicators. Normally 0=alive and 1=dead.}

\item{X}{Matrix of covariates. The first column must be the treatment indicator.}

\item{S}{Vector of integers, where each integer represents the stratum that the subject belongs to. For example, if there are three strata, S can take values 1, 2 or 3.}

\item{historical}{List of historical dataset(s). East historical dataset is stored in a list which contains four \emph{named} elements: \code{time}, \code{event}, \code{X} and \code{S}.
\itemize{
\item \code{time} is a vector of follow up times.
\item \code{event} is a vector of status indicators. Normally 0=alive and 1=dead.
\item \code{X} is a matrix of covariates. The first column must be the treatment indicator.
\item \code{S} is a vector of integers, where each integer represents the stratum that the subject belongs to. For example, if there are three strata, S can take values 1, 2 or 3.
}}

\item{n.intervals}{Vector of integers, indicating the number of intervals for the baseline hazards for each stratum. The length of the vector should be equal to the total number of strata.}

\item{change.points}{List of vectors. Each vector in the list contains the change points for the baseline hazards for each stratum. The length of the list should be equal to the total number of strata.
For a given stratum, if there is only one interval, then \code{change.points} should be \code{NULL} for that stratum.
By default, we assign the change points so that the same number of events are observed in all the intervals in the pooled current and historical data.}

\item{prior.beta.mvn}{List of multivariate normal approximations of the normalized power prior for \eqn{\beta}. Each element in the list is a list with three components,
the mean vector, the covariance matrix and the weight of the multivariate normal distribution. The normalized power prior for \eqn{\beta}
is approximated by the weighted mixture of the multivariate normal distributions provided. By default (\code{prior.beta.mvn=NULL}), a single multivariate normal distribution is assumed.
The user can use the \code{\link{approximate.prior.beta}} function to obtain samples of \eqn{\beta} from the normalized power prior, and use any mixture of multivariate normals to approximate
the normalized power prior for \eqn{\beta}.}

\item{prior.beta.mean}{(Only applies if \code{prior.beta.mvn=NULL}) Vector of means of the normal initial prior on \eqn{\beta}. The default value is zero for all the elements of \eqn{\beta}.}

\item{prior.beta.sd}{(Only applies if \code{prior.beta.mvn=NULL}) Vector of standard deviations of the normal initial prior on \eqn{\beta}. The default value is 10^3 for all the elements of \eqn{\beta}.}

\item{prior.lambda0.hp1}{(Only applies if \code{prior.beta.mvn=NULL}) Vector of first hyperparameters of the Gamma prior on \eqn{\lambda_0}.
The length of the vector should be equal to the dimension of \eqn{\lambda_0}, i.e., the total number of intervals for all strata. The default value is 10^(-5) for all the elements of \eqn{\lambda_0}.}

\item{prior.lambda0.hp2}{(Only applies if \code{prior.beta.mvn=NULL}) Vector of second hyperparameters of the Gamma prior on \eqn{\lambda_0}.
The length of the vector should be equal to the dimension of \eqn{\lambda_0}, i.e., the total number of intervals for all strata. The default value is 10^(-5) for all the elements of \eqn{\lambda_0}.}

\item{prior.a0.shape1}{(Only applies if \code{prior.beta.mvn=NULL}) Vector of the first shape parameters of the independent beta priors for \eqn{a_0}. The length of the vector should be equal to the number of historical datasets. The default is a vector of one's.}

\item{prior.a0.shape2}{(Only applies if \code{prior.beta.mvn=NULL}) Vector of the second shape parameters of the independent beta priors for \eqn{a_0}. The length of the vector should be equal to the number of historical datasets. The default is a vector of one's.}

\item{prior.lambda.hp1}{Vector of first hyperparameters of the Gamma prior on \eqn{\lambda}.
The length of the vector should be equal to the dimension of \eqn{\lambda}, i.e., the total number of intervals for all strata. The default value is 10^(-5) for all the elements of \eqn{\lambda}.}

\item{prior.lambda.hp2}{Vector of second hyperparameters of the Gamma prior on \eqn{\lambda}.
The length of the vector should be equal to the dimension of \eqn{\lambda}, i.e., the total number of intervals for all strata. The default value is 10^(-5) for all the elements of \eqn{\lambda}.}

\item{lower.limits}{Vector of lower limits for parameters (\eqn{\beta} and \eqn{\lambda}, in this order) to be used by the slice sampler. The length of the vector should be equal to the total number of parameters. The default is -100 for \eqn{\beta} and 0 for \eqn{\lambda} (may not be appropriate for all situations).}

\item{upper.limits}{Vector of upper limits for parameters (\eqn{\beta} and \eqn{\lambda}, in this order) to be used by the slice sampler. The length of the vector should be equal to the total number of parameters. The default is 100 for all parameters (may not be appropriate for all situations).}

\item{slice.widths}{Vector of initial slice widths for parameters (\eqn{\beta} and \eqn{\lambda}, in this order) to be used by the slice sampler. The length of the vector should be equal to the total number of parameters. The default is 0.1 for all parameters (may not be appropriate for all situations).}

\item{nMC}{Number of iterations (excluding burn-in samples) for the slice sampler. The default is 10,000.}

\item{nBI}{Number of burn-in samples for the slice sampler. The default is 250.}
}
\value{
Posterior samples of \eqn{\beta} and \eqn{\lambda} are returned.
}
\description{
Model fitting using the normalized power prior for the proportional hazards model with piecewise constant hazard and random \eqn{a_0}
}
\details{
The proportional hazards model with piecewise constant hazard is implemented.
We assume \eqn{\beta} is the regression coefficients. We assume the first column of the covariate matrix is the treatment indicator,
and the corresponding parameter is \eqn{\beta_1}. Here \eqn{a_0} is modeled as random with a normalized power prior.

The normalized power prior for \eqn{\beta} is approximated by a weighted mixture of multivariate normal distributions provided in \code{prior.beta.mvn}.
The user can use the \code{\link{approximate.prior.beta}} function to obtain samples of \eqn{\beta} from the normalized power prior, and use any mixture of multivariate normals to approximate
the normalized power prior for \eqn{\beta}. By default, a single multivariate normal distribution is assumed.

Posterior samples are obtained through slice sampling.
The default lower limits are -100 for \eqn{\beta} and 0 for \eqn{\lambda}. The default upper limits
for the parameters are 100. The default slice widths for the parameters are 0.1.
The defaults may not be appropriate for all situations, and the user can specify the appropriate limits
and slice width for each parameter.
}
\examples{



set.seed(1)
# Simulate current data
n <- 50
P <- 4
time <- round(rexp(n, rate=0.5),1)
event <- rep(1,n)
X <- matrix(rbinom(n*P,prob=0.5,size=1), ncol=P)
S <- c(rep(1,n/2),rep(2,n/2))

# Simulate two historical datasets
n <- 100
time1 <- round(rexp(n, rate=0.5),1)
event1 <- rep(1,n)
X1 <- matrix(rbinom(n*P,prob=0.5,size=1), ncol=P)
S1 <- c(rep(1,n/2),rep(2,n/2))
time2 <- round(rexp(n, rate=0.7),1)
event2 <- rep(1,n)
X2 <- matrix(rbinom(n*P,prob=0.5,size=1), ncol=P)
S2 <- c(rep(1,n/2),rep(2,n/2))
historical <- list(list(time=time1, event=event1, X=X1, S=S1),
                   list(time=time2, event=event2, X=X2, S=S2))

# We choose three intervals for the first stratum and two intervals for the second stratum
n.intervals <- c(3,2) 
change.points <- list(c(1,2), 2)

# Get samples from the approximate normalized power prior for beta
nMC <- 100 # nMC should be larger in practice
nBI <- 50
prior.beta <- approximate.prior.beta(historical, n.intervals, change.points=change.points,
                                     prior.a0.shape1=c(1,1), prior.a0.shape2=c(1,1), 
                                     nMC=nMC, nBI=nBI)
prior_beta_mu=colMeans(prior.beta)
prior_beta_sigma=cov(prior.beta) 

# Aprroximate the discrete sames with a single multivariate normal with weight one
prior.beta.mvn <- list(list(prior_beta_mu, prior_beta_sigma, 1))

result <- phm.random.a0(time=time, event=event, X=X, S=S,
                        historical=historical, n.intervals=n.intervals, 
                        change.points=change.points, 
                        prior.beta.mvn=prior.beta.mvn,
                        nMC=nMC, nBI=nBI)

# posterior mean of beta
colMeans(result$beta_samples) 
# posterior mean of baseline hazards for stratum 1
colMeans(result$lambda_samples[[1]]) 
# posterior mean of baseline hazards for stratum 2
colMeans(result$lambda_samples[[2]]) 



}
\references{
Ibrahim, J. G., Chen, M.-H. and Sinha, D. (2001). Bayesian Survival Analysis. New York: Springer Science & Business Media.

Psioda, M. A. and Ibrahim, J. G. (2019). Bayesian clinical trial design using historical data that inform the treatment effect. Biostatistics 20, 400–415.

Shen, Y., Psioda, M. A., and Joseph, J. G. (2023). BayesPPD: an R package for Bayesian sample size determination using the power and normalized power prior for generalized linear models. The R Journal, 14(4).
}
\seealso{
\code{\link{power.phm.random.a0}} and \code{\link{approximate.prior.beta}}
}
