% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{bdSVD_hdf5}
\alias{bdSVD_hdf5}
\title{Singular Value Decomposition for HDF5-Stored Matrices}
\usage{
bdSVD_hdf5(
  filename,
  group = NULL,
  dataset = NULL,
  k = 2L,
  q = 1L,
  bcenter = TRUE,
  bscale = TRUE,
  rankthreshold = 0,
  overwrite = NULL,
  method = NULL,
  threads = NULL
)
}
\arguments{
\item{filename}{Character string. Path to the HDF5 file containing the input matrix.}

\item{group}{Character string. Path to the group containing the input dataset.}

\item{dataset}{Character string. Name of the input dataset to decompose.}

\item{k}{Integer. Number of local SVDs to concatenate at each level (default = 2).
Controls the trade-off between memory usage and computation speed.}

\item{q}{Integer. Number of levels for SVD computation (default = 1).
Higher values can improve accuracy but increase computation time.}

\item{bcenter}{Logical. If TRUE (default), centers the data by subtracting column means.}

\item{bscale}{Logical. If TRUE (default), scales the centered columns by their
standard deviations or root mean square.}

\item{rankthreshold}{Numeric. Threshold for determining matrix rank (default = 0).
Must be between 0 and 0.1. Used to approximate rank for nearly singular matrices.}

\item{overwrite}{Logical. If TRUE, allows overwriting existing results.}

\item{method}{Character string. Computation method:
\itemize{
\item "auto": Automatically selects between "full" and "blocks" based on matrix size
\item "blocks": Uses block-based computation (recommended for large matrices)
\item "full": Performs direct computation without partitioning
}}

\item{threads}{Integer. Number of threads for parallel computation.}
}
\value{
A list with the following elements:
\describe{
\item{fn}{Path to the HDF5 file}
\item{ds_d}{Path to the dataset containing singular values}
\item{ds_u}{Path to the dataset containing left singular vectors}
\item{ds_v}{Path to the dataset containing right singular vectors}
}
}
\description{
Computes the Singular Value Decomposition (SVD) of a large matrix stored in an HDF5 file.
The SVD decomposes a matrix A into a product A = UDV' where U and V are orthogonal
matrices and D is a diagonal matrix containing the singular values.
}
\details{
This function implements a block-based SVD algorithm suitable for large matrices
that may not fit in memory. Key features include:
\itemize{
\item Automatic method selection based on matrix size
\item Block-based computation for large matrices
\item Data centering and scaling options
\item Parallel processing support
\item Rank approximation through threshold
\item Memory-efficient incremental algorithm
}

The implementation uses an incremental algorithm with two key parameters:
\itemize{
\item k: number of local SVDs to concatenate at each level
\item q: number of levels in the computation
}
}
\examples{
\dontrun{
# Create a sample large matrix in HDF5

library(BigDataStatMeth)
library(rhdf5)

# Create a sample large matrix in HDF5
A <- matrix(rnorm(10000), 1000, 10)

fn <- "test_temp.hdf5"
bdCreate_hdf5_matrix(filename = fn, object = A, group = "data", dataset = "matrix")

# Compute SVD with default parameters
res <- bdSVD_hdf5(fn, "data", "matrix")

# Compute SVD with custom parameters
res <- bdSVD_hdf5(fn, "data", "matrix",
           k = 4, q = 2,
           bcenter = TRUE, bscale = TRUE,
           method = "blocks",
           threads = 4)

# list contents
h5ls(res$fn)

# Extract the result from HDF5 (d)
result_d_hdf5 <- h5read(res$fn, res$ds_d)
result_d_hdf5

# Compute the same SVD in R
result_d_r <- svd(A)$d
result_d_r

# Compare both results (should be TRUE)
all.equal(result_d_hdf5, result_d_r)

# Remove file
if (file.exists(fn)) {
  file.remove(fn)
}

}

}
\references{
\itemize{
\item Halko, N., Martinsson, P. G., & Tropp, J. A. (2011). Finding structure with randomness:
Probabilistic algorithms for constructing approximate matrix decompositions.
SIAM Review, 53(2), 217-288.
}
}
\seealso{
\itemize{
\item \code{\link{bdPCA_hdf5}} for Principal Component Analysis
\item \code{\link{bdQR_hdf5}} for QR decomposition
}
}
